package clinical.test;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import junit.framework.TestSuite;
import clinical.server.vo.Assessment;
import clinical.server.vo.Experiment;
import clinical.utils.FileUtils;
import clinical.web.DAOFactory;
import clinical.web.IAssessmentService;
import clinical.web.ServiceFactory;
import clinical.web.actions.AsQueryHelper;
import clinical.web.actions.MultiSiteQueryHelper;
import clinical.web.common.AssessmentMapping;
import clinical.web.common.IDBCache;
import clinical.web.common.query.BinaryOperator;
import clinical.web.common.query.Operator;
import clinical.web.common.query.ParallelQueryResult;
import clinical.web.common.query.SearchPredicate;
import clinical.web.common.query.UnaryOperator;
import clinical.web.common.vo.AsScoreInfo;
import clinical.web.common.vo.AssessmentSelectionInfo;
import clinical.web.helpers.ScoreValueSummary;
import clinical.web.helpers.ScoreValueSummaryHeader;
import clinical.web.vo.SubjectAsScoreValueSummary;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: MultiSiteQueryTest.java,v 1.1.1.1 2007/03/27 03:22:19 bozyurt
 *          Exp $
 */
public class MultiSiteQueryTest extends MyTestCase {
	protected Map<Integer, List<AssessmentMapping>> siteAsMap = null;

	public MultiSiteQueryTest(String testName, String propsFile)
			throws IOException {
		// don't cache DB metadata for QueryProcessor
		super(testName, propsFile, true);
	}

	protected void setUp() throws Exception {
		super.setUp();
		DAOFactory.initialize();
	}

	public static Map<String, Experiment> toMap(List<Experiment> experiments) {
		Map<String, Experiment> map = new HashMap<String, Experiment>();
		for (Iterator<Experiment> iter = experiments.iterator(); iter.hasNext();) {
			Experiment exp = iter.next();
			map.put(exp.getUniqueid().toString(), exp);
		}
		return map;
	}

	protected Operator buildOpTree2(List<AssessmentSelectionInfo> assessments) {
		String qmAsID = "252";
		AssessmentSelectionInfo asi = new AssessmentSelectionInfo();
		asi.setAssessmentID(new BigDecimal(qmAsID));
		asi.setName("quick mood scale");
		AsScoreInfo si1 = new AsScoreInfo(asi.getAssessmentID());
		si1.setType("integer");
		si1.setName("wide awake");
		asi.addScore(si1);
		AsScoreInfo si2 = new AsScoreInfo(asi.getAssessmentID());
		si2.setType("integer");
		si2.setName("relaxed");
		asi.addScore(si2);
		assessments.add(asi);

		UnaryOperator lhs = new UnaryOperator(new SearchPredicate(si1,
				new Integer(1), SearchPredicate.GREATER_EQUAL,
				SearchPredicate.INTEGER), Operator.NONE);
		UnaryOperator rhs = new UnaryOperator(new SearchPredicate(si2,
				new Integer(1), SearchPredicate.GREATER_EQUAL,
				SearchPredicate.INTEGER), Operator.NONE);

		BinaryOperator bop = new BinaryOperator(lhs, rhs, Operator.AND);
		return bop;
	}

	@SuppressWarnings("unused")
	protected Operator buildOpTree(List<AssessmentSelectionInfo> assessments) {
		AssessmentSelectionInfo asi = new AssessmentSelectionInfo();
		// will change every time data is reuploaded
		String MMSEAssessmentID = "1263";
		String DemographicsAssessmentID = "1264";
		String diagnosisID = "1265";

		asi.setAssessmentID(new BigDecimal(MMSEAssessmentID));
		asi.setName("MMSE");
		AsScoreInfo si1 = new AsScoreInfo(asi.getAssessmentID());
		si1.setType("integer");
		si1.setName("MMSE Score");
		asi.addScore(si1);
		assessments.add(asi);

		asi = new AssessmentSelectionInfo();
		asi.setAssessmentID(new BigDecimal(DemographicsAssessmentID));
		asi.setName("Demographics");
		AsScoreInfo si2 = new AsScoreInfo(asi.getAssessmentID());
		si2.setType("varchar");
		si2.setName("Gender");
		asi.addScore(si2);
		assessments.add(asi);

		asi = new AssessmentSelectionInfo();
		asi.setAssessmentID(new BigDecimal(diagnosisID));
		asi.setName("Diagnosis");
		AsScoreInfo si3 = new AsScoreInfo(asi.getAssessmentID());
		si3.setType("varchar");
		si3.setName("Diagnosis");
		asi.addScore(si3);
		assessments.add(asi);

		// SearchPredicate.Range range = new SearchPredicate.Range(
		// new Integer(15), new Integer(25));

		// prepare Operator tree
		/*
		 * UnaryOperator lhs = new UnaryOperator( new SearchPredicate(si1, new
		 * Integer(21) , SearchPredicate.GREATER, SearchPredicate.INTEGER),
		 * Operator.NONE);
		 */
		UnaryOperator lhs = new UnaryOperator(new SearchPredicate(si3, "A%",
				SearchPredicate.STARTS_WITH, SearchPredicate.STRING),
				Operator.NONE);

		
      UnaryOperator rhs = new UnaryOperator(new SearchPredicate(si2, "M",
				SearchPredicate.EQUAL, SearchPredicate.STRING), Operator.NONE);

		// BinaryOperator bop = new BinaryOperator(lhs, rhs, Operator.AND);

		return lhs;
	}

	protected Operator buildNAARTOpTree(List<Assessment> assessments,
			List<AssessmentSelectionInfo> asiList) {
		AssessmentSelectionInfo naart = null;
		for (Assessment as : assessments) {
			if (as.getName().indexOf("NAART") != -1) {
				naart = new AssessmentSelectionInfo(as.getName(), as
						.getAssessmentid());
				break;
			}
		}

		AsScoreInfo si1 = new AsScoreInfo(naart.getAssessmentID());
		si1.setName("VIQ");
		si1.setType("float");
		naart.addScore(si1);
		asiList.add(naart);

		UnaryOperator lhs = new UnaryOperator(new SearchPredicate(si1, "*",
				SearchPredicate.EQUAL, SearchPredicate.FLOAT), Operator.NONE);

		return lhs;
	}

	public void testQuery() throws Exception {
		String dbID = "ucsd_fbirn";

		List<SubjectAsScoreValueSummary> summaryList = null;
		List<AssessmentSelectionInfo> assessments = new ArrayList<AssessmentSelectionInfo>();
		Operator root = buildOpTree2(assessments);

		ParallelQueryResult queryResult = MultiSiteQueryHelper.doForkAndJoin(
				getUi(), dbID, root, assessments, siteAsMap, null, null, true,
				null);
		summaryList = queryResult.getSummaryList();

		for (Iterator<SubjectAsScoreValueSummary> it = summaryList.iterator(); it
				.hasNext();) {
			SubjectAsScoreValueSummary sasv = it.next();
			System.out.println(sasv.toString());
		}

		ScoreValueSummaryHeader svHeader = new ScoreValueSummaryHeader();

		Map<String, AssessmentSelectionInfo> asMap = AsQueryHelper
				.toAssessmentMap(assessments, "ucsd");

		serialize("/home/bozyurt/qstate.ser", summaryList, asMap, svHeader,
				siteAsMap);

		IDBCache dbCache = ServiceFactory.getDBCache(dbID);
		List<Experiment> experiments = dbCache.getExperiments(getUi(), false);
		Map<String, Experiment> expMap = toMap(experiments);

		// FIXME
		List<ScoreValueSummary> results = AsQueryHelper.processQueryResults(
				summaryList, asMap, svHeader, null, siteAsMap, dbID, super
						.getUi(), false, null, expMap);

		System.out.println("Results:" + results.size());

	}

	public void testMSQuery() throws Exception {

		String dbID = "ucsd_fbirn";

		List<SubjectAsScoreValueSummary> summaryList = null;
		Map<String, List<Assessment>> asMap = new HashMap<String, List<Assessment>>(
				7);
		for (int i = 0; i < getDbIDs().length; i++) {
			System.out.println("DB:" + getDbIDs()[i]);
			IAssessmentService ia = ServiceFactory
					.getAssessmentService(getDbIDs()[i]);
			List<Assessment> assessments = ia.getAllAssessments(getUi());
			asMap.put(getDbIDs()[i], assessments);
			for (Object element : assessments) {
				System.out.println(element);
			}
		}
		List<Assessment> assessments = asMap.get(dbID);
		List<AssessmentSelectionInfo> asiList = new ArrayList<AssessmentSelectionInfo>();
		Operator root = buildNAARTOpTree(assessments, asiList);

		ParallelQueryResult queryResult = MultiSiteQueryHelper
				.doForkAndJoin(getUi(), dbID, root, asiList, siteAsMap, null,
						null, true, null);
		summaryList = queryResult.getSummaryList();

		for (Iterator<SubjectAsScoreValueSummary> it = summaryList.iterator(); it
				.hasNext();) {
			SubjectAsScoreValueSummary sasv = it.next();
			System.out.println(sasv.toString());
		}

	}

	public void testQueryResults() throws Exception {
		QueryResultState state = deserialize("/home/bozyurt/qstate.ser");
		String dbID = "ucsd_fbirn";

		IDBCache dbCache = ServiceFactory.getDBCache(dbID);
		List<Experiment> experiments = dbCache.getExperiments(getUi(), false);
		Map<String, Experiment> expMap = toMap(experiments);

		// FIXME
		List<ScoreValueSummary> results = AsQueryHelper.processQueryResults(
				state.summaryList, state.asMap, state.svHeader, null,
				state.siteAsMap, dbID, super.getUi(), false, null, expMap);

		for (ScoreValueSummary svs : results) {
			System.out.println(svs.toString());
		}
	}

	void serialize(String filename,
			List<SubjectAsScoreValueSummary> summaryList,
			Map<String, AssessmentSelectionInfo> asMap,
			ScoreValueSummaryHeader svHeader,
			Map<Integer, List<AssessmentMapping>> siteAsMap) throws IOException {
		// FileOutputStream out = null;
		ObjectOutputStream oos = null;
		try {
			oos = new ObjectOutputStream(new FileOutputStream(filename));
			oos.writeObject(summaryList);
			oos.writeObject(asMap);
			oos.writeObject(svHeader);
			oos.writeObject(siteAsMap);
		} finally {
			FileUtils.close(oos);
		}
	}

	class QueryResultState {
		List<SubjectAsScoreValueSummary> summaryList;
		Map<String, AssessmentSelectionInfo> asMap;
		ScoreValueSummaryHeader svHeader;
		Map<Integer, List<AssessmentMapping>> siteAsMap;
	}

	@SuppressWarnings("unchecked")
   QueryResultState deserialize(String filename) throws Exception {
		ObjectInputStream in = null;
		QueryResultState state = new QueryResultState();
		try {
			in = new ObjectInputStream(new FileInputStream(filename));
			state.summaryList = (List<SubjectAsScoreValueSummary>) in
					.readObject();
			state.asMap = (Map<String, AssessmentSelectionInfo>) in
					.readObject();
			state.svHeader = (ScoreValueSummaryHeader) in.readObject();
			state.siteAsMap = (Map<Integer, List<AssessmentMapping>>) in
					.readObject();

		} finally {
			FileUtils.close(in);
		}
		return state;
	}

	public static TestSuite suite() throws IOException {
		TestSuite suite = new TestSuite();
		// suite.addTest( new MultiSiteQueryTest("testQuery", "test.properties")
		// );
		// suite.addTest( new MultiSiteQueryTest("testQueryResults",
		// "test.properties") );
		suite.addTest(new MultiSiteQueryTest("testMSQuery", "test.properties"));

		return suite;
	}

	public static void main(String[] args) throws IOException {
		junit.textui.TestRunner.run(suite());
	}

}
