package clinical.test.framework;

import java.sql.SQLException;

import org.dbunit.DatabaseUnitException;
import org.dbunit.database.DatabaseConfig;
import org.dbunit.database.IDatabaseConnection;
import org.dbunit.database.statement.IStatementFactory;
import org.dbunit.dataset.Column;
import org.dbunit.dataset.DataSetException;
import org.dbunit.dataset.IDataSet;
import org.dbunit.dataset.ITable;
import org.dbunit.dataset.ITableIterator;
import org.dbunit.dataset.ITableMetaData;
import org.dbunit.dataset.NoColumnsFoundException;
import org.dbunit.dataset.RowOutOfBoundsException;
import org.dbunit.operation.AbstractOperation;
import org.dbunit.operation.OperationData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * deletes all rows exactly matching the ones in the provided {@link IDataSet}.
 * Used for tables without a primary key.
 *
 * @author I. Burak Ozyurt
 * @version $Id: DeleteIfExistsOperation.java,v 1.1 2008/09/04 00:31:58 bozyurt
 *          Exp $
 */
public class DeleteIfExistsOperation extends AbstractOperation {
   private static final Logger logger = LoggerFactory
         .getLogger(DeleteIfExistsOperation.class);

   public DeleteIfExistsOperation() {}

   @Override
   public void execute(IDatabaseConnection connection, IDataSet dataSet)
         throws DatabaseUnitException, SQLException {
      logger.debug("execute(connection=" + connection + ", dataSet) - start");
      DatabaseConfig databaseConfig = connection.getConfig();
      IStatementFactory factory = (IStatementFactory) databaseConfig
            .getProperty(DatabaseConfig.PROPERTY_STATEMENT_FACTORY);

      // for each table
      ITableIterator iterator = dataSet.reverseIterator();
      while (iterator.next()) {
         ITable table = iterator.getTable();

         // skip empty tables
         if (DBUnitUtils.isEmpty(table)) {
            continue;
         }

        // System.out.println("deleting records from table "
         //      + table.getTableMetaData().getTableName());
         ITableMetaData metaData = DBUnitUtils.getOperationMetaData(connection,
               table.getTableMetaData());
         RowExistOperation reOp = new RowExistOperation(connection, metaData);
         DeleteRowOperation delRowOp = new DeleteRowOperation(connection,
               metaData, factory);
         try {
            // refresh all rows
            for (int i = 0;; i++) {
               if (reOp.execute(table, i)) {
                  delRowOp.execute(table, i);
               }
            }
         } catch (RowOutOfBoundsException e) {
            // This exception occurs when records are exhausted
            // and we reach the end of the table. Ignore this error.

            // end of table
         } finally {
            // cleanup
            reOp.close();
            delRowOp.close();
         }

      }

   }

   class DeleteRowOperation extends RowOperation {
      public DeleteRowOperation(IDatabaseConnection connection,
            ITableMetaData metaData, IStatementFactory factory)
            throws DataSetException, SQLException {
         _operationData = get2DeleteRowData(metaData, connection);
         _statement = factory.createPreparedBatchStatement(_operationData
               .getSql(), connection);
      }

      protected OperationData get2DeleteRowData(ITableMetaData metaData,
            IDatabaseConnection connection) throws DataSetException {
         Column[] columns = metaData.getColumns();

         if (columns.length == 0) {
            throw new NoColumnsFoundException(metaData.getTableName());
         }
         StringBuilder sb = new StringBuilder(128);
         sb.append("delete from ");
         sb.append(getQualifiedName(connection.getSchema(), metaData
               .getTableName(), connection));
         sb.append(" where ");

         for (int i = 0; i < columns.length; i++) {
            Column column = columns[i];
            if (i > 0) {
               sb.append(" and ");
            }
            sb
                  .append(getQualifiedName(null, column.getColumnName(),
                        connection));
            sb.append(" = ?");
         }

         return new OperationData(sb.toString(), columns);
      }
   }

}
