package clinical.tools.db;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.jdom.Element;

import clinical.utils.FileUtils;
import clinical.web.DBUtils;

/**
 * @author I. Burak Ozyurt
 * @version $Id: SchemaExtractor.java 750 2012-12-03 20:45:56Z bozyurt $
 */
public class SchemaExtractor {
	Map<String, DBTable> tableMap = new HashMap<String, DBTable>();

	public SchemaExtractor() {
	}
	
	public void loadDBTables(Connection con, String schemaName, String schemaXmlFile,
			boolean loadViewsAlso) throws Exception 
	{
		 DatabaseMetaData meta = con.getMetaData();
		 ResultSet rs = meta.getCatalogs();
		 while(rs.next()) {
			 System.out.println( rs.getString(1));
		 }
		 rs.close();
		 String[] tableTypes = new String[] {"TABLE" };
		 if ( loadViewsAlso) {
			 tableTypes = new String[] {"TABLE", "VIEW"};
		 }
		 @SuppressWarnings("unused")
		int count = 0;
		 
		 rs = meta.getTables("", schemaName, "%", tableTypes);
		 while( rs.next() ) {
			 String tableName = rs.getString(3);
			 System.out.println("table: " + tableName);
			 if (!tableName.toLowerCase().startsWith("nc_") ) {
				 continue;
			 }
			 System.out.println("table: " + tableName);
			 
			 DBTable table = new DBTable(con, tableName, schemaName);
			 tableMap.put(tableName,table);
			 table.numRows = getNumRows(con, tableName);
			 ++count;
			 // if (count > 10) break;
		 }
		 rs.close();
		 
		 buildXML(schemaXmlFile, tableMap);
	}
	
	public void buildXML(String fname, Map<String, DBTable> tm) throws Exception {
		Element root = new Element("schema");
		for(DBTable dt : tm.values()) {
			Element dtElem = new Element("dt");
			root.addContent(dtElem);
			dtElem.setAttribute("name", dt.name);
			dtElem.setAttribute("sorted", String.valueOf(dt.sorted));
			dtElem.setAttribute("numCols", String.valueOf(dt.columns.size()));
			dtElem.setAttribute("numRows", String.valueOf(dt.numRows));
			for(DBColumn col: dt.columns) {
				Element colEl = new Element("col");
				dtElem.addContent(colEl);
				colEl.setAttribute("name", col.name);
				colEl.setAttribute("type", col.type);
				colEl.setAttribute("width", String.valueOf(col.width) );
				colEl.setAttribute("fracDigit", String.valueOf(col.fracDigit) );
				colEl.setAttribute("primaryKey", String.valueOf(col.primaryKey) );
				colEl.setAttribute("pkOrder", String.valueOf(col.pkOrder) );								
				
			}
			
			Element ekRootEl = new Element("exported-keys");
			dtElem.addContent(ekRootEl);
			for(DBFK ek : dt.exportedKeys) {
				Element ekEl = new Element("exported-key");
				ekRootEl.addContent(ekEl);
				ekEl.setAttribute("name", ek.name);
				ekEl.setAttribute("pkTableName", ek.pkTableName);
				ekEl.setAttribute("pkColumnName", ek.pkColumnName);
				ekEl.setAttribute("fkTableName", ek.fkTableName);
				ekEl.setAttribute("fkColumnName", ek.fkColumnName);
				ekEl.setAttribute("keySeq", String.valueOf( ek.keySeq));								
			}
			
			Element ikRootEl = new Element("imported-keys");
			dtElem.addContent(ikRootEl);
			for(DBFK ik : dt.importedKeys) {
				Element ikEl = new Element("imported-key");
				ikRootEl.addContent(ikEl);
				ikEl.setAttribute("name", ik.name);
				ikEl.setAttribute("pkTableName", ik.pkTableName);
				ikEl.setAttribute("pkColumnName", ik.pkColumnName);
				ikEl.setAttribute("fkTableName", ik.fkTableName);
				ikEl.setAttribute("fkColumnName", ik.fkColumnName);
				ikEl.setAttribute("keySeq", String.valueOf( ik.keySeq));								
			}			
		}
		
		FileUtils.saveXML(root, fname);
	}
	
	
	protected int getNumRows(Connection con, String tableName) throws SQLException {
		Statement st = null;
		ResultSet rs = null;
		try {
			st = con.createStatement();
			rs = st.executeQuery("select count(*) as cnt from " + tableName);
			if ( rs.next()) {
				return rs.getInt(1);
			}
			
		} finally {
			DBUtils.close(st, rs);
		}	
		return -1;
	}
	
	
	
	class DBTable {
		String name;
		List<DBColumn> columns = new ArrayList<DBColumn>(10);
		boolean sorted;
		List<DBFK> importedKeys = new ArrayList<DBFK>(3);
		List<DBFK> exportedKeys = new ArrayList<DBFK>(3);
		int numRows;
		
		DBTable(Connection con, String tableName, String schema) throws SQLException {
			this.name = tableName;
			extractTableInfo(con, tableName, schema);
		}
		
		public void extractTableInfo(Connection con, String tableName, String schema) throws SQLException {
			DatabaseMetaData meta = con.getMetaData();
			ResultSet rs = meta.getColumns("", schema, tableName,"%");
			 while( rs.next()) {
			    DBColumn col = new DBColumn();
			    col.name = rs.getString(4);
			    col.type = rs.getString(6);
			    col.width = rs.getInt(7);
			    col.fracDigit = rs.getInt(9);
			    col.order = rs.getInt(17);
			    /*
			    name: rs.getString(4), 
						type: rs.getString(6), width: rs.getInt(7),
						fracDigit: rs.getInt(9), order: rs.getInt(17));
				*/		
				columns.add(col);		
			 }
			 rs.close();
			 
			 getPrimaryKeys(con,tableName, schema);
			 getImportedKeys(con, tableName, schema);
			 getExportedKeys(con, tableName, schema);
		}
		
		public String[] getColumnNames() {
			String[] colNames = new String[ columns.size() ];
			int i = 0;
			for (DBColumn col : columns) {
				colNames[i++] = col.name;
			}
			return colNames;
		}
		
		
		protected void getImportedKeys(Connection con, String tableName, String schema) throws SQLException {
			DatabaseMetaData meta = con.getMetaData();
			ResultSet rs = meta.getImportedKeys("", schema, tableName);
			while( rs.next()) {
			   DBFK fk = createFK(rs);		
			   importedKeys.add(fk);
			}
			rs.close();
		}

		private DBFK createFK(ResultSet rs) throws SQLException {
			DBFK fk = new DBFK(); 
			   fk.name = rs.getString(12);
			   fk.pkTableName = rs.getString(3);
			   fk.pkColumnName = rs.getString(4);
			   fk.fkTableName = rs.getString(7);
			   fk.fkColumnName = rs.getString(8);
			   fk.keySeq = rs.getInt(9);
			return fk;
		}
		
		protected void getExportedKeys(Connection con, String tableName, String schema) throws SQLException {
			DatabaseMetaData meta = con.getMetaData();
			ResultSet rs = meta.getExportedKeys("", schema, tableName);
			while( rs.next()) {
			   DBFK fk = createFK(rs);
			   exportedKeys.add(fk);
			}
			rs.close();
		}
		
		private void getPrimaryKeys(Connection con, String tableName, String schema) throws SQLException {
			DatabaseMetaData meta = con.getMetaData();
			ResultSet rs = meta.getPrimaryKeys("", schema, tableName);
			while( rs.next()) {
				String pkName = rs.getString(4);
				for(DBColumn col : columns) {
					if(col.name.equals(pkName) ) {
						col.primaryKey = true;
						col.pkOrder = rs.getInt(5);
						break;
					}
				}
			}
			rs.close();
		}
		
		protected List<DBColumn> getPKColumns() {
			List<DBColumn> pkColList = new ArrayList<DBColumn>(1);
			for(DBColumn col : columns) {
				pkColList.add(col);
			}
			return pkColList;
		}
		
		protected List<Integer> getPKIndices() {
			List<Integer> idxList = new ArrayList<Integer>(1);
			for (DBColumn col : columns) {
				idxList.add( col.pkOrder );
			}
			return idxList;
		}
		
		public String toString() {
			StringBuffer sb = new StringBuffer(256);
			sb.append(name).append(": Columns:");
			for (Iterator<DBColumn> it = columns.iterator(); it.hasNext();) {
				DBColumn col = it.next();
				sb.append(col.name);
				if ( it.hasNext()) sb.append(',');
			}
			sb.append("\nPrimary Key: ");
			List<DBColumn> pkCols = getPKColumns();
			for (Iterator<DBColumn> it = pkCols.iterator(); it.hasNext();) {
				DBColumn pkc = it.next();
				sb.append(pkc.name);
				if ( it.hasNext()) sb.append(',');
			}
			
			sb.append("\nImported Keys (").append(importedKeys.size()).append(")\nExported Keys (");
			sb.append(exportedKeys.size()).append(")");
			if ( !importedKeys.isEmpty() ) {
				sb.append("\nImportedKeys:\n");
				for (DBFK ik : importedKeys) {
					sb.append(ik.toString()).append('\n');
				}
			}
			if ( !exportedKeys.isEmpty() ) {
				sb.append("\nExportedKeys:\n");
				for (DBFK ek : exportedKeys) {
					sb.append(ek.toString()).append('\n');
				}
			}
			return sb.toString();
		}
	}

	public class DBColumn {
		String name;
		String type;
		int width;
		int fracDigit;
		boolean notNullable;
		int order;
		boolean primaryKey;
		int pkOrder;
		
		String toDDL() {
			StringBuffer buf = new StringBuffer();
			buf.append(name).append(' ').append(type);
			if ( width > 0 && !type.equals("DATE") ) {
				buf.append('(').append(width);
				if (fracDigit > 0) 
					buf.append(',').append(fracDigit);
				buf.append(") ");	
			} else {
				buf.append("  ");
			}
			if (notNullable) 
				buf.append("NOT NULL");
			return buf.toString();
		}
	}

	public static class DBFK {
		String name;
		String pkTableName;
		String pkColumnName;
		String fkTableName;
		String fkColumnName;
		int keySeq;
		
		public String toString() {
			StringBuffer sb = new StringBuffer(128);
			sb.append("DBFK::[").append(name).append("\n\t");
			sb.append("fk: ").append(fkTableName).append('.').append(fkColumnName);
			sb.append("pk: ").append(pkTableName).append('.').append(pkColumnName);
			sb.append(")]");
			return sb.toString();
		}
	}
}
