package clinical.tools.dbadmin;

import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import clinical.server.dao.JobProvParamTypeDAO;
import clinical.server.vo.JobProvenanceParam;
import clinical.utils.Assertion;
import clinical.utils.GenUtils;
import clinical.web.ConnectionSupportMixin;
import clinical.web.DAOFactory;
import clinical.web.ISequenceHelper;
import clinical.web.MinimalServiceFactory;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.services.SecurityService;
import clinical.web.vo.JobProvenanceInfo.JobProvenanceParamType;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class JobProvParamTypeAdmin extends AbstractAdmin {
	private final static Pattern datePattern = Pattern
			.compile("\\d{2,2}\\/\\d{2,2}\\/\\d{4,4}");

	public JobProvParamTypeAdmin(String usersXmlFile) throws Exception {
		csm = new ConnectionSupportMixin(usersXmlFile);
		csm.startup();

		SecurityService ss = SecurityService.getInstance(
				csm.getDbPoolService(), csm.getDbID(), csm.getDbType());
		ss.startup();
	}

	public void populateJobProvParamType() throws Exception {
		Connection con = null;
		try {
			con = csm.getConnection();
			con.setAutoCommit(false);
			Map<String, Set<String>> map = getProvParamNames(con);
			List<String> names = new ArrayList<String>(map.keySet());
			Collections.sort(names);
			for (String name : names) {
				Set<String> valueSet = map.get(name);
				JobProvenanceParamType type = getTypeFromValues(valueSet);
				Assertion.assertNotNull(type);
				addProvenanceParamType(con, name, type.toString());
			}
			con.commit();
		} catch (Exception x) {
			con.rollback();
			x.printStackTrace();
		} finally {
			csm.releaseConnection(con);
		}
	}

	private clinical.server.vo.JobProvParamType addProvenanceParamType(
			Connection con, String name, String type) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(csm.getDbID());
		JobProvParamTypeDAO dao = DAOFactory.createJobProvParamTypeDAO(csm
				.getDbID());

		clinical.server.vo.JobProvParamType cr = new clinical.server.vo.JobProvParamType();
		cr.setName(name);
		List<clinical.server.vo.JobProvParamType> list = dao.find(con, cr);
		if (list.isEmpty()) {
			BigDecimal uniqueid = sequenceHelper.getNextUID(con,
					"nc_job_prov_param_type", "uniqueid");
			clinical.server.vo.JobProvParamType jppt = new clinical.server.vo.JobProvParamType();
			jppt.setUniqueid(uniqueid);
			jppt.setName(name);
			jppt.setDataType(type);

			dao.insert(con, jppt);
			return jppt;
		} else {
			Assertion.assertTrue(list.get(0).getDataType().equals(type));
			System.out.println("provenance param:" + name
					+ " already has a type:" + list.get(0).getDataType());
			return list.get(0);
		}
	}

	public static JobProvenanceParamType getTypeFromValues(Set<String> valueSet) {
		for (String value : valueSet) {
			if (value.equals("yes") || value.equals("no")) {
				return JobProvenanceParamType.BOOL;
			} else {
				Number number = GenUtils.toNumber(value);
				if (number == null) {
					if (value.indexOf('/') != -1) {
						if (datePattern.matcher(value).find()) {
							return JobProvenanceParamType.DATE;
						}
					}
					return JobProvenanceParamType.STRING;
				} else {
					return (number instanceof Integer) ? JobProvenanceParamType.INT
							: JobProvenanceParamType.REAL;
				}
			}
		}
		return null;
	}

	private Map<String, Set<String>> getProvParamNames(Connection con)
			throws Exception {
		Map<String, Set<String>> map = new HashMap<String, Set<String>>();
		TSQLProcessor tsp = new TSQLProcessor(csm.getSqlDialect());
		StringBuilder sb = new StringBuilder(128);
		sb.append("select distinct a.name, a.value from JobProvenanceParam as a ");
		List<?> results = tsp.executeQuery(con, sb.toString());
		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			JobProvenanceParam jpp = (JobProvenanceParam) it.next();
			String key = jpp.getName();
			Set<String> valueSet = map.get(key);
			if (valueSet == null) {
				valueSet = new HashSet<String>();
				map.put(key, valueSet);
			}
			valueSet.add(jpp.getValue());
		}
		return map;
	}

	public static void main(String[] args) throws Exception {
		JobProvParamTypeAdmin admin = null;
		try {
			admin = new JobProvParamTypeAdmin("users.xml");
			admin.populateJobProvParamType();
		} finally {
			if (admin != null) {
				admin.shutdown();
			}
		}
	}

}
