package clinical.tools.install;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.Date;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.tools.ant.Project;
import org.apache.tools.ant.input.InputRequest;
import org.jdom.Element;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

import clinical.utils.GenUtils;

/**
 * @author I. Burak Ozyurt
 * @version $Id: MinimalUserFileCreator.java,v 1.1 2008/01/17 20:33:58 bozyurt
 *          Exp $
 */
public class MinimalUserFileCreator implements IUserInput {
	protected BufferedReader in;
	protected Project project;

	public MinimalUserFileCreator() {
		in = new BufferedReader(new InputStreamReader(System.in));
	}

	public MinimalUserFileCreator(Project project) {
		this.project = project;
	}

	public String getInput(String prompt) throws IOException {
		if (in != null) {
			System.out.print(prompt);
			return in.readLine().trim();
		} else {
			InputRequest request = new InputRequest(prompt);
			project.getInputHandler().handleInput(request);
			String value = request.getInput();
			return value.trim();
		}
	}

	public void prepareUsersFile(String xmlFilename) throws IOException,
			SQLException {
		UsersFileInfo ui = new UsersFileInfo();

		System.out.println("Preparing mimimal users.xml file...");
		DatabaseInfo dbi = new DatabaseInfo();
		DBUserInfo dbui = new DBUserInfo();
		do {
			System.out
					.println("------------------ Database Section -----------------");
			dbi.dbType = Utils.getUserInput("Please enter the database type",
					new String[] { "postgres", "oracle" }, this);

			dbi.hostName = Utils
					.getUserInput(
							"In most cases, the fully qualified hostname of your gpop is the host name\n"
									+ "Please enter the database server host name",
							null, this);

			dbi.dbID = Utils
					.getUserInput(
							"The database id is expected to be unique and made up from characters and '_' only\n"
									+ "A valid database ID is, for example, ucsd_fbirn .\n"
									+ "The database id is, usually, representative of the corresponding HID database \n"
									+ "and is used to identify different data sources by the web app.\n\n"
									+ "Please enter the database id", null, this);
			dbi.defaultDB = true;

			if (dbi.dbType.equalsIgnoreCase("oracle")) {
				dbi.port = GenUtils.toInt(Utils.getUserInput(
						"In most cases, the Oracle database connection port is 1521.\n\n"
								+ "Please enter the database server connection port",
						null, this), 1521);

				dbi.SID = Utils.getUserInput(
						"For most default BIRN rack Oracle installation the SID is orcl1\n\n"
								+ "Please enter the SID of your database", null, this);
			} else if (dbi.dbType.equalsIgnoreCase("postgres")) {
				dbi.SID = Utils
						.getUserInput(
								"Please enter the name of your Postgres database",
								null, this);
				String s = Utils.getUserInput(
						"The default port for Postgres is 5432.\n\n"
								+ "Please enter the database server connection port",
						null, this);

				dbi.port = GenUtils.toInt(s, 5432);
			}

			ui.addDatabase(dbi);
			PrivilegeInfo pi = new PrivilegeInfo("admin", "", false);
			ui.addPrivilege(pi);
			pi = new PrivilegeInfo("create-project", "", false);
			ui.addPrivilege(pi);
			ui.addPrivilege(new PrivilegeInfo("create-user",
					"Privilege to add/invite new web users to the system.", false));

			ui.addPrivilege(new PrivilegeInfo("create-subject", "", true));
			ui.addPrivilege(new PrivilegeInfo("read", "", true));
			ui.addPrivilege(new PrivilegeInfo("update", "", true));			
			ui.addPrivilege(new PrivilegeInfo("delete", "", true));		
			ui.addPrivilege(new PrivilegeInfo("execute", "", true));		
			
			
			// add roles
			ui.addRole(new RoleInfo("principal investigator", null));
			ui.addRole(new RoleInfo("researcher", null));

			dbui.name = Utils.getUserInput(
					"Please enter the named database connection user name", null,
					this);
			dbui.password = Utils.getUserInput(
					"Please enter the named database connection password", null,
					this);
			dbui.dbID = dbi.dbID;

			if (testJDBCConnection(ui.findDatabase(dbui.dbID), dbui)) {
				System.out
						.println("Database connection test is successful for user "
								+ dbui.name + ".");
				System.out
						.println("==============================================================");
				System.out.println();
			} else {
				dbi = new DatabaseInfo();
				dbui = new DBUserInfo();
				ui = new UsersFileInfo();
				continue;
			}
			ui.addDBUser(dbui);

			// add admin web user

			WebUserInfo wui = new WebUserInfo();
			wui.name = "admin";
			wui.dbID = dbi.dbID;

			wui.dbUser = dbui.name;

			wui.password = Utils.getUserInput(
					"Please enter the admin web user password for database user '"
							+ wui.dbUser + "' for database '" + wui.dbID + "' ", null,
					this);
			
			for(PrivilegeInfo pri : ui.privileges) {
				if (pri.name.equalsIgnoreCase("admin")) {
				wui.addPrivilege( new PrivilegeInfo(pri.name, null, pri.projectSpecific) );
				}
			}
			
			if (ui.findWebUser(wui.name, wui.dbID, wui.dbUser) == null) {
				ui.addWebUser(wui);
			}

			break;
		} while (true);
		Map<String, SiteInfo> siMap = getSites(dbi, dbui);
		// get site info from the user
		int maxSiteUniqueId = -1;
		while (true) {
			int idx = 0;
			SiteInfo[] siArr = new SiteInfo[siMap.size()];
			for (SiteInfo si : siMap.values()) {
				System.out.println(idx + ") " + si.siteName + " (" + si.siteID
						+ ")");
				siArr[idx] = si;
				++idx;
				if (si.uniqueId > maxSiteUniqueId) {
					maxSiteUniqueId = si.uniqueId;
				}
			}
			String ans = Utils
					.getUserInput(
							"Please select a site from the menu or enter the keyword 'new' for a new site",
							null, this);
			if (ans.equalsIgnoreCase("new")) {
				// add a new site
				String siteId = null;
				String siteName = null;
				while (true) {
					siteId = Utils.getUserInput(
							"Please enter four digit zero padded Site ID (e.g. 0008)",
							null, this);
					if (!isValidSiteId(siteId)) {
						continue;
					}
					siteName = Utils.getUserInput(
							"Please enter a site name (e.g. UCSD)", null, this);
					break;
				}
				SiteInfo si = new SiteInfo(siteId, siteName, true);
				si.uniqueId = maxSiteUniqueId + 1;
				dbi.siteId = siteId;
				dbi.siteName = siteName;
				addSite(dbi, dbui, si);
			} else {
				idx = GenUtils.toInt(ans, -1);
				if (idx == -1) {
					continue;
				} else {
					dbi.siteId = siArr[idx].siteID;
					dbi.siteName = siArr[idx].siteName;
				}
			}
			break;
		}

		BufferedOutputStream bout = null;
		try {
			bout = new BufferedOutputStream(new FileOutputStream(xmlFilename));
			saveAsXML(bout, ui.toXML());
			System.out
					.println("Saved bootstrapping information to " + xmlFilename);
		} finally {
			if (bout != null)
				try {
					bout.close();
				} catch (Exception ex) { /* ignore */
				}
		}
	}

	public static boolean isValidSiteId(String siteId) {
		if (siteId == null || siteId.length() != 4)
			return false;
		return GenUtils.toInt(siteId, -1) != -1;
	}

	protected void saveAsXML(OutputStream out, Element parent)
			throws IOException {
		Format format = Format.getPrettyFormat();
		format.setLineSeparator(System.getProperty("line.separator"));
		XMLOutputter xmlOut = new XMLOutputter(format);
		xmlOut.output(parent, out);
	}

	protected static Map<String, SiteInfo> getSites(DatabaseInfo dbi,
			DBUserInfo dbUserInfo) throws SQLException {
		Map<String, SiteInfo> siteNameIDMap = new LinkedHashMap<String, SiteInfo>();
		Connection con = null;
		Statement st = null;
		try {
			con = DriverManager.getConnection(dbi.getDBURL(), dbUserInfo.name,
					dbUserInfo.password);
			st = con.createStatement();
			ResultSet rs = st
					.executeQuery("select siteid,sitename, isprimary, uniqueid from nc_site");
			while (rs.next()) {
				String siteID = rs.getString(1);
				String siteName = rs.getString(2);
				String isPrimaryStr = rs.getString(3);
				int uniqueId = rs.getInt(4);
				boolean isPrimary = (isPrimaryStr.equalsIgnoreCase("true") || isPrimaryStr
						.equals("1"));
				SiteInfo si = new SiteInfo(siteID, siteName, isPrimary);
				si.uniqueId = uniqueId;
				siteNameIDMap.put(siteID, si);
			}
			rs.close();
			return siteNameIDMap;
		} finally {
			close(con, st);
		}
	}

	protected static void addSite(DatabaseInfo dbi, DBUserInfo dbUserInfo,
			SiteInfo si) throws SQLException {
		Connection con = null;
		PreparedStatement pst = null;
		Statement st = null;
		try {
			con = DriverManager.getConnection(dbi.getDBURL(), dbUserInfo.name,
					dbUserInfo.password);

			con.setAutoCommit(false);
			int tableId = getTableId(con, "nc_site");
			if (dbi.dbType.equalsIgnoreCase("oracle")) {
				st = con.createStatement();
				ResultSet rs = st
						.executeQuery("select u.uniqueid from nc_databaseuser u, nc_userclass c "
								+ "where u.isgroup=0 and u.userclass = c.uniqueid and c.name='admin'");

				if (rs.next()) {
					String userId = rs.getString(1);
					if (si.primary)
						resetPrimaryDatabase(con, dbi.dbType);
					pst = con
							.prepareStatement("insert into nc_site values(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
					pst.clearParameters();
					pst.setBigDecimal(1, new BigDecimal(si.uniqueId));
					pst.setBigDecimal(2, new BigDecimal(tableId)); // tableID!!
					pst.setBigDecimal(3, new BigDecimal(userId)); // owner
					pst.setDate(4, new Date(System.currentTimeMillis())); // modTime
					pst.setBigDecimal(5, new BigDecimal(userId)); // modUser
					pst.setString(6, si.siteID);
					pst.setString(7, si.siteName);
					pst.setBigDecimal(8, new BigDecimal(si.primary ? 1 : 0)); // isprimary
					pst.setString(9, "databaseAddress");
					pst.setString(10, dbi.SID); // databaseName
					pst.setString(11, "Oracle"); // databaseVendor
					pst.setString(12, "databaseVersion");
					pst.setBigDecimal(13, new BigDecimal(dbi.port));
					pst.setString(14, "contactFirstName");
					pst.setString(15, "contactLastName");
					pst.setString(16, "contactEmail");
					pst.setString(17, "contactPhone");

					pst.execute();
				}
				rs.close();
				con.commit();
			} else {
				st = con.createStatement();
				ResultSet rs = st
						.executeQuery("select u.uniqueid from nc_databaseuser u, nc_userclass c "
								+ "where u.isgroup=false and u.userclass = c.uniqueid and c.name='admin'");

				if (rs.next()) {
					String userId = rs.getString(1);
					if (si.primary)
						resetPrimaryDatabase(con, dbi.dbType);
					pst = con
							.prepareStatement("insert into nc_site values(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
					pst.clearParameters();
					pst.setLong(1, new Long(si.uniqueId));
					pst.setLong(2, new Long(tableId)); // tableID!!
					pst.setLong(3, new Long(userId)); // owner
					pst.setDate(4, new Date(System.currentTimeMillis())); // modTime
					pst.setLong(5, new Long(userId)); // modUser
					pst.setString(6, si.siteID);
					pst.setString(7, si.siteName);
					pst.setLong(8, new Long(si.primary ? 1 : 0)); // isprimary
					pst.setString(9, "databaseAddress");
					pst.setString(10, dbi.SID); // databaseName
					pst.setString(11, "Postgres"); // databaseVendor
					pst.setString(12, "databaseVersion");
					pst.setLong(13, new Long(5432)); // default port
					pst.setString(14, "contactFirstName");
					pst.setString(15, "contactLastName");
					pst.setString(16, "contactEmail");
					pst.setString(17, "contactPhone");

					pst.execute();
				}
				rs.close();
				con.commit();

			}
		} catch (SQLException x) {
			if (con != null) {
				con.rollback();
			}
			throw x;
		} finally {
			close(st);
			close(con, pst);
		}
	}

	public static int getTableId(Connection con, String tableName)
			throws SQLException {
		PreparedStatement pst = null;
		try {
			pst = con
					.prepareStatement("select tableid from nc_tableid where tablename = ?");
			pst.setString(1, tableName.toUpperCase());
			ResultSet rs = pst.executeQuery();
			int tableID = -1;
			if (rs.next()) {
				tableID = rs.getInt(1);
			}
			rs.close();
			return tableID;
		} finally {
			close(pst);
		}
	}

	public static void resetPrimaryDatabase(Connection con, String dbType)
			throws SQLException {
		PreparedStatement pst = null;
		try {
			pst = con.prepareStatement("update nc_site set isprimary = 0");
			pst.execute();
		} finally {
			close(pst);
		}
	}

	public static boolean testJDBCConnection(DatabaseInfo dbi,
			DBUserInfo dbUserInfo) {
		Connection con = null;
		Statement st = null;
		String testQuery = "select 2 from dual";
		try {
			if (dbi.dbType.equalsIgnoreCase("oracle")) {
				Class.forName("oracle.jdbc.driver.OracleDriver");
			} else if (dbi.dbType.equalsIgnoreCase("postgres")) {
				Class.forName("org.postgresql.Driver");
				testQuery = "select 2";
			}
			con = DriverManager.getConnection(dbi.getDBURL(), dbUserInfo.name,
					dbUserInfo.password);
			st = con.createStatement();
			ResultSet rs = st.executeQuery(testQuery);
			rs.close();
		} catch (Exception x) {
			x.printStackTrace();
			return false;
		} finally {
			close(con, st);
		}
		return true;
	}

	protected static void close(Statement st) {
		if (st != null)
			try {
				st.close();
			} catch (Exception ex) { /* ignore */
			}
	}

	protected static void close(Connection con, Statement st) {
		if (st != null)
			try {
				st.close();
			} catch (Exception ex) { /* ignore */
			}
		if (con != null)
			try {
				con.close();
			} catch (Exception ex) { /* ignore */
			}
	}

	public static String[] getPrivilegeNames(List<PrivilegeInfo> privileges) {
		String[] privNames = new String[privileges.size()];
		int i = 0;
		for (Iterator<PrivilegeInfo> iter = privileges.iterator(); iter.hasNext();) {
			PrivilegeInfo pi = iter.next();
			privNames[i++] = pi.name;
		}
		return privNames;
	}

	public static class SiteInfo {
		String siteID;
		String siteName;
		boolean primary;
		int uniqueId;

		public SiteInfo(String siteID, String siteName, boolean primary) {
			this.siteID = siteID;
			this.siteName = siteName;
			this.primary = primary;
		}

		public String toString() {
			StringBuffer buf = new StringBuffer();
			buf.append("SiteInfo::[");
			buf.append("siteID=").append(siteID);
			buf.append(",siteName=").append(siteName);
			buf.append(",primary=").append(primary);
			buf.append(",uniqueId=").append(uniqueId);
			buf.append(']');
			return buf.toString();
		}
	}// ;

	public static void main(String[] args) throws Exception {
		DatabaseInfo dbi = new DatabaseInfo();
		dbi.hostName = "fmri-gpop.nbirn.net";
		DBUserInfo dbui = new DBUserInfo();
		dbui.name = "fbirn_test2";
		dbui.password = "testit";

		testJDBCConnection(dbi, dbui);
		Map<String, SiteInfo> map = getSites(dbi, dbui);

		System.out.println(map);
		SiteInfo si = new SiteInfo("1111", "test_site", false);
		si.uniqueId = 357280;
		// addSite(dbi, dbui, si);

		MinimalUserFileCreator muc = new MinimalUserFileCreator();
		String homeDir = System.getProperty("user.home");
		String xmlFilename = homeDir + "/users.xml";
		muc.prepareUsersFile(xmlFilename);

	}
}
