package clinical.tools.upload;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import clinical.server.vo.Experiment;
import clinical.server.vo.Humansubject;
import clinical.server.vo.Researchgroup;
import clinical.server.vo.Subjexperiment;
import clinical.utils.Assertion;
import clinical.utils.BIRNURIException;
import clinical.utils.BIRNURIUtils;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.ConnectionSupportMixin;
import clinical.web.Constants;
import clinical.web.IDerivedImageDataService;
import clinical.web.IExperimentManagement;
import clinical.web.ISubjectVisitManagement;
import clinical.web.MinimalServiceFactory;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.exception.SubjectVisitManagementException;
import clinical.web.vo.DerivedDataInfo;
import clinical.web.vo.Study;
import clinical.web.vo.StudySegment;
import clinical.web.vo.Subject;
import clinical.web.vo.Visit;
import edu.sdsc.grid.io.local.LocalFile;
import edu.sdsc.grid.io.srb.SRBFile;
import edu.sdsc.grid.io.srb.SRBFileSystem;

/**
 * Uploads partial Freesurfer derived image data WITHOUT image metadata to SRB
 * and to the primary HID. Official Freesurfer upload needs to use XCEDE + image
 * data.
 *
 * @author I. Burak Ozyurt
 * @version $Id: FSDataUpload4Slicer.java,v 1.1.2.1 2008/09/17 00:38:13 bozyurt
 *          Exp $
 */
public class FSDataUpload4Slicer {
	protected ConnectionSupportMixin mixin;
	protected String subjectID;
	protected String expName;
	protected String visitID;
	protected String segmentID;
	protected String studyID = "MRI__0001";
	protected String mrmlFile;
	protected String ddTarFile;
	protected String analysisShortName = "FreeSurfer";
	protected String snapshotVersion = "0001";

	public FSDataUpload4Slicer() throws Exception {
		// URL usersFile = getClass().getClassLoader().getResource("users.xml");
		mixin = new ConnectionSupportMixin("users.xml");
		ServiceFactory.setMimimalOpMode(true);
		MinimalServiceFactory.setMimimalOpMode(true);
		mixin.startup();
	}

	public void shutdown() {
		if (mixin != null)
			try {
				mixin.shutdown();
			} catch (Exception e) {
				e.printStackTrace();
			}
	}

	public void upload(boolean skipImageUpload) throws Exception {
		Assertion.assertNotNull(subjectID);
		Assertion.assertNotNull(visitID);
		Assertion.assertNotNull(segmentID);
		Assertion.assertNotNull(expName);

		String dbID = mixin.getDbID();
		UserInfo ui = mixin.getUi();

		IDBCache dbCache = ServiceFactory.getDBCache(dbID);
		IExperimentManagement em = ServiceFactory.getExperimentManagement(dbID);
		ISubjectVisitManagement svm = ServiceFactory
				.getSubjectVisitManagement(dbID);
		Humansubject hs = svm.getSubjectByID(ui, subjectID);
		Experiment exp = em.getExperiment(ui, expName);
		Assertion.assertNotNull(exp);

		if (hs == null) {
			Subject subject = new Subject(subjectID);

			List<Researchgroup> rgList = dbCache.getResearchGroups(ui, true);
			Researchgroup rg = selectRG(rgList, exp);

			svm.addSubject(ui, subject);

			em.enrollSubject(ui, exp.getUniqueid().intValue(), subjectID, rg
					.getUniqueid().intValue());
			addVisit(ui, svm, exp);

		} else {
			List<Subjexperiment> enrolledSubjects = em.getEnrolledSubjects(ui,
					exp.getUniqueid().intValue());
			Subjexperiment theSE = null;
			for (Subjexperiment se : enrolledSubjects)
				if (se.getSubjectid().equals(subjectID)) {
					theSE = se;
					break;
				}
			Assertion.assertNotNull(theSE);
			List<Visit> visits = svm.getAllVisitsForSubject(ui, subjectID);
			if (!hasVisit(visits)) {
				addVisit(ui, svm, exp);
			}
			// throw new Exception("Not supported yet!");
		}

		if (skipImageUpload)
			return;

		IDerivedImageDataService dids = ServiceFactory
				.getDerivedImageDataService(dbID);
		if (ddTarFile != null) {
			File f = new File(ddTarFile);
			Assertion.assertTrue(f.isFile());
			String dataURI = storeInSRB(ddTarFile, f.getName(), exp, false,
					"t1");
			addDDRec(ui, exp, dids, dataURI + "/" + f.getName());
		}

		if (mrmlFile != null) {
			File f = new File(mrmlFile);
			Assertion.assertTrue(f.isFile());
			String dataURI = storeInSRB(mrmlFile, f.getName(), exp, false, "t1");
			addDDRec(ui, exp, dids, dataURI + "/" + f.getName());
		}

	}

	protected boolean hasVisit(List<Visit> visits) {
		if (visits == null || visits.isEmpty())
			return false;
		int vid = GenUtils.toInt(visitID, 1);
		for (Visit v : visits) {
			if (v.getComponentID() == vid)
				return true;
		}
		return false;
	}

	protected void addVisit(UserInfo ui, ISubjectVisitManagement svm,
			Experiment exp) throws SubjectVisitManagementException {
		Visit v = new Visit();
		Timestamp ts = new Timestamp(System.currentTimeMillis());
		v.setSubjectID(subjectID);

		v.setVisitType("fMRI scan");
		v.setExperimentID(exp.getUniqueid().intValue());
		v.setName("slicer test");
		v.setComponentID(GenUtils.toInt(visitID, 1));
		v.setTimeStamp(ts);

		Study study = new Study();
		study.setName("fmri");
		study.setSubjectID(subjectID);
		study.setComponentID(GenUtils.toInt(visitID, 1));
		study.setExperimentID(exp.getUniqueid().intValue());
		study.setTimeStamp(ts);
		study.setIsTimeInterval(false);
		study.setStudyID(GenUtils.toInt(studyID, 1));

		StudySegment ss = new StudySegment();
		ss.setExperimentID(exp.getUniqueid().intValue());
		ss.setName("t1");
		ss.setSegmentID(1);
		ss.setStudyID(study.getStudyID());
		ss.setVisitID(v.getComponentID());
		ss.setTimeStamp(ts);
		ss.setProtocolID("anatomic scan: 3D FSPGR or T1 scan");
		ss.setProtocolVersion(1);
		ss.setSubjectID(subjectID);
		study.addStudySegment(ss);

		v.addStudy(study);
		svm.addVisitForSubject(ui, v);
	}

	protected void addDDRec(UserInfo ui, Experiment exp,
			IDerivedImageDataService dids, String dataURI) throws Exception {
		DerivedDataInfo ddi = prepareDerivedDataInfo(exp, dataURI);
		List<DerivedDataInfo> ddiList = new ArrayList<DerivedDataInfo>(1);
		ddiList.add(ddi);
		dids.addDerivedData(ui, ddiList);
	}

	protected Researchgroup selectRG(List<Researchgroup> rgList, Experiment exp)
			throws IOException {
		List<Researchgroup> ergList = new ArrayList<Researchgroup>(5);
		for (Researchgroup rg : rgList) {
			if (rg.getNcExperimentUniqueid().intValue() == exp.getUniqueid()
					.intValue())
				ergList.add(rg);
		}
		BufferedReader console = new BufferedReader(new InputStreamReader(
				System.in));
		Researchgroup theRG = null;
		while (true) {
			System.out.println(subjectID);
			System.out.println("Researchgroup\n-------------");
			int i = 0;
			for (Researchgroup rg : ergList) {
				System.out.println(i + ") " + rg.getName());
				i++;
			}
			System.out.print("Your selection:");
			String ans = console.readLine().trim();
			int selIdx = GenUtils.toInt(ans, -1);
			if (selIdx != -1) {
				theRG = ergList.get(selIdx);
				break;
			}
		}
		return theRG;
	}

	protected DerivedDataInfo prepareDerivedDataInfo(Experiment exp,
			String dataURI) {
		DerivedDataInfo ddi = new DerivedDataInfo();
		ddi.setDataURI(dataURI);
		ddi.setSubjectID(subjectID);
		ddi.setExperimentID(new Integer(exp.getUniqueid().intValue()));
		ddi.setVisitID(new Integer(visitID));
		ddi.setSegmentID(new Integer(segmentID));
		ddi.setBad(false);
		ddi.setRaw(false);
		return ddi;
	}

	protected String storeInSRB(String localPath, String imageFilename,
			Experiment theExp, boolean skip, String episodeID) throws Exception {
		String existingURI = prepDataURI(theExp, episodeID);
		System.out.println("existingURI:" + existingURI);
		SRBFileSystem srbFileSystem = null;
		String analysisURI = null;
		try {
			srbFileSystem = new SRBFileSystem();
			prepareSRBPath(srbFileSystem, existingURI, theExp.getBaseuri());

			SRBFile srbFile = new SRBFile(srbFileSystem, existingURI);
			Assertion.assertTrue(srbFile.exists() && srbFile.isDirectory());
			analysisURI = prepAnalysisURI(existingURI);
			System.out.println("analysisURI =" + analysisURI);
			if (!skip) {
				srbFile = new SRBFile(srbFileSystem, analysisURI);
				if (srbFile.exists() || srbFile.mkdirs()) {

					LocalFile lf = null;
					lf = new LocalFile(localPath);
					String uri = analysisURI + "/" + imageFilename;
					System.out.println("uri=" + uri);
					SRBFile srbF = new SRBFile(srbFileSystem, uri);
					lf.copyTo(srbF);
				} else {
					throw new Exception("Cannot create collection: " + srbFile);
				}
			}
		} finally {
			if (srbFileSystem != null) {
				try {
					srbFileSystem.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
		return analysisURI;
	}

	protected void prepareSRBPath(SRBFileSystem sfs, String path,
			String basePath) {
		Assertion.assertTrue(path.startsWith(basePath));
		String remPath = path.substring(basePath.length());
		remPath = remPath.replaceFirst("^\\/+", "");
		String[] toks = remPath.split("\\/+");
		boolean pathOk = true;
		int idx = 0;
		String s = basePath;
		s = s.replaceFirst("\\/+$", "");
		for (int i = 0; i < toks.length; i++) {
			s += "/" + toks[i];
			idx = i;
			SRBFile srbFile = new SRBFile(sfs, s);
			if (!srbFile.exists() || !srbFile.isDirectory()) {
				pathOk = false;
				break;
			}
		}
		if (!pathOk) {
			s = basePath;
			s = s.replaceFirst("\\/+$", "");
			for (int i = 0; i < idx; i++) {
				s += "/" + toks[i];
			}
			for (int i = idx; i < toks.length; i++) {
				s += "/" + toks[i];
				SRBFile srbFile = new SRBFile(sfs, s);
				srbFile.mkdir();
			}
		}
	}

	protected String prepAnalysisURI(String uri) {
		StringBuffer buf = new StringBuffer(256);
		buf.append(uri).append("/Analysis/");
		// TODO which siteID?
		String siteID = subjectID.substring(0, 4);
		buf.append(analysisShortName).append("__");
		buf.append(siteID).append("__").append(snapshotVersion);
		return buf.toString();
	}

	protected String prepDataURI(Experiment exp, String episodeID)
			throws BIRNURIException {
		StringBuilder buf = new StringBuilder(256);
		String baseURI = exp.getBaseuri();

		if (Constants.USE_BIRN_URIS) {
			baseURI = BIRNURIUtils.getPath(baseURI);
		}

		buf.append(baseURI).append('/');
		buf.append(subjectID).append('/');
		String siteID = subjectID.substring(0, 4);
		// FIXME which siteID?
		buf.append("scanVisit__").append(siteID);
		String visitIDStr = GenUtils
				.leftPad(visitID, '0', 4 - visitID.length());
		buf.append("__").append(visitIDStr).append('/');
		buf.append(studyID).append('/');
		buf.append(episodeID);
		String uri = buf.toString();
		uri = uri.replaceAll("/+", "/");
		return uri;
	}

	public String getSubjectID() {
		return subjectID;
	}

	public void setSubjectID(String subjectID) {
		this.subjectID = subjectID;
	}

	public String getVisitID() {
		return visitID;
	}

	public void setVisitID(String visitID) {
		this.visitID = visitID;
	}

	public String getSegmentID() {
		return segmentID;
	}

	public void setSegmentID(String segmentID) {
		this.segmentID = segmentID;
	}

	public String getMrmlFile() {
		return mrmlFile;
	}

	public void setMrmlFile(String mrmlFile) {
		this.mrmlFile = mrmlFile;
	}

	public String getDdTarFile() {
		return ddTarFile;
	}

	public void setDdTarFile(String ddTarFile) {
		this.ddTarFile = ddTarFile;
	}

	public String getExpName() {
		return expName;
	}

	public void setExpName(String expName) {
		this.expName = expName;
	}

	public static ArgInfo getArgValue(String[] args, int idx,
			Set<String> boolArgs) {
		if (boolArgs.contains(args[idx])) {
			return new ArgInfo("true", 1);
		}
		if (args.length <= (idx + 1)) {
			return null;
		}
		if (args[idx + 1].startsWith("-")) {
			return null;
		}
		String s = args[idx + 1];
		int i = idx + 2;
		int incr = 2;
		s = s.replaceFirst("^\"|'", "");
		while (i < args.length && !args[i].startsWith("-")) {
			String tok = args[i].replaceFirst("\"|'$", "");
			s += " " + tok;
			i++;
			incr++;
		}
		// System.out.println(">> " + s);
		return new ArgInfo(s, incr);
	}

	public static class ArgInfo {
		String argValue;
		int incr;

		public ArgInfo(String argValue, int idx) {
			super();
			this.argValue = argValue;
			this.incr = idx;
		}

		public String getArgValue() {
			return argValue;
		}

		public int getIncr() {
			return incr;
		}
	}

	public static void usage() {
		System.err.println("Usage:");
		System.err.println("FSDataUpload4Slicer -conf <properties-file>");
		System.exit(1);
	}

	public static Properties getProps(String configFilePath) throws IOException {
		BufferedInputStream in = null;
		try {
			in = new BufferedInputStream(new FileInputStream(configFilePath));
			Properties props = new Properties();
			props.load(in);

			return props;
		} finally {
			FileUtils.close(in);
		}
	}

	public static void main(String[] args) throws Exception {

		Set<String> boolArgs = new HashSet<String>(3);
		int i = 0;
		String propsFile = null;
		while (i < args.length) {
			ArgInfo ai = getArgValue(args, i, boolArgs);
			if (ai == null) {
				usage();
			} else if (args[i].equals("-conf")) {
				propsFile = ai.getArgValue();
			}
			i += ai.getIncr();
		}
		if (propsFile == null)
			usage();

		FSDataUpload4Slicer upload = null;
		try {
			upload = new FSDataUpload4Slicer();

			Properties props = getProps(propsFile);
			upload.setSubjectID(props.getProperty("subjectID"));
			upload.setExpName(props.getProperty("expName"));
			upload.setVisitID(props.getProperty("visitID"));
			upload.setSegmentID(props.getProperty("segmentID"));
			upload.setDdTarFile(props.getProperty("ddTarFile"));
			upload.setMrmlFile(props.getProperty("mrmlFile"));
			/*
			 * upload.setSubjectID("000670986943");
			 * upload.setExpName("fBIRNPhaseII__0010"); upload.setVisitID("1");
			 * upload.setSegmentID("1"); upload
			 * .setDdTarFile("/home/bozyurt3/data/000670986943/4slicer.tar.gz");
			 * upload
			 * .setMrmlFile("/home/bozyurt3/data/000670986943/FreeSurferOnly.mrml");
			 */
			upload.upload(false);
		} finally {
			if (upload != null)
				upload.shutdown();
		}
	}
}
