package clinical.tools.upload;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;

import org.apache.oro.text.perl.Perl5Util;

import clinical.server.dao.AssessmentDAO;
import clinical.server.dao.AssessmentscoreDAO;
import clinical.server.vo.Assessment;
import clinical.server.vo.Assessmentscore;
import clinical.utils.CSVParser;
import clinical.utils.GenUtils;
import clinical.web.DAOFactory;
import clinical.web.common.UserInfo;
import clinical.web.common.security.DBConfig;
import clinical.web.services.DBPoolService;
import clinical.web.services.SimpleSecurityService;

public class RangeDataPLSQLGenerator {
	CSVParser parser;
	String cvsFilename;
	UserInfo ui;
	DBPoolService dbPoolService;
	List<Assessmentscore> scoreList;
	Map<BigDecimal, Assessment> assessmentMap;
	Perl5Util perl = new Perl5Util();
	int userID = 1;
	int rangeValidatorTableID = 53;
	int scoreValidatorTableID = 54;
	String dbID;

	public RangeDataPLSQLGenerator(String cvsFilename) throws IOException {
		parser = new CSVParser();
		parser.extractData(cvsFilename);
	}

	protected void startup(String propsFile) throws Exception {

		Properties props = GenUtils.loadProperties(propsFile);
		String driverClass = props.getProperty("mbirn.driver_class");
		Class.forName(driverClass);

		String usersFile = props.getProperty("mbirn.users_file");
		String adminUser = props.getProperty("mbirn.admin_user");
		String adminPwd = props.getProperty("mbirn.admin_pwd");
		this.dbID = props.getProperty("mbirn.dbid");

		SimpleSecurityService secService = SimpleSecurityService
				.getInstance(usersFile);

		dbPoolService = DBPoolService.getInstance(secService, secService
				.getDBConfigMap());
		DBPoolService poolService;
		for (DBConfig dbConfig : secService.getDBConfigMap().values()) {
			poolService = DBPoolService.getInstance(dbConfig.getId());
			poolService.startup();
			secService.prepareTableCache(dbConfig.getId());

		}
		ui = secService.authenticate(adminUser, adminPwd, dbID);
	}

	protected void shutdown() throws Exception {
		if (dbPoolService != null)
			try {
				dbPoolService.shutdown();
			} catch (Exception x) {
			}
	}

	public void generate() throws Exception {
		loadScores();

		List<List<String>> rows = parser.getRows();
		PrintWriter out = new PrintWriter(new OutputStreamWriter(System.out));
		prepareHeader(out);

		int rowIdx = 0;
		for (List<String> row : rows) {
			if (rowIdx++ < 2)
				continue;
			// System.out.println( row.get(1) );
			Range range = parseRange(row.get(2));
			if (range != null) {
				// System.out.println("range="+ range.low + " - " +
				// range.upper);
			}
			if (range != null) {
				Assessmentscore as = findScore(row.get(1));
				if (as != null) {
					// System.out.println("found score "+ as.toString() );
					prepareRangeEntry(out, range, as);
				}
			}
		}
		prepareFooter(out);
		out.flush();
	}

	private void prepareHeader(PrintWriter out) throws IOException {
		out.println("declare");
		out.println("\tuid NUMBER;");
		out.println("BEGIN");
		out.println();
	}

	private void prepareRangeEntry(PrintWriter out, Range range,
			Assessmentscore score) throws IOException {
		out.println();
		out.println("select uid_seq.nextval into uid from dual;");

		StringBuffer sb = new StringBuffer(128);
		sb.append("insert into nc_rangevalidator values(uid,").append(
				rangeValidatorTableID);
		sb.append(",").append(userID).append(", SYSDATE,").append(userID);
		sb.append(",").append(range.low).append(",").append(range.upper)
				.append(", 'Y','Y');");
		out.println(sb.toString());

		sb = new StringBuffer(128);
		sb.append("insert into nc_scorevalidator values(uid_seq.nextval,");
		sb.append(scoreValidatorTableID).append(',').append(userID).append(
				",SYSDATE,");
		sb.append(userID).append(',')
				.append(score.getAssessmentid().intValue());
		sb.append(",'").append(score.getScorename()).append("',").append(
				rangeValidatorTableID);
		sb.append(",uid);");
		out.println(sb.toString());
	}

	private void prepareFooter(PrintWriter out) throws IOException {
		out.println("END;");
		out.println("/");
	}

	private Range parseRange(String rangeStr) {
		rangeStr = rangeStr.trim();
		if (perl.match("m/(\\d+)-(\\d+)/", rangeStr)) {
			String lb = perl.group(1);
			String ub = perl.group(2);
			Range r = new Range();
			r.low = Integer.parseInt(lb);
			r.upper = Integer.parseInt(ub);
			return r;
		} else if (perl.match("m/([-]?\\d+)\\s*to\\s*[\\+]?(\\d+)/", rangeStr)) {
			String lb = perl.group(1);
			String ub = perl.group(2);
			Range r = new Range();
			r.low = Integer.parseInt(lb);
			r.upper = Integer.parseInt(ub);
			return r;
		}
		return null;
	}

	private Assessmentscore findScore(String scoreDesc) {
		scoreDesc = scoreDesc.trim();
		StringTokenizer stok = new StringTokenizer(scoreDesc);
		List<String> tokens = new ArrayList<String>();
		while (stok.hasMoreTokens()) {
			String tok = stok.nextToken();
			if (tok.indexOf("(") != -1) {
				break;
			}
			tokens.add(tok.toLowerCase());
		}

		for (Assessmentscore score : scoreList) {
			String scoreName = score.getScorename().toLowerCase();
			if (scoreName.startsWith(tokens.get(0))) {
				if (doesMatch(tokens, scoreName)) {
					return score;
				}
			}
		}

		return null;
	}

	private boolean doesMatch(List<String> tokens, String scoreName) {

		// just for the scaled scores
		if (!foundToken(tokens, "scale")) {
			return false;
		}

		StringTokenizer stok = new StringTokenizer(scoreName);

		String[] scoreNameToks = new String[stok.countTokens()];
		int i = 0;
		while (stok.hasMoreTokens()) {
			scoreNameToks[i++] = stok.nextToken();
		}
		int count = 0;
		i = 0;
		for (String tok : tokens) {
			if (i >= scoreNameToks.length)
				break;
			if (tok.equals(scoreNameToks[i++])) {
				count++;
			}
		}
		double ratio = count / (double) tokens.size();

		if (ratio > 0.5 && foundToken(tokens, "score")) {
			dumpScoreNames(tokens, scoreName, ratio);
		}

		return (ratio >= 0.9);
	}

	static boolean foundToken(List<String> tokens, String tok) {
		for (String item : tokens) {
			if (tok.equals(item)) {
				return true;
			}
		}
		return false;
	}

	static void replaceToken(List<String> tokens, String oldTok, String newTok1,
			String newTok2) {
		int i = -1;
		int idx = -1;
		for (String item : tokens) {
			++i;
			if (oldTok.equals(item)) {
				idx = i;
				break;
			}
		}
		if (idx != -1) {
			tokens.set(idx, newTok1);
			tokens.add(idx, newTok2);
		}

	}

	static void dumpScoreNames(List<String> tokens, String scoreName, double ratio) {
		StringBuffer sb = new StringBuffer();
		for (String item : tokens) {
			sb.append(item).append(' ');
		}
		System.err.println(sb.toString() + " - " + scoreName + "(" + ratio
				+ ")");
	}

	protected void loadScores() throws Exception {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			this.assessmentMap = getAssessments(con);
			this.scoreList = getScores(con);

		} finally {
			if (con != null)
				dbPoolService.releaseConnection(ui.getName(), con);
		}
	}

	Map<BigDecimal, Assessment> getAssessments(Connection con) throws Exception {
		AssessmentDAO dao = DAOFactory.createAssessmentDAO(this.dbID);
		List<Assessment> asList = dao.find(con, new Assessment());
		Map<BigDecimal, Assessment> asMap = new LinkedHashMap<BigDecimal, Assessment>();
		for (Assessment as : asList) {
			asMap.put(as.getAssessmentid(), as);
		}
		return asMap;
	}

	List<Assessmentscore> getScores(Connection con) throws Exception {
		AssessmentscoreDAO dao = DAOFactory.createAssessmentscoreDAO(dbID);
		List<Assessmentscore> scoreList = dao.find(con, new Assessmentscore());
		return scoreList;
	}

	class Range {
		int low, upper;
	}//;

	public static void main(String[] args) {
		RangeDataPLSQLGenerator gen = null;
		try {
			gen = new RangeDataPLSQLGenerator(
					"/home/bozyurt/mbirn/MBIRN_clin_Vars01_28_04_revised.csv");
			gen.startup("test.properties");

			gen.generate();
		} catch (Exception x) {
			x.printStackTrace();
		} finally {
			if (gen != null) {
				try {
					gen.shutdown();
				} catch (Exception e) {
				}
			}
		}

	}
}
