package clinical.utils;

import java.io.DataInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import clinical.web.vo.HeaderFieldType;

public class AnalyzeHeaderReader {
	private String headerFile;
	private boolean littleEndian = false;
	public final static SimpleDateFormat tsFormat = new SimpleDateFormat(
			"dd-MMM-yyyy hh:mm:ss");
	Map<String, Field<Integer>> integerFieldMap = new HashMap<String, Field<Integer>>(
			7);
	Map<String, Field<String>> stringFieldMap = new HashMap<String, Field<String>>(
			7);
	Map<String, Field<Float>> realFieldMap = new HashMap<String, Field<Float>>(
			7);

	public AnalyzeHeaderReader(String headerFile) {
		super();
		this.headerFile = headerFile;
	}
	
	public HeaderFields extractFields(List<HeaderFieldType> hftList) {
		HeaderFields hf = new HeaderFields();
		Set<String> seenSet = new HashSet<String>();
		for (HeaderFieldType hft : hftList) {
			if (hft.getType().equals(HeaderFieldType.STRING)) {
				Field<String> field = getStringField(hft.getExtractionName());
				if (field != null) {
					hf.addStringField(field, hft);
					seenSet.add(hft.getExtractionName());
				}
			}
		}
		return hf;
	}
	
	public Field<String> getStringField(String name) {
		return stringFieldMap.get(name);
	}
	
	public List<String> getStringFieldNames() {
		List<String> names = new ArrayList<String>(stringFieldMap.keySet());
		Collections.sort(names);
		return names;
	}

	/**
	 * based on ImageJ plugin by Guy Williams, gbw1000@wbic.cam.ac.uk
	 * http://rsb.info.nih.gov/ij/plugins/download/Analyze_Reader.java
	 * 
	 * @throws IOException
	 */
	public void readHeader() throws IOException {
        Pattern datePattern = Pattern.compile("(\\d{2,2}\\-\\w{3,3}\\-\\d{4,4})\\s+(\\d{2,2}:\\d+:\\d+)\\.");
		DataInputStream in = null;
		int i;
		byte[] units = new byte[4];

		try {
			in = new DataInputStream(new FileInputStream(this.headerFile));
			// header_key

			in.readInt(); // sizeof_hdr
			for (i = 0; i < 10; i++)
				in.read(); // data_type
			for (i = 0; i < 18; i++)
				in.read(); // db_name
			in.readInt(); // extents
			in.readShort(); // session_error
			in.readByte(); // regular
			in.readByte(); // hkey_un0

			// image_dimension

			short endian = readShort(in); // dim[0]
			if ((endian < 0) || (endian > 15)) {
				littleEndian = true;
			}
			readShort(in); // dim[1]
			readShort(in); // dim[2]
			readShort(in); // dim[3]
			in.readShort(); // dim[4]
			for (i = 0; i < 3; i++)
				in.readShort(); // dim[5-7]
			in.read(units, 0, 4); // vox_units

			for (i = 0; i < 8; i++)
				in.read(); // cal_units[8]
			in.readShort(); // unused1
			readShort(in); // datatype
			readShort(in); // bitpix
			in.readShort(); // dim_un0
			in.readFloat(); // pixdim[0]
			readFloat(in); // pixdim[1]
			readFloat(in); // pixdim[2]
			readFloat(in); // pixdim[3]
			for (i = 0; i < 4; i++)
				in.readFloat(); // pixdim[4-7]
			readFloat(in); // vox_offset
			in.readFloat(); // roi_scale
			in.readFloat(); // funused1
			in.readFloat(); // funused2
			in.readFloat(); // cal_max
			in.readFloat(); // cal_min
			in.readInt(); // compressed
			in.readInt(); // verified
			readInt(in); // (int) s.max // glmax
			readInt(in); // (int) s.min // glmin

			// data_history

			StringBuilder sb = new StringBuilder(81);
			for (i = 0; i < 80; i++) {
				sb.append((char) in.read());
			} // descrip
			String description = sb.toString();
			System.out.println("Description:" + description);
			Matcher m = datePattern.matcher(description);
			if (m.find()) {
				System.out.println("date:" + m.group(1));
				System.out.println("time:" + m.group(2));
				Field<String> field = new Field<String>("timestamp");
				field.value = m.group(1) + " " + m.group(2);
				stringFieldMap.put("timestamp", field);
			}
			
			for (i = 0; i < 24; i++) {
				in.read();
			}// aux_file
			in.read(); // orient
			for (i = 0; i < 10; i++) {
				in.read();
			} // originator
			for (i = 0; i < 10; i++) {
				in.read();
			} // generated
			for (i = 0; i < 10; i++) {
				in.read();
			} // scannum
			for (i = 0; i < 10; i++) {
				in.read();
			}// patient_id
			sb.setLength(0);
			for (i = 0; i < 10; i++) {
				sb.append((char) in.read());
			}// exp_date
			System.out.println("Exp Date:" + sb.toString());
			sb.setLength(0);
			for (i = 0; i < 10; i++) {
				sb.append((char) in.read());
			} // exp_time
			System.out.println("Exp Time:" + sb.toString());
			for (i = 0; i < 3; i++) {
				in.read();
			} // hist_un0
			in.readInt(); // views
			in.readInt(); // vols_added
			in.readInt(); // start_field
			in.readInt(); // field_skip
			in.readInt(); // omax
			in.readInt(); // omin
			in.readInt(); // smax
			in.readInt(); // smin

		} finally {
			FileUtils.close(in);
		}

	}

	private int readInt(DataInputStream input) throws IOException {
		if (!littleEndian)
			return input.readInt();
		byte b1 = input.readByte();
		byte b2 = input.readByte();
		byte b3 = input.readByte();
		byte b4 = input.readByte();
		return ((((b4 & 0xff) << 24) | ((b3 & 0xff) << 16) | ((b2 & 0xff) << 8) | (b1 & 0xff)));
	}

	private short readShort(DataInputStream input) throws IOException {
		if (!littleEndian)
			return input.readShort();
		byte b1 = input.readByte();
		byte b2 = input.readByte();
		return ((short) (((b2 & 0xff) << 8) | (b1 & 0xff)));
	}

	private float readFloat(DataInputStream input) throws IOException {
		if (!littleEndian)
			return input.readFloat();
		int orig = readInt(input);
		return (Float.intBitsToFloat(orig));
	}

	public static void main(String[] args) throws Exception {

		String headerFile = "/tmp/PEDSS016/PEDS016_s100901183246DST131221107523235343-0002-00001-000192-01.hdr";
		AnalyzeHeaderReader reader = new AnalyzeHeaderReader(headerFile);
		reader.readHeader();
	}
}
