package clinical.web;

import java.io.File;

import clinical.web.common.IAuthenticationService;
import clinical.web.common.IAuthorizationService;
import clinical.web.common.IDBCache;
import clinical.web.common.IDBPoolService;
import clinical.web.common.ISecurityAdminService;
import clinical.web.common.ISecurityService;
import clinical.web.common.query.IQueryProcessor;
import clinical.web.common.query.OracleQueryProcessor;
import clinical.web.common.query.OracleSQLDialect;
import clinical.web.common.query.PostgresQueryProcessor;
import clinical.web.common.query.PostgresSQLDialect;
import clinical.web.common.query.SearchPredicateList;
import clinical.web.common.security.DBConfig;
import clinical.web.exception.BaseException;
import clinical.web.services.AnalysisResultServiceImpl;
import clinical.web.services.AppConfigService;
import clinical.web.services.AssessmentEntryServiceImpl;
import clinical.web.services.AssessmentService;
import clinical.web.services.BatchAssessmentServiceImpl;
import clinical.web.services.BugReportServiceImpl;
import clinical.web.services.CBFGroupAnalysisServiceImpl;
import clinical.web.services.CBFImageTypeLookupService;
import clinical.web.services.CBFProcReportQueryService;
import clinical.web.services.CBFROIGroupAnalysisServiceImpl;
import clinical.web.services.CBFSearchServiceImpl;
import clinical.web.services.CBFStandardSpaceGroupAnalysisServiceImpl;
import clinical.web.services.CBFWFManagementService;
import clinical.web.services.ContactInfoService;
import clinical.web.services.DBCache;
import clinical.web.services.DBPoolService;
import clinical.web.services.DBUserManServiceImpl;
import clinical.web.services.DataArchivalService;
import clinical.web.services.DerivedDataService;
import clinical.web.services.DerivedImageDataServiceImpl;
import clinical.web.services.DownloadJobServiceImpl;
import clinical.web.services.ExperimentManagementImpl;
import clinical.web.services.GroupCBFReportQueryService;
import clinical.web.services.HelpService;
import clinical.web.services.IAssessmentEntryService;
import clinical.web.services.IBIRNIDGenerationService;
import clinical.web.services.IBatchAssessmentService;
import clinical.web.services.IBugReportService;
import clinical.web.services.ICBFGroupAnalysisService;
import clinical.web.services.ICBFROIGroupAnalysisService;
import clinical.web.services.ICBFSearchService;
import clinical.web.services.ICBFStandardSpaceGroupAnalysisService;
import clinical.web.services.IContactInfoService;
import clinical.web.services.IDataArchivalService;
import clinical.web.services.IDownloadJobService;
import clinical.web.services.IGroupCBFReportQueryService;
import clinical.web.services.IJobFilterManagementService;
import clinical.web.services.IJobProvenanceService;
import clinical.web.services.IJobResultService;
import clinical.web.services.IUsageManagementService;
import clinical.web.services.ImageDataServices;
import clinical.web.services.ImageHandlerImpl;
import clinical.web.services.JobFilterManagementServiceImpl;
import clinical.web.services.JobManagementServiceImpl;
import clinical.web.services.JobProvenanceManServiceImpl;
import clinical.web.services.JobResultManService;
import clinical.web.services.OracleSequenceHelper;
import clinical.web.services.PostgresSequenceHelper;
import clinical.web.services.PostgresSubjectAssessmentManImpl;
import clinical.web.services.ProvenanceServiceImpl;
import clinical.web.services.RandomBIRNIDGenerator;
import clinical.web.services.RemoteDBServicesImpl;
import clinical.web.services.SecurityService;
import clinical.web.services.SimpleSecurityService;
import clinical.web.services.SubjectAssessmentManagementImpl;
import clinical.web.services.SubjectVisitHandlerImpl;
import clinical.web.services.SubjectVisitManagementImpl;
import clinical.web.services.SynonymExpansionService;
import clinical.web.services.SynonymExpansionServiceImpl;
import clinical.web.services.UploadManagementService;
import clinical.web.services.UsageManagementService;
import clinical.web.services.XCEDEDerivedDataServiceImpl;

/**
 * 
 * collection of factory methods for different services
 * 
 * @author I. Burak Ozyurt
 * @version $Id: ServiceFactory.java 834 2013-11-05 02:05:37Z bozyurt $
 */

public class ServiceFactory {
	private static boolean mimimalOpMode = false;

	protected ServiceFactory() {
	}

	public static IDBPoolService getPoolService(String dbID) {
		return DBPoolService.getInstance(dbID);
	}

	public static ISecurityService getSecurityService() throws BaseException {
		if (isMimimalOpMode()) {
			return SimpleSecurityService.getInstance();
		} else {
			return (ISecurityService) SecurityService.getInstance();
		}
	}

	public static IAuthenticationService getAuthenticationService()
			throws BaseException {
		return (IAuthenticationService) SecurityService.getInstance();
	}

	public static IAuthorizationService getAuthorizationService()
			throws BaseException {
		return (IAuthorizationService) SecurityService.getInstance();
	}

	public static ISecurityAdminService getSecurityAdminService()
			throws BaseException {
		return (ISecurityAdminService) SecurityService.getInstance();
	}

	public static IAssessmentService getAssessmentService(String dbID)
			throws BaseException {
		return new AssessmentService(dbID);
	}

	public static IDerivedDataService getDerivedDataService(String dbID)
			throws BaseException {
		return new DerivedDataService(dbID);
	}

	public static IJobManagementService getJobManagementService(String dbID)
			throws BaseException {
		return new JobManagementServiceImpl(dbID);
	}

	public static IJobProvenanceService getJobProvenanceService(String dbID)
			throws BaseException {
		return new JobProvenanceManServiceImpl(dbID);
	}

	public static IJobResultService getJobResultService(String dbID)
			throws BaseException {
		return new JobResultManService(dbID);
	}

	public static ImageHandler getImageHandlerService() throws BaseException {
		// TODO ImageHandlerImpl needs updating
		String srbHome = getAppConfigService().getParamValue("srb.home");
		String srbPath = srbHome + File.separator; // + ".srb";
		return (ImageHandler) ImageHandlerImpl.getInstance(srbPath);
	}

	public static ISubjectVisitHandler getSubjectVisitHandler()
			throws BaseException {
		return (ISubjectVisitHandler) SubjectVisitHandlerImpl.getInstance();
	}

	public static ISubjectVisitManagement getSubjectVisitManagement(String dbID)
			throws BaseException {
		return (ISubjectVisitManagement) new SubjectVisitManagementImpl(dbID);
	}

	public static IDBCache getDBCache(String dbID) throws BaseException {
		return (IDBCache) DBCache.getInstance(dbID);
	}

	public static IHelpService getHelpService() throws BaseException {
		try {
			return (IHelpService) HelpService.getInstance();
		} catch (Exception e) {
			throw new BaseException(e);
		}
	}

	public static ISubjectAssessmentManagement getSubjectAssessmentManagement(
			String dbID) throws BaseException {
		ISecurityService isec = getSecurityService();
		String dbType = isec.getDBType(dbID);
		if (dbType.equals(DBConfig.ORACLE)) {
			return (ISubjectAssessmentManagement) new SubjectAssessmentManagementImpl(
					dbID);
		} else if (dbType.equals(DBConfig.POSTGRES)) {
			return (ISubjectAssessmentManagement) new PostgresSubjectAssessmentManImpl(
					dbID);
		} else {
			throw new BaseException("Not a supported database type:" + dbType);
		}
	}

	public static IProvenanceService getProvenanceService(String dbID)
			throws BaseException {
		return new ProvenanceServiceImpl(dbID);
	}

	public static IXCEDEDerivedDataService getXCEDEDerivedDataService(
			String dbID) throws BaseException {
		return new XCEDEDerivedDataServiceImpl(dbID);
	}

	public static IDerivedImageDataService getDerivedImageDataService(
			String dbID) throws BaseException {
		return new DerivedImageDataServiceImpl(dbID);
	}

	public static IQueryProcessor getQueryProcessor(String dbID,
			Class<?> valueObjectClass, SearchPredicateList spList)
			throws BaseException {
		ISecurityService isec = getSecurityService();
		String dbType = isec.getDBType(dbID);
		if (dbType.equals(DBConfig.ORACLE)) {
			try {
				return (IQueryProcessor) new OracleQueryProcessor(
						valueObjectClass, spList);
			} catch (Exception ex) {
				throw new BaseException(ex);
			}
		} else if (dbType.equals(DBConfig.POSTGRES)) {
			try {
				return (IQueryProcessor) new PostgresQueryProcessor(
						valueObjectClass, spList);
			} catch (Exception ex1) {
				throw new BaseException(ex1);
			}
		} else {
			throw new BaseException("Not a supported database type:" + dbType);
		}
	}

	public static IImageDataService getImageDataService(String dbID)
			throws BaseException {
		return new ImageDataServices(dbID);
	}

	public static ISQLDialect getSQLDialect(String dbID) throws BaseException {
		ISecurityService isec = getSecurityService();
		String dbType = isec.getDBType(dbID);
		if (dbType.equals(DBConfig.ORACLE)) {
			return new OracleSQLDialect();
		} else if (dbType.equals(DBConfig.POSTGRES)) {
			return new PostgresSQLDialect();
		} else {
			throw new BaseException("Not a supported database type:" + dbType);
		}
	}

	public static IExperimentManagement getExperimentManagement(String dbID)
			throws BaseException {
		return (IExperimentManagement) new ExperimentManagementImpl(dbID);
	}

	public static IDataArchivalService getDataArchivalService(String dbID)
			throws BaseException {
		return new DataArchivalService(dbID);
	}

	public static ISequenceHelper getSequenceHelper(String dbID)
			throws BaseException {
		ISecurityService isec = getSecurityService();
		String dbType = isec.getDBType(dbID);
		if (dbType.equals(DBConfig.ORACLE)) {
			return (ISequenceHelper) OracleSequenceHelper.getInstance(dbID,
					dbType);
		} else if (dbType.equals(DBConfig.POSTGRES)) {
			return (ISequenceHelper) PostgresSequenceHelper.getInstance(dbID,
					dbType);
		} else {
			throw new BaseException("Not a supported database type:" + dbType);
		}
	}

	public static IAppConfigService getAppConfigService() throws BaseException {
		return AppConfigService.getInstance();
	}

	public static IRemoteDBServices getRemoteDBServices() {
		return RemoteDBServicesImpl.getInstance();
	}

	public synchronized static boolean isMimimalOpMode() {
		return mimimalOpMode;
	}

	public synchronized static void setMimimalOpMode(boolean mimimalOpMode) {
		ServiceFactory.mimimalOpMode = mimimalOpMode;
	}

	public static IDownloadJobService getDownloadJobService() {
		return new DownloadJobServiceImpl();
	}

	public static IDBUserManService getDBUserManService(String dbID)
			throws BaseException {
		return new DBUserManServiceImpl(dbID);
	}

	public static IAnalysisResultService getAnalysisResultService(String dbID)
			throws BaseException {
		return new AnalysisResultServiceImpl(dbID);
	}

	public static IUploadManagementService getUploadManagementService(
			String dbID) throws BaseException {
		return new UploadManagementService(dbID);
	}

	public static IBIRNIDGenerationService getBIRNIDGenerationService(
			String dbID) {
		return RandomBIRNIDGenerator.getInstance(dbID);
	}

	public static ICBFWFManagementService getCBFWFManagementService(String dbID)
			throws BaseException {
		return new CBFWFManagementService(dbID);
	}

	public static IBugReportService getBugReportService(String dbID)
			throws BaseException {
		return new BugReportServiceImpl(dbID);
	}

	public static ICBFProcReportQueryService getCBFProcReportQueryService(
			String dbID) throws BaseException {
		return new CBFProcReportQueryService(dbID);
	}

	public static IUsageManagementService getUsageManagementService(String dbID)
			throws BaseException {
		return new UsageManagementService(dbID);
	}

	public static IContactInfoService getContactInfoService(String dbID)
			throws BaseException {
		return new ContactInfoService(dbID);
	}

	public static CBFImageTypeLookupService getCBFImageTypeLookupService() {
		return CBFImageTypeLookupService.getInstance();
	}

	public static ICBFSearchService getCBFSearchService(String dbID)
			throws BaseException {
		return new CBFSearchServiceImpl(dbID);
	}

	public static ICBFROIGroupAnalysisService getCBFROIGroupAnalysisService(
			String dbID) throws BaseException {
		return new CBFROIGroupAnalysisServiceImpl(dbID);
	}

	public static IBatchAssessmentService getBatchAssessmentService(String dbID)
			throws BaseException {
		return new BatchAssessmentServiceImpl(dbID);
	}

	public static ICBFStandardSpaceGroupAnalysisService getCBFStandardSpaceGroupAnalysisService(
			String dbID) throws BaseException {
		return new CBFStandardSpaceGroupAnalysisServiceImpl(dbID);
	}

	public static ICBFGroupAnalysisService getCBFGroupAnalysisService(
			String dbID) throws BaseException {
		return new CBFGroupAnalysisServiceImpl(dbID);
	}

	public static IGroupCBFReportQueryService getGroupCBFReportQueryService(
			String dbID) throws BaseException {
		return new GroupCBFReportQueryService(dbID);
	}

	public static IJobFilterManagementService getJobFilterManagementService(
			String dbID) throws BaseException {
		return new JobFilterManagementServiceImpl(dbID);
	}

	public static IAssessmentEntryService getAssessmentEntryService(String dbID)
			throws BaseException {
		return new AssessmentEntryServiceImpl(dbID);
	}
	
	public static SynonymExpansionService getSynonymExpansionService(String dbID) throws BaseException {
		return new SynonymExpansionServiceImpl(dbID);
	}
}
