package clinical.web;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.imageio.ImageIO;
import javax.servlet.ServletException;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts.action.ActionServlet;
import org.apache.struts.action.PlugIn;
import org.apache.struts.config.ModuleConfig;
import org.apache.struts.config.PlugInConfig;

import clinical.cache.DBChangeNotifySupport;
import clinical.server.vo.Site;
import clinical.utils.GenUtils;
import clinical.utils.ThreadPoolMan;
import clinical.web.common.AssessmentMapping;
import clinical.web.common.MediatedQueryHelper;
import clinical.web.common.UserInfo;
import clinical.web.common.security.DBConfig;
import clinical.web.common.security.User;
import clinical.web.download.DownloadCachePolicy;
import clinical.web.exception.DBPoolServiceException;
import clinical.web.scheduler.JobScheduler;
import clinical.web.services.AppConfigService;
import clinical.web.services.CBFImageTypeLookupService;
import clinical.web.services.CBFROILookupService;
import clinical.web.services.DBPoolService;
import clinical.web.services.FederatedDBCacheServiceImpl;
import clinical.web.services.FileCacheCleanupService;
import clinical.web.services.HelpService;
import clinical.web.services.PeriodicConfigUpdateService;
import clinical.web.services.PeriodicServiceManagementService;
import clinical.web.services.SecurityService;
import clinical.web.services.SimpleSecurityService;
import clinical.web.services.WFFileCacheCleanupService;
import clinical.web.workflow.cbf.AbstractCBFProcessingJob.CBFProcessingJobFactory;

import com.jamonapi.Monitor;
import com.jamonapi.MonitorFactory;

/**
 * Provides lifecycle management for this application. At servlet container
 * startup, the following servies are started.
 * <ul>
 * <li>bootstrap Security Service</li>
 * <li>prepare query processor global cache</li>
 * <li>get the application specific properties and save them to application
 * scope</li>
 * <li>initialize ServiceFactory</li>
 * <li>initialize DAOFactory</li>
 * <li>initialize the assessment variable mapping for quasi-mediator</li>
 * <li>startup image file cache cleanup thread</li>
 * </ul>
 * 
 * @author I. Burak Ozyurt
 * @version $Id: ServicesPlugin.java 627 2012-07-14 00:53:57Z bozyurt $
 */

public class ServicesPlugin implements PlugIn {
	private DBPoolService poolService;
	private JobScheduler scheduler;
	private Log log = LogFactory.getLog(ServicesPlugin.class);

	public ServicesPlugin() {
	}

	public void destroy() {
		if (poolService != null) {
			try {
				poolService.shutdown();
			} catch (Exception x) {
			}
		}
		// gracefully shutdown scheduler
		if (scheduler != null) {
			scheduler.shutdown();
		}
		// shutdown cache manager
		CacheManager.getInstance().shutdown();
		ThreadPoolMan.getInstance().getExecutorService().shutdownNow();
	}

	/**
	 * Initializes this application when the Struts framework starts.
	 * <ul>
	 * <li>bootstrap Security Service</li>
	 * <li>prepare query processor global cache</li>
	 * <li>get the application specific properties and save them to application
	 * scope</li>
	 * <li>initialize ServiceFactory</li>
	 * <li>initialize the assessment variable mapping for mediator</li>
	 * <li>startup image file cache cleanup thread</li>
	 * </ul>
	 * 
	 * @param servlet
	 * @param config
	 * @throws ServletException
	 */
	public void init(ActionServlet servlet, ModuleConfig config)
			throws ServletException {
		PlugInConfig[] configs = config.findPlugInConfigs();
		PlugInConfig theConfig = null;
		String className = ServicesPlugin.class.getName();
		for (int i = 0; i < configs.length; i++) {
			if (configs[i].getClassName().equals(className)) {
				theConfig = configs[i];
				break;
			}
		}

		// disable performance monitoring
		MonitorFactory.setEnabled(false);

		Map<?, ?> props = theConfig.getProperties();
		String userInfoFile = (String) props.get("user_info_file");
		String userInfoXMLSchema = (String) props.get("user_info_xml_schema");
		String seriesTypeRulesXmlFile = (String) props
				.get("series_type_rules_file");
		String roiLutTypesXmlFile = (String) props.get("roi_lut_types_file");

		String dbURL = null;

		log.info("userInfoFile=" + userInfoFile);

		SimpleSecurityService secService = null;
		DBPoolService poolService = null;
		try {

			CBFImageTypeLookupService.getInstance(servlet.getServletContext()
					.getRealPath(seriesTypeRulesXmlFile));

			CBFROILookupService.getInstance(servlet.getServletContext()
					.getRealPath(roiLutTypesXmlFile));

			// initialize cache manager
			CacheManager.create();

			// initialize thread pool
			ThreadPoolMan.create(10);

			secService = SimpleSecurityService.getInstance(servlet
					.getServletContext().getRealPath(userInfoFile), servlet
					.getServletContext().getRealPath(userInfoXMLSchema));

			// set the dbURL to the dbURL of the current database
			dbURL = secService.getCurrentDBConfig().getDbURL();

			log.info("current dbURL=" + dbURL);
			poolService = DBPoolService.getInstance(secService,
					secService.getDBConfigMap());

			if (poolService == null) {
				throw new Exception("No reachable default database is setup!");
			}

			// bootstrap database based security service using a minimal
			// users.xml file
			String defaultDBID = secService.getDefaultDBID();
			String defaultDbType = secService.getDBType(defaultDBID);
			SecurityService ss = SecurityService.getInstance(poolService,
					defaultDBID, defaultDbType);
			DAOFactory.initialize();
			ss.startup();

			// prepare distributed cache event mechanism
			FederatedDBCacheServiceImpl fdbcs = new FederatedDBCacheServiceImpl();
			DBChangeNotifySupport dbcns = DBChangeNotifySupport.getInstance();
			dbcns.addDBChangeNotifyListener(fdbcs);

			// initialize application configuration service
			IAppConfigService configService = AppConfigService.getInstance(
					poolService, defaultDBID);

			// get the application specific properties and save them to
			// application scope
			// Properties appProps = GenUtils
			// .loadProperties("clinical.properties");
			// servlet.getServletContext().setAttribute(
			// Constants.APP_PROPERTIES_KEY, appProps);

			List<String> dbIDList = new ArrayList<String>(ss.getDBConfigMap()
					.size());

			Map<String, String> dbID2SiteIDMap = Collections
					.synchronizedMap(new HashMap<String, String>(17));

			Result result = initializeDBPoolService(ss, poolService, dbIDList,
					dbID2SiteIDMap);

			if (dbIDList.isEmpty()) {
				throw new Exception("Database ID list cannot be empty!");
			}
			Cache cache = CacheManager.getInstance().getCache("remote");
			Element el = new Element(Constants.DBID2SITEID_MAP, dbID2SiteIDMap);
			el.setEternal(true);
			cache.put(el);

			// servlet.getServletContext().setAttribute(Constants.DBID2SITEID_MAP,
			// dbID2SiteIDMap);

			log.info(" --- dbIDList --- ");
			for (String adbId : dbIDList) {
				log.info(adbId);
			}
			log.info(" --- end of dbIDList --- ");

			servlet.getServletContext().setAttribute(Constants.DBPOOL_KEY,
					poolService);

			servlet.getServletContext().setAttribute(Constants.DBID_LIST_KEY,
					dbIDList);

			servlet.getServletContext().setAttribute(
					Constants.DEFAULT_DBID_KEY, result.defaultDBID);

			//
			String mriJpegsDir = (String) props.get("mri_jpegs_dir");
			String cacheRoot = (String) props.get("cache_root");

			servlet.getServletContext().setAttribute(
					Constants.MRI_JPEGS_DIR_KEY, mriJpegsDir);
			servlet.getServletContext().setAttribute(Constants.CACHE_ROOT_KEY,
					cacheRoot);

			List<Site> allSites = ss.getAllSites();
			Properties siteIDMapProps = new Properties();
			for (Site site : allSites) {
				siteIDMapProps.put(site.getSiteid(), site.getSitename());
			}

			GenUtils.populateSiteIDMap(siteIDMapProps);

			boolean queryOnly = GenUtils.toBoolean(configService
					.getParamValue(Constants.QUERYONLY_OP_PROPERTY), false);
			if (!queryOnly) {
				// prepare data processing users mapped from the web users
				IDBUserManService dbums = ServiceFactory
						.getDBUserManService(defaultDBID);
				dbums.prepareKeyers(ss.getCurrentDBConfig());
			}

			// initialize service factory
			// ServiceFactory.setAppProperties(appProps);

			// initialize the assessment variable mapping for mediator
			String mappingFile = configService
					.getParamValue("mediator.mapping.file");
			Map<Integer, List<AssessmentMapping>> siteAsMap = MediatedQueryHelper
					.prepareAssesmentMapping(servlet.getServletContext()
							.getRealPath(mappingFile));

			servlet.getServletContext().setAttribute(
					Constants.SITE_ASSESSMENT_MAP_KEY, siteAsMap);

			// start image file cache cleanup service

			boolean useFileCacheCleanup = GenUtils.toBoolean(
					configService.getParamValue("file.cache.cleanup"), false);
			if (useFileCacheCleanup) {
				log.info("starting file cache cleanup service...");
				String fullCacheRoot = servlet.getServletContext().getRealPath(
						cacheRoot);
				FileCacheCleanupService cleanupService = FileCacheCleanupService
						.getInstance(fullCacheRoot, configService);

				startThread(cleanupService);
			}

			boolean showStats = GenUtils.toBoolean(
					configService.getParamValue("connection.pool.showStats"),
					false);
			for (DBConfig dbConfig : ss.getDBConfigMap().values()) {
				try {
					poolService = DBPoolService.getInstance(dbConfig.getId());
					poolService.setShowStats(showStats);
					if (showStats) {
						log.info("enabling statistics dump for "
								+ "connection pool for " + dbConfig.getId());
					}
				} catch (RuntimeException re) {
					log.error(re);
				}
			}

			// start job scheduler
		  log.info("realPath:" +  servlet.getServletContext().getRealPath("/"));

			String schedulerDir = servlet.getServletContext().getRealPath(
					Constants.SCHEDULER_DIR);
			schedulerDir = configService.getParamValue("download.cacheroot");
         log.info("schedulerDir:" + schedulerDir);
			if (!GenUtils.isNotEmpty(schedulerDir)
					|| !new File(schedulerDir).exists()) {
				// set default
				File sd = new File(System.getProperty("user.home"),
						"download_cache");
				if (!sd.exists()) {
					if (!sd.mkdir()) {
						throw new Exception(
								"Cannot create the default 'download.cacheroot' directory!:"
										+ schedulerDir);
					}
				}
				if (!sd.exists()) {
					throw new Exception(
							"'download.cacheroot' parameter needs to be set to "
									+ "a valid directory in the database! :"
									+ schedulerDir);
				}
				schedulerDir = sd.getAbsolutePath();
				configService.updateParameter("download.cacheroot",
						schedulerDir);
			}

			log.info("**** starting job scheduler with scheduler cache root:"
					+ schedulerDir);
			scheduler = JobScheduler.getInstance(schedulerDir,
					DownloadCachePolicy.getInstance(schedulerDir));

			// register JobFactories for recovery
			scheduler.registerJobFactory(
					CBFBIRNConstants.CBF_PROCESSING_WORKFLOW,
					new CBFProcessingJobFactory());

			// start the job scheduler
			startThread(scheduler);

			// start download cache cleanup service
			if (GenUtils.isNotEmpty(schedulerDir)
					&& new File(schedulerDir).exists()) {
				// initialize cache policy
				double lowMarkPercent = GenUtils
						.toDouble(
								configService
										.getParamValue("download.cache.low.mark.percent"),
								25.0);
				DownloadCachePolicy.getInstance(schedulerDir)
						.setLowMarkPercent(lowMarkPercent);

				// for CBFBIRN download cache is never cleaned up (IBO
				// 08/2/2010)

				/*
				 * log.info("starting download cache cleanup service...");
				 * DownloadCacheCleanupService dccService = new
				 * DownloadCacheCleanupService( schedulerDir, configService,
				 * ss.getDefaultDBID()); startThread(dccService);
				 */
			}

			// start periodic services manager after registering periodic jobs

			PeriodicConfigUpdateService pcus = new PeriodicConfigUpdateService(
					configService);
			WFFileCacheCleanupService pwfcs = new WFFileCacheCleanupService(
					schedulerDir, ss.getDefaultDBID());

			PeriodicServiceManagementService psms = new PeriodicServiceManagementService();
			psms.register(pcus);
			psms.register(pwfcs);

			log.info("starting PeriodicServiceManagementService with "
					+ psms.getNumRegisteredServices()
					+ " registered services ...");
			startThread(psms);

			// initialize help service
			String helpMsgRepFile = servlet.getServletContext().getRealPath(
					Constants.HELP_MSG_REPOSITORY_FILE);
			log.info("**** initializing help message service using:"
					+ helpMsgRepFile);

			HelpService.getInstance(helpMsgRepFile);

			// cleanup orphan jobs
			IJobManagementService jobManService = ServiceFactory
					.getJobManagementService(defaultDBID);

			jobManService.removeOrphanJobs(new UserInfo(Constants.ADMIN_USER,
					null, null));

			System.out.println("Image reader format names");
			System.out.println("=========================");
			for (String s : ImageIO.getReaderFormatNames()) {
				System.out.println(s);
			}

		} catch (Throwable t) {
			t.printStackTrace();
			throw new ServletException(t);
		}
	}

	protected Result initializeDBPoolService(SecurityService secService,
			DBPoolService poolService, List<String> dbIDList,
			Map<String, String> dbID2SiteIDMap) throws DBPoolServiceException,
			Exception {
		Result result = new Result();
		result.poolService = poolService;
		Monitor primary = MonitorFactory.startPrimary("initDBPools");

		for (DBConfig dbConfig : secService.getDBConfigMap().values()) {
			log.info("initializing db " + dbConfig.getId());

			dbID2SiteIDMap.put(dbConfig.getId(), dbConfig.getSiteID());

			Monitor psMon = MonitorFactory.start("initDBPools.poolStartup");
			try {
				poolService = DBPoolService.getInstance(dbConfig.getId());
			} catch (RuntimeException re) {
				log.error(re);
				// skip badly/partially configured database connection pools
				continue;
			}

			poolService.startup();
			psMon.stop();

			Monitor ptcMon = MonitorFactory.start("initDBPools.prepTableCache");
			secService.prepareTableCache(dbConfig.getId());
			ptcMon.stop();

			// check if default database has the mandatory ADMIN user
			User u = dbConfig.getUser(Constants.ADMIN_USER);

			if (u != null && dbConfig.isDefaultDB()) {
				result.defaultDBID = dbConfig.getId();
				log.info("default db " + dbConfig.getId());
			} else {
				if (dbConfig.isDefaultDB() && u == null) {
					throw new Exception("No admin user for database "
							+ dbConfig.getId());
				}
			}

			dbIDList.add(dbConfig.getId());
		}
		primary.stop();
		return result;
	}

	public static class Result {
		DBPoolService poolService;
		String defaultDBID;
	}// ;

	protected void startThread(Runnable runner) {
		Thread thread = new Thread(runner);
		thread.setDaemon(true);
		thread.setPriority(Thread.NORM_PRIORITY - 1);
		thread.start();
	}

}
