package clinical.web.common;

import java.sql.Connection;

import clinical.web.common.security.User;
import clinical.web.exception.DBPoolServiceException;

/**
 * The interface to a named user database connection pool for maintaining named
 * user identity but providing connection pooling advantages.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: IDBPoolService.java 366 2011-05-05 20:06:27Z bozyurt $
 */

public interface IDBPoolService {

	/**
	 * Given a logged in username returns a pooled database connection.
	 * 
	 * @param user
	 *            user logged in through the UI
	 * 
	 * @throws DBPoolServiceException
	 *             if the if user is not assigned to a named database user or in
	 *             maximum pool size is reached or there is a problem connecting
	 *             to the database
	 * 
	 */
	public Connection getConnection(String user) throws DBPoolServiceException;

	/**
	 * Given a logged in username, returns a database connection back to the
	 * connection pool to be used by others.
	 * 
	 * @param user
	 *            user logged in through the UI
	 * @param con
	 *            the database connection to be returned to the database
	 * @throws DBPoolServiceException
	 *             if user is not assigned to a named database user or the
	 *             maximum pool size is reached or there is a problem connecting
	 *             to the database
	 */
	public void releaseConnection(String user, Connection con)
			throws DBPoolServiceException;

	/**
	 * Starts the named user database connection pool for maintaining named user
	 * identity but providing connection pooling advantages.
	 * 
	 * @throws DBPoolServiceException
	 */
	public void startup() throws DBPoolServiceException;

	/**
	 * Shuts down the connection pool service by closing all the database
	 * connections belonging to each named user's subpool.
	 * 
	 * @throws DBPoolServiceException
	 */
	public void shutdown() throws DBPoolServiceException;

	/**
	 * dynamically add a new user
	 * 
	 * @param user
	 */
	public void addUser(User user);

	/**
	 * dynamically remove user
	 * 
	 * @param webUser
	 */
	public void removeUser(User webUser);
}