package clinical.web.common.query;

import java.util.Date;

import clinical.utils.DateTimeUtils;
import clinical.web.ISQLDialect;

public abstract class AbstractQueryBuilder {

	public static enum CastingOption {
		CAST_TO_NUM, NONE
	}

	/**
	 * 
	 * @param value
	 *            the user supplied value for search predicate
	 * @param opType
	 *            the relational operator type. The possible values are, for
	 *            numbers <code>SearchPredicate.LESS</code>,
	 *            <code>SearchPredicate.EQUAL</code>,
	 *            <code>SearchPredicate.GREATER</code>,
	 *            <code>SearchPredicate.LESS_EQUAL</code>,
	 *            <code>SearchPredicate.GREATER_EQUAL</code>,
	 *            <code>SearchPredicate.NOT_EQUAL</code> and
	 *            <code>SearchPredicate.STARTS_WITH</code>
	 *            <code>SearchPredicate.END_WITH</code>,
	 *            <code>SearchPredicate.EQUAL</code> for strings.
	 * 
	 * @return the SQL snippet generated for this search predicate
	 */
	protected String createPredicate(String predicateString, Object value,
			int opType, int dataType, CastingOption castingOption)
			throws Exception {
		StringBuilder buf = new StringBuilder(64);
		boolean done = false;
		if (castingOption == CastingOption.CAST_TO_NUM) {
			if (dataType == SearchPredicate.INTEGER) {
				// FIXME only for Postgres
				buf.append("CAST(").append(predicateString)
						.append(" as integer)");
			} else if (dataType == SearchPredicate.FLOAT) {
				// FIXME only for Postgres
				buf.append("CAST(").append(predicateString)
						.append(" as float8)");
			} else {
				buf.append(predicateString);
			}
		} else {
			buf.append(predicateString);
		}
		if (value instanceof String && ((String) value).trim().equals("*")) {
			buf.append(" IS NOT NULL ");
			return buf.toString();
		}

		switch (opType) {
		case SearchPredicate.EQUAL:
			buf.append(" = ");
			if (value instanceof String) {
				buf.append("'").append((String) value).append("' ");
			} else if (value instanceof Number) {
				buf.append((Number) value).append(' ');
			} else if (value instanceof Boolean) {
				buf.append("'");
				buf.append(((Boolean) value).booleanValue() ? "yes" : "no");
				buf.append("' ");
			} else
				throw new Exception("Not an supported value type:"
						+ value.getClass().getName());
			done = true;
			break;
		case SearchPredicate.LESS:
			buf.append(" < ");
			break;
		case SearchPredicate.GREATER:
			buf.append(" > ");
			break;
		case SearchPredicate.LESS_EQUAL:
			buf.append(" <= ");
			break;
		case SearchPredicate.GREATER_EQUAL:
			buf.append(" >= ");
			break;
		case SearchPredicate.NOT_EQUAL:
			buf.append(" <> ");
			break;
		case SearchPredicate.STARTS_WITH:
			if (!(value instanceof String))
				throw new Exception(
						"Only varchar type can be used with Starts_With comparator!");
			buf.append(" LIKE '").append((String) value).append("%' ");
			done = true;
			break;
		case SearchPredicate.ENDS_WITH:
			if (!(value instanceof String))
				throw new Exception(
						"Only varchar type can be used with Ends_With comparator!");
			buf.append(" LIKE '%").append((String) value).append("' ");
			done = true;
			break;
		case SearchPredicate.ANY:
			if (!(value instanceof String))
				throw new Exception(
						"Only varchar type can be used with Any comparator!");
			String strValue = (String) value;
			String[] toks = strValue.split("\\s*,\\s*");
			buf.append(" IN(");
			for(int i = 0; i < toks.length; i++) {
				buf.append("'").append(toks[i]).append("'");
				if ((i+1) < toks.length) 
					buf.append(',');
			}
			buf.append(") ");
			done = true;
			break;
		}

		if (!done) {
			if (dataType == SearchPredicate.DATE) {
				String strValue = value.toString();
				if (value instanceof java.util.Date) {
					strValue = DateTimeUtils.formatDate((java.util.Date) value);
				}
				buf.append(" TO_TIMESTAMP('").append(strValue).append("',");
				buf.append("'MM/DD/YYYY') ");
			} else {
				if (!(value instanceof Number)) {
					throw new Exception(
							"Only numeric type can be used with comparison operators!");
				}
				buf.append((Number) value).append(' ');
			}

		}
		return buf.toString();
	}

	protected String toTimeStamp(Date value, ISQLDialect sqlDialect) {
		StringBuilder buf = new StringBuilder();
		String strValue = DateTimeUtils.formatDate((java.util.Date) value);
		buf.append(" to_timestamp(");
		buf.append(sqlDialect.prepareString(strValue)).append(",");
		buf.append(sqlDialect.prepareString("MM/DD/YYYY")).append(") ");
		return buf.toString();
	}
}
