package clinical.web.common.vo;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class Factor implements Serializable {
	private static final long serialVersionUID = 1L;
	private final String asName;
	private final String name;
	private final String key;
	private final Map<String, LevelCount> levelSet = new HashMap<String, LevelCount>(
			7);

	public Factor(String asName, String name) {
		super();
		this.asName = asName;
		this.name = name;
		if (asName == null) {
			// for Project factor
			this.key = ":" + name;
		} else {
			this.key = asName + ":" + name;
		}
	}

	public void addLevel(String level) {
		LevelCount lv = levelSet.get(level);
		if (lv == null) {
			lv = new LevelCount(level);
			levelSet.put(level, lv);
		}
		lv.incr();
	}

	public String getAsName() {
		return asName;
	}

	public String getName() {
		return name;
	}

	public List<LevelCount> getLevelSet() {
		return new ArrayList<Factor.LevelCount>(levelSet.values());
	}

	public String getKey() {
		return key;
	}

	public static String toFactorsArgString(List<Factor> factors) {
		StringBuilder sb = new StringBuilder();
		for (Iterator<Factor> it = factors.iterator(); it.hasNext();) {
			Factor f = it.next();
			if (f.getAsName() != null) {
				String asName = f.getAsName().replaceAll("\\s+", "_");
				sb.append(asName).append("__");
			}
			sb.append(f.getName().replaceAll("\\s+", "_"));
			if (it.hasNext())
				sb.append(';');
		}
		return sb.toString();
	}

	public static String toLevelsArgString(List<Factor> factors) {
		StringBuilder sb = new StringBuilder();
		for (Iterator<Factor> it = factors.iterator(); it.hasNext();) {
			Factor f = it.next();
			List<LevelCount> lcList = f.getLevelCounts();
			for (Iterator<LevelCount> it2 = lcList.iterator(); it2.hasNext();) {
				LevelCount lc = it2.next();
				sb.append(lc.getLevel());
				if (it2.hasNext()) {
					sb.append(',');
				}
			}
			if (it.hasNext())
				sb.append(';');
		}
		return sb.toString();
	}

	public JSONObject toJSON() throws JSONException {
		JSONObject js = new JSONObject();
		js.put("name", name);
		js.put("asName", asName == null ? "" : asName);
		JSONArray jsArr = new JSONArray();
		for (LevelCount lc : levelSet.values()) {
			jsArr.put(lc.toJSON());
		}
		js.put("levels", jsArr);
		return js;
	}

	public static Factor fromJSON(JSONObject js) throws JSONException {
		String name = js.getString("name");
		String asName = js.getString("asName");
		if (asName.length() == 0) {
			asName = null;
		}
		Factor f = new Factor(asName, name);
		JSONArray jsArr = js.getJSONArray("levels");
		for (int i = 0; i < jsArr.length(); i++) {
			LevelCount lc = LevelCount.fromJSON(jsArr.getJSONObject(i));
			f.levelSet.put(lc.getLevel(), lc);
		}
		return f;
	}

	public List<LevelCount> getLevelCounts() {
		List<LevelCount> lcList = new ArrayList<Factor.LevelCount>(
				levelSet.values());
		Collections.sort(lcList, new Comparator<LevelCount>() {
			@Override
			public int compare(LevelCount o1, LevelCount o2) {
				return o1.getLevel().compareTo(o2.getLevel());
			}
		});
		return lcList;
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("Factor::[asName=").append(asName);
		sb.append(",name=").append(name);
		sb.append("\n");
		for (LevelCount lc : levelSet.values()) {
			sb.append("\t").append(lc).append("\n");
		}
		sb.append("]");
		return sb.toString();
	}

	public static class LevelCount implements Serializable {
		private static final long serialVersionUID = 1L;
		private final String level;
		private int count = 0;

		public LevelCount(String level) {
			super();
			this.level = level;
		}

		private void incr() {
			count++;
		}

		public String getLevel() {
			return level;
		}

		public int getCount() {
			return count;
		}

		public JSONObject toJSON() throws JSONException {
			JSONObject js = new JSONObject();
			js.put("level", level);
			js.put("cnt", count);
			return js;
		}

		public static LevelCount fromJSON(JSONObject js) throws JSONException {
			LevelCount lc = new LevelCount(js.getString("level"));
			lc.count = js.getInt("cnt");
			return lc;
		}

		@Override
		public String toString() {
			return "LevelCount [level=" + level + ", count=" + count + "]";
		}
	}// ;

}
