package clinical.web.dd;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jdom.Element;

import clinical.server.dao.MeasurementunitDAO;
import clinical.server.vo.Measurementunit;
import clinical.server.vo.Transformation;
import clinical.utils.Assertion;
import clinical.utils.FileUtils;
import clinical.utils.XCEDEUtils;
import clinical.web.ConnectionSupportMixin;
import clinical.web.DAOFactory;
import clinical.web.IProvenanceService;
import clinical.web.IXCEDEDerivedDataService;
import clinical.web.MinimalServiceFactory;
import clinical.web.ServiceFactory;
import clinical.web.services.XCEDEDerivedDataServiceImpl.MeasurementInfo;
import clinical.web.vo.AnalysisCompInfo;
import clinical.web.vo.AnalysisInfo;
import clinical.web.vo.TransformationInfo;
import clinical.xml.xcede2.AnalysisT;
import clinical.xml.xcede2.MeasurementGroupT;
import clinical.xml.xcede2.ObservationT;
import clinical.xml.xcede2.ProcessStepT;
import clinical.xml.xcede2.ProvenanceT;
import clinical.xml.xcede2.XCEDE;

/**
 * @author I. Burak Ozyurt
 * @version $Id: XCEDE2AnalysisMetaDataTool.java,v 1.2 2008/02/16 02:09:50
 *          bozyurt Exp $
 */
public class XCEDE2AnalysisMetaDataTool {
	protected ConnectionSupportMixin csm;

	public XCEDE2AnalysisMetaDataTool(String propsFile) throws Exception {
		csm = new ConnectionSupportMixin(propsFile, true);
		MinimalServiceFactory.setMimimalOpMode(true);
		ServiceFactory.setMimimalOpMode(true);
		csm.startup();
	}

	public XCEDE2AnalysisMetaDataTool(File usersXMLFile) throws Exception {
		csm = new ConnectionSupportMixin(usersXMLFile.getName());
		MinimalServiceFactory.setMimimalOpMode(true);
		ServiceFactory.setMimimalOpMode(true);
		csm.startup();
	}

	public void shutdown() {
		if (csm != null)
			try {
				csm.shutdown();
			} catch (Exception e) {
				e.printStackTrace();
			}
	}

	public List<AnalysisT> loadMergedFSXCEDE(String mergedXcedeFile)
			throws Exception {
		XCEDE xcede = XCEDEUtils.unmarshal(mergedXcedeFile);
		List<Object> aList = xcede.getAnnotationListOrRevisionListOrProject();
		List<AnalysisT> analysisTypes = new ArrayList<AnalysisT>(5);
		if (!aList.isEmpty() && aList.get(0) instanceof AnalysisT) {
			for (Iterator<?> it = aList.iterator(); it.hasNext();) {
				AnalysisT a = (AnalysisT) it.next();
				analysisTypes.add(a);
			}
		}
		return analysisTypes;
	}

	public void deleteMetaData(List<AnalysisT> analysisTypeList)
			throws Exception {
		IXCEDEDerivedDataService xdds = ServiceFactory
				.getXCEDEDerivedDataService(csm.getDbID());
		IProvenanceService ps = ServiceFactory.getProvenanceService(csm
				.getDbID());
		AnalysisT prAt = findProvenanceAnalysis(analysisTypeList);
		String analysisTypeName = prAt.getType();
		String version = "0001_0001";
		AnalysisInfo ai = new AnalysisInfo(analysisTypeName, version);
		System.out.println("analysisTypeName:" + analysisTypeName);
		
		System.out.println("deleting analysis " + ai );
		ps.deleteAnalysis(csm.getUi(), ai);
		for (AnalysisT at : analysisTypeList) {
			if (!at.getMeasurementGroup().isEmpty()) {
				String tupleName = at.getType();
				System.out.println("deleting extended tuple: " + tupleName);
				xdds.deleteExtendedTuple(csm.getUi(), tupleName);
			}
		}
	}

	public void importMetaData(List<AnalysisT> analysisTypeList)
			throws Exception {
		IXCEDEDerivedDataService xdds = ServiceFactory
				.getXCEDEDerivedDataService(csm.getDbID());

		for (AnalysisT at : analysisTypeList) {
			if (!at.getMeasurementGroup().isEmpty()) {
				MeasurementGroupT mg = at.getMeasurementGroup().get(0);
				// use analysis type as the extended tuple name
				String tupleName = at.getType();
				String tupleClass = "nc_researchdata";
				String tupleSubClass = "nc_analysisresult";

				xdds.addExtendedTuple(csm.getUi(), tupleName, mg, tupleClass,
						tupleSubClass);

			}
		}
	}

	protected Map<String, Measurementunit> getMeasurementUnitsFromDB()
			throws Exception {
		MeasurementunitDAO muDAO = DAOFactory.createMeasurementunitDAO(csm
				.getDbID());

		List<Measurementunit> muList = muDAO.find(csm.getConnection(),
				new Measurementunit());
		Map<String, Measurementunit> muMap = new HashMap<String, Measurementunit>(
				11);
		for (Measurementunit mu : muList) {
			muMap.put(mu.getUnit(), mu);
		}
		return muMap;
	}

	protected void getUnitsPresent(List<AnalysisT> analysisTypeList)
			throws Exception {
		Map<String, Measurementunit> muMap = getMeasurementUnitsFromDB();

		Set<String> unitSet = new HashSet<String>(17);
		for (AnalysisT at : analysisTypeList) {
			if (!at.getMeasurementGroup().isEmpty()) {
				for (MeasurementGroupT mg : at.getMeasurementGroup()) {
					for (Iterator<ObservationT> it = mg.getObservation()
							.iterator(); it.hasNext();) {
						ObservationT ot = it.next();
						if (ot.getUnits() != null) {
							if (!muMap.containsKey(ot.getUnits().trim())) {
								unitSet.add(ot.getUnits().trim());
							}
						}
					}
				}
			}
		}
		for (String unit : unitSet) {
			System.out.println(unit);
		}
	}

	public void updateMeasurementUnitsInDB(String analysisMetaDataXmlFile)
			throws Exception {
		Map<String, Measurementunit> muMap = getMeasurementUnitsFromDB();
		IXCEDEDerivedDataService xdds = ServiceFactory
				.getXCEDEDerivedDataService(csm.getDbID());

		Element rootEl = FileUtils.loadXML(analysisMetaDataXmlFile);
		List<MeasurementInfo> miList = new ArrayList<MeasurementInfo>(10);
		List<?> unitEls = rootEl.getChild("measurement-units").getChildren(
				"unit");
		for (Iterator<?> it = unitEls.iterator(); it.hasNext();) {
			Element unitEl = (Element) it.next();
			String unit = unitEl.getAttributeValue("name");
			if (muMap.get(unit) != null) {
				continue;
			}
			String ms = unitEl.getAttributeValue("measurementSystem");
			String mt = unitEl.getAttributeValue("measurementType");
			MeasurementInfo mi = new MeasurementInfo(unit, ms, mt);
			miList.add(mi);
		}

		for (MeasurementInfo mi : miList) {
			xdds.addMeasurementInfo(csm.getUi(), mi);
		}
	}

	public AnalysisT findProvenanceAnalysis(List<AnalysisT> analysisTypeList) {
		for (AnalysisT a : analysisTypeList) {
			if (!a.getInput().isEmpty() && !a.getProvenance().isEmpty()) {
				return a;
			}
		}
		return null;
	}

	public void addAnalyis(AnalysisT analysis) throws Exception {
		ProvenanceT pt = analysis.getProvenance().get(0);
		String analysisTypeName = analysis.getType();
		String version = "0001_0001";
		AnalysisInfo ai = new AnalysisInfo(analysisTypeName, version);
		List<TransformationInfo> tiList = new ArrayList<TransformationInfo>(10);
		Map<String, TransformationInfo> uniqTIMap = new HashMap<String, TransformationInfo>(
				7);
		Map<ProcessStepT, TransformationInfo> process2TIMap = new HashMap<ProcessStepT, TransformationInfo>(
				17);
		for (ProcessStepT pst : pt.getProcessStep()) {
			String packageName = "NO-PACKAGE";
			String packageVersion = "NO-PACKAGE-VERSION";
			if (pst.getPackage() != null) {
				packageName = pst.getPackage().getValue();
				if (pst.getPackage().getVersion() != null)
					packageVersion = pst.getPackage().getVersion();
			}
			String program = pst.getProgram().getValue();
			String programVersion = pst.getProgram().getVersion();
			if (programVersion == null) {
				programVersion = pst.getCvs();
			}
			if (programVersion == null) {
				programVersion = "NO-PROGRAM-VERSION";
			}
			TransformationInfo ti = new TransformationInfo(packageName,
					packageVersion, program, programVersion);
			String key = prepKey(ti);
			if (!uniqTIMap.containsKey(key)) {
				tiList.add(ti);
				uniqTIMap.put(key, ti);
			}
			TransformationInfo ti2 = uniqTIMap.get(key);
			process2TIMap.put(pst, ti2);
		}
		IProvenanceService ps = ServiceFactory.getProvenanceService(csm
				.getDbID());
		// add transformations
		System.out.println("Transformations to add\n--------------------");
		List<Transformation> tList = ps.addTransformations(csm.getUi(), tiList);
		for (Transformation t : tList) {
			System.out.println(t);
		}

		int id = 1;
		List<AnalysisCompInfo> flowList = new ArrayList<AnalysisCompInfo>(pt
				.getProcessStep().size());

		for (ProcessStepT pst : pt.getProcessStep()) {
			TransformationInfo ti = process2TIMap.get(pst);
			AnalysisCompInfo aci = createACI(id, 1, ai, ti, null);
			ai.addComponent(aci);
			flowList.add(aci);
			id++;
		}
		ps.addAnalysisRecWithComponents(csm.getUi(), ai, flowList);

	}

	private String prepKey(TransformationInfo ti) {
		StringBuffer buf = new StringBuffer();
		buf.append(ti.getPackage()).append('_');
		buf.append(ti.getPackageVersion()).append('_');
		buf.append(ti.getTransform()).append('_');
		buf.append(ti.getTransformVersion());
		return buf.toString();
	}

	private AnalysisCompInfo createACI(int id, int nodeLevel, AnalysisInfo ai,
			TransformationInfo ti, AnalysisCompInfo parent) {
		AnalysisCompInfo aci = new AnalysisCompInfo(id);
		aci.setTransform(ti);
		aci.setAnalysis(ai);
		aci.setNodeLevel(nodeLevel);
		aci.setParent(parent);
		return aci;
	}

	public static void importMetaData(String usersXMLFile, String xcedeDir,
			String metaDataFile) throws Exception {
		XCEDE2AnalysisMerger merger = new XCEDE2AnalysisMerger();
		List<File> xcedeFiles = merger.findXCEDEFiles(new File(xcedeDir));
		File mergedFile = new File(xcedeDir, "merged.xml");
		XCEDE xcede = merger.merge(xcedeFiles);

		merger.saveMerged(xcede, mergedFile);
		xcede = null;
		XCEDE2AnalysisMetaDataTool tool = null;
		try {
			tool = new XCEDE2AnalysisMetaDataTool(new File(usersXMLFile));
			List<AnalysisT> atList = tool.loadMergedFSXCEDE(mergedFile
					.getAbsolutePath());
			tool.updateMeasurementUnitsInDB(metaDataFile);
			tool.importMetaData(atList);
			AnalysisT at = tool.findProvenanceAnalysis(atList);
			Assertion.assertNotNull(at);
			tool.addAnalyis(at);
			System.out.println("finished.");
		} finally {
			if (tool != null)
				tool.shutdown();
		}
	}

	public static void deleteMetaData(String usersXMLFile, String xcedeDir)
			throws Exception {
		XCEDE2AnalysisMerger merger = new XCEDE2AnalysisMerger();
		List<File> xcedeFiles = merger.findXCEDEFiles(new File(xcedeDir));
		File mergedFile = new File(xcedeDir, "merged.xml");
		XCEDE xcede = merger.merge(xcedeFiles);

		merger.saveMerged(xcede, mergedFile);
		xcede = null;
		XCEDE2AnalysisMetaDataTool tool = null;
		try {
			tool = new XCEDE2AnalysisMetaDataTool(new File(usersXMLFile));
			List<AnalysisT> atList = tool.loadMergedFSXCEDE(mergedFile
					.getAbsolutePath());

			tool.deleteMetaData(atList);

			System.out.println("finished.");
		} finally {
			if (tool != null)
				tool.shutdown();
		}
	}

	@SuppressWarnings("unused")
	public static void testDriver() throws Exception {
		String rootDir = "/home/bozyurt/work/fsanalysis/fBIRNPhaseII__0010/Data/001029291693/scanVisit__0010__0002/MRI__0001/t1/Analysis/FreeSurfer__0010__0001";
		String metaDataFile = "/home/bozyurt/dev/java/BIRN/clinical/conf/analysis-metadata.xml";
		XCEDE2AnalysisMetaDataTool tool = null;
		try {
			tool = new XCEDE2AnalysisMetaDataTool("test.properties");
			String mergedXcedeFile = rootDir + "/merged.xml";
			List<AnalysisT> atList = tool.loadMergedFSXCEDE(mergedXcedeFile);
			// tool.getUnitsPresent(atList);
			// tool.updateMeasurementUnitsInDB(metaDataFile);

			// tool.importMetaData(atList);

			AnalysisT at = tool.findProvenanceAnalysis(atList);
			Assertion.assertNotNull(at);
			tool.addAnalyis(at);
		} finally {
			if (tool != null)
				tool.shutdown();
		}
	}

	public static void usage() {
		System.err
				.println("Usage: XCEDE2AnalysisMetaDataTool <usersXMLFile> <xcedeDir> <metaDataFile>");
		System.err.println("\tor");
		System.err
				.println("XCEDE2AnalysisMetaDataTool -delete <usersXMLFile> <xcedeDir>");
		System.exit(-1);
	}

	public static void main(String[] args) throws Exception {
		if (args.length != 3) {
			usage();
		}
		String usersXMLFile = null;
		String xcedeDir = null;
		String metaDataFile = null;

		if (args[0].equals("-delete")) {
			usersXMLFile = args[1];
			xcedeDir = args[2];
			System.out.println("usersXMLFile=" + usersXMLFile + "\nxcedeDir="
					+ xcedeDir);
			deleteMetaData(usersXMLFile, xcedeDir);
			
		} else {
			usersXMLFile = args[0];
			xcedeDir = args[1];
			metaDataFile = args[2];
			System.out.println("usersXMLFile=" + usersXMLFile + "\nxcedeDir="
					+ xcedeDir + "\nmetaDataFile=" + metaDataFile);
			importMetaData(usersXMLFile, xcedeDir, metaDataFile);
		}		
	}

}
