package clinical.web.download;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.storage.IPlugin;
import clinical.storage.PluginManager;
import clinical.storage.StorageUtils;
import clinical.utils.BIRNURIException;
import clinical.utils.FileUtils;
import clinical.utils.JargonUtils;
import clinical.utils.SRBFileSystemAdapter;
import clinical.web.IAppConfigService;
import clinical.web.IAssessmentService;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.common.vo.AssessmentSelectionInfo;
import clinical.web.helpers.BatchQueryHelper;
import clinical.web.helpers.EmailHelper;
import clinical.web.scheduler.IJobFactory;
import clinical.web.scheduler.JobException;
import clinical.web.scheduler.remote.WFESLocation;
import clinical.web.services.BatchQueryResult;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: DownloadJob.java 384 2011-06-23 00:15:36Z bozyurt $
 */
public class DownloadJob extends AbstractJob {
	protected FileBundleConfig config;
	protected List<SubjectImageDataBundle> caSIDBList;
	protected List<AssessmentSelectionInfo> filteredAsiList;
	protected File assessmentsFile;
	protected EmailHelper emailHelper;
	protected String[] tarBallNames;
	List<FileBundleConfig> fbcList;

	public final static int SUCCESS = 1;
	public final static int CANCELED = 2;
	/** maximum unzipped size for a tar ball */
	public final static long MAX_TARBALL_SIZE = 1000000000l; // for test 1Mb

	protected boolean testMode = true;

	private Log log = LogFactory.getLog(DownloadJob.class);
	protected String status;

	public DownloadJob(String id, UserInfo ui, String description,
			FileBundleConfig config) {
		this(id, ui, description, config, null, null, null);
	}

	public DownloadJob(String id, UserInfo ui, String description,
			FileBundleConfig config, String dbID,
			List<SubjectImageDataBundle> caSIDBList,
			List<AssessmentSelectionInfo> filteredAsiList) {
		super(id, ui, description, dbID);
		this.config = config;
		this.caSIDBList = caSIDBList;
		this.filteredAsiList = filteredAsiList;

		this.fbcList = config.split(MAX_TARBALL_SIZE);

		if (fbcList.size() > 1) {
			this.description += " (" + fbcList.size() + " tarballs)";
		}
		this.tarBallNames = new String[fbcList.size()];
		int idx = 0;
		for (FileBundleConfig fbc : fbcList) {
			tarBallNames[idx++] = fbc.getBundleName();
		}
	}

	public String getType() {
		return "Image Download";
	}

	protected void downloadAssessments() throws Exception {
		if (dbID == null || caSIDBList == null || filteredAsiList == null)
			return;

		IAssessmentService ias = ServiceFactory.getAssessmentService(dbID);

		BatchQueryResult bqr = ias.getFullAssessments(ui, caSIDBList,
				filteredAsiList);

		String prefix = config.getBundleName();
		prefix = prefix.replaceAll("[\\s,\\?!:;]+", "_");
		File file = new File(config.getCacheDir(), prefix + "_assessments.csv");
		BatchQueryHelper.prepareCSVFile(file, filteredAsiList, bqr);
		log.info("wrote assessments to file:" + file);

		this.assessmentsFile = file;
	}

	public void execute(int stageId) throws JobException {

		try {
			downloadAssessments();
			if (isCanceled()) {
				log.info("Canceling job " + getID());
				cleanupCache(getID(), log);
				return;
			}
		} catch (Exception ex) {
			throw new JobException(ex);
		}
		try {
			Map<String, Object> paramMap = new HashMap<String, Object>();
			PluginManager pm = PluginManager.getInstance();

			boolean first = true;
			for (FileBundleConfig fbc : fbcList) {
				paramMap.put("cacheDir", fbc.getCacheDir());

				List<PathWrapper> allFiles = new ArrayList<PathWrapper>();
				if (fbc.hasSRBFiles()) {
					IPlugin plugin = pm.getPlugin(FileInfo.SRB_STR, paramMap);
					try {
						plugin.startup();
						int rc = loadFromStorage(fbc, allFiles, plugin, FileInfo.SRB);
						if (rc == CANCELED) {
							return;
						}
					} finally {
						plugin.shutdown();
					}
				}
				if (fbc.hasLocalFiles()) {
					IPlugin plugin = pm.getPlugin(FileInfo.LOCAL_STR, paramMap);
					try {
						plugin.startup();
						int rc = loadFromStorage(fbc, allFiles, plugin,
								FileInfo.LOCAL);
						if (rc == CANCELED) {
							return;
						}
					} finally {
						plugin.shutdown();
					}
				}
				if (fbc.hasGridFtpFiles()) {
					IAppConfigService acs = ServiceFactory.getAppConfigService();
					StorageUtils.prepareGridFtpPluginParams(acs, paramMap, ui
							.getCredential());

					IPlugin plugin = pm.getPlugin(FileInfo.GRIDFTP_STR, paramMap);
					try {
						plugin.startup();
						int rc = loadFromStorage(fbc, allFiles, plugin,
								FileInfo.GRIDFTP);
						if (rc == CANCELED) {
							return;
						}
					} finally {
						plugin.shutdown();
					}
				}

				// do packaging

				Packager p = new Packager(fbc.getCacheDir(), fbc.getBundleName(),
						config.getAction());

				p.includeFiles(fbc.getCacheDir(), allFiles);
				if (isCanceled()) {
					log.info("Canceling job " + getID());
					cleanupCache(getID(), log);
					return;
				}
				if (first && assessmentsFile != null) {
					addAssessmentFile(p);
					first = false;
				}
				p.setDoCleanup(true);
				p.pack();

			}// for
			handleEmail(".\nYou can download it at your convenience by login to the HID web application.");

		} catch (Throwable t) {
			t.printStackTrace();
			throw new JobException(t);
		}
	}

	protected int loadFromStorage(FileBundleConfig fbc,
			List<PathWrapper> allFiles, IPlugin plugin, int storageType)
			throws Exception {
		List<PathWrapper> loadedFiles = new ArrayList<PathWrapper>(fbc
				.getFileInfoList().size());
		for (FileInfo fi : fbc.getFileInfoList()) {
			if (fi.getStorageType() == storageType) {
				int rc = loadFromStorage(plugin, fi, loadedFiles);
				if (rc == CANCELED) {
					return rc;
				}
				allFiles.addAll(loadedFiles);
			}
		}
		return SUCCESS;
	}

	public void executeOld() throws JobException {

		try {
			downloadAssessments();
			if (isCanceled()) {
				log.info("Canceling job " + getID());
				cleanupCache(getID(), log);
				return;
			}
		} catch (Exception ex) {
			throw new JobException(ex);
		}

		try {
			boolean first = true;
			for (FileBundleConfig fbc : fbcList) {
				List<PathWrapper> allFiles = new ArrayList<PathWrapper>();
				if (fbc.hasSRBFiles()) {
					SRBFileSystemAdapter sfsAdapter = null;
					try {
						sfsAdapter = new SRBFileSystemAdapter();

						List<PathWrapper> srbFiles = new ArrayList<PathWrapper>(fbc
								.getFileInfoList().size());
						for (FileInfo fi : fbc.getFileInfoList()) {
							if (fi.getStorageType() == FileInfo.SRB) {
								String srbPath = fi.getPath();

								int rc = loadFromSRB2(sfsAdapter, srbFiles, fi, srbPath);
								if (rc == CANCELED) {
									return;
								}
								allFiles.addAll(srbFiles);
							}
						}
					} finally {
						if (sfsAdapter != null)
							sfsAdapter.shutdown();
					}
				}

				if (fbc.hasLocalFiles()) {
					int rc = handleLocal(fbc, allFiles);
					if (rc == CANCELED) {
						return;
					}
				}

				// do packaging

				Packager p = new Packager(fbc.getCacheDir(), fbc.getBundleName(),
						config.getAction());

				p.includeFiles(fbc.getCacheDir(), allFiles);
				if (isCanceled()) {
					log.info("Canceling job " + getID());
					cleanupCache(getID(), log);
					return;
				}
				if (first && assessmentsFile != null) {
					addAssessmentFile(p);
					first = false;
				}
				p.setDoCleanup(true);
				p.pack();

			}// for
			handleEmail(".\nYou can download it at your convenience by login to the HID web application.");

		} catch (Throwable t) {
			t.printStackTrace();
			throw new JobException(t);
		}
	}

	protected int loadFromStorage(IPlugin storagePlugin, FileInfo fi,
			List<PathWrapper> pwList) throws Exception {
		if (isCanceled()) {
			log.info("Canceling job " + getID());
			cleanupCache(getID(), log);
			return CANCELED;
		}
		List<PathWrapper> list = storagePlugin.load(fi);
		pwList.addAll(list);
		if (isCanceled()) {
			log.info("Canceling job " + getID());
			cleanupCache(getID(), log);
			return CANCELED;
		}
		return SUCCESS;
	}

	protected int loadFromSRB2(SRBFileSystemAdapter sfsAdapter,
			List<PathWrapper> srbFiles, FileInfo fi, String srbPath)
			throws Exception {
		if (fi.isDir) {
			File dir = FileInfo.getFullPathFile(fi, fi.getSourceRootDir());
			String dirPath = dir.getAbsolutePath();

			if (isCanceled()) {
				log.info("Canceling job " + getID());
				cleanupCache(getID(), log);
				return CANCELED;
			}

			FileInfo nfi = new FileInfo(dirPath, fi.getFileType(), fi
					.getStorageType(), fi.getSourceRootDir(), false);

			String relDirPath = nfi.getRelativePath();
			String localPath = FileUtils.createFullPath(config.getCacheDir(),
					relDirPath);

			System.out.println("**** from SRB '" + srbPath + "' \nto '"
					+ localPath + "'");

			JargonUtils.copyFromSRB(dir.getAbsolutePath(), localPath, sfsAdapter);

			File localDir = new File(FileUtils.createFullPath(
					config.getCacheDir(), relDirPath));
			File[] localFilePaths = localDir.listFiles();
			for (int i = 0; i < localFilePaths.length; i++) {
				if (localFilePaths[i].isFile()) {
					String filePath = localFilePaths[i].getAbsolutePath();
					if (filePath.startsWith(config.getCacheDir())) {
						filePath = filePath.substring(config.getCacheDir().length());
						filePath = filePath.replaceFirst("^\\/+", "");
					}
					srbFiles.add(new PathWrapper(filePath, nfi.getSourceRootDir()));
				} else {
					StorageUtils.recurseDir(srbFiles, localFilePaths[i], config
							.getCacheDir(), nfi.getSourceRootDir());
					// throw new Exception("Unexpected dir: " + localFilePaths[i]);
				}
			}

		} else {
			if (isCanceled()) {
				log.info("Canceling job " + getID());
				cleanupCache(getID(), log);
				return CANCELED;
			}
			String relPath = fi.getRelativePath();
			if (FileUtils.buildLocalPath(config.getCacheDir(), relPath)) {
				String localPath = FileUtils.createFullPath(config.getCacheDir(),
						relPath);
				JargonUtils.copyFromSRB(srbPath, localPath, sfsAdapter);
				srbFiles.add(new PathWrapper(relPath, fi.getSourceRootDir()));
			}
		}
		return SUCCESS;
	}

	protected int handleLocal(FileBundleConfig fbc, List<PathWrapper> files)
			throws BIRNURIException {
		String rootDir = FileInfo.findCommonPath(fbc.getFileInfoList());

		for (FileInfo fi : config.getFileInfoList()) {
			if (fi.isDir) {
				File dir = FileInfo.getFullPathFile(fi, fi.getSourceRootDir());
				List<File> fileList = FileUtils.getAllFilesUnderDir(dir
						.getAbsolutePath());
				for (File file : fileList) {
					if (isCanceled()) {
						log.info("Canceling job " + getID());
						return CANCELED;
					}
					FileInfo nfi = new FileInfo(file.getAbsolutePath(), fi
							.getFileType(), fi.getStorageType(), rootDir, false);

					files.add(new PathWrapper(nfi.getRelativePath(), rootDir));
				}

			} else {
				FileInfo nfi = new FileInfo(fi.getPath(), fi.getFileType(), fi
						.getStorageType(), rootDir, false);
				files.add(new PathWrapper(nfi.getRelativePath(), rootDir));
			}
		}
		return SUCCESS;
	}

	protected void handleLocal() throws BIRNURIException {
		String rootDir = FileInfo.findCommonPath(config.getFileInfoList());

		List<PathWrapper> files = new ArrayList<PathWrapper>(config
				.getFileInfoList().size());
		for (FileInfo fi : config.getFileInfoList()) {
			if (fi.isDir) {
				File dir = FileInfo.getFullPathFile(fi, fi.getSourceRootDir());
				List<File> fileList = FileUtils.getAllFilesUnderDir(dir
						.getAbsolutePath());
				for (File file : fileList) {
					if (isCanceled()) {
						log.info("Canceling job " + getID());
						return;
					}
					FileInfo nfi = new FileInfo(file.getAbsolutePath(), fi
							.getFileType(), fi.getStorageType(), rootDir, false);

					files.add(new PathWrapper(nfi.getRelativePath(), rootDir));
				}

			} else {
				FileInfo nfi = new FileInfo(fi.getPath(), fi.getFileType(), fi
						.getStorageType(), rootDir, false);
				files.add(new PathWrapper(nfi.getRelativePath(), rootDir));
			}
		}

		Packager packager = new Packager(config.getCacheDir(), config
				.getBundleName(), config.getAction());
		packager.includeFiles(files);

		if (assessmentsFile != null) {
			addAssessmentFile(packager);
		}

		packager.setDoCleanup(false);
		if (isCanceled()) {
			log.info("Canceling job " + getID());
			cleanupCache(getID(), log);
			return;
		}
		packager.pack();
		if (isCanceled()) {
			log.info("Canceling job " + getID());
			cleanupCache(getID(), log);
			return;
		}
		handleEmail(".\nYou can download it at your convenience by login to the HID web application.");
	}

	protected void handleEmail(String status) {
		if (emailHelper == null)
			return;
		Map<String, String> paramMap = new HashMap<String, String>(7);
		String client = emailHelper.getToUser();
		int idx = client.indexOf('@');
		if (idx != -1) {
			client = client.substring(0, idx);
		}

		paramMap.put("subject",
				"your image download bundle is ready for download!");
		paramMap.put("client", client);
		paramMap.put("job_type", "image download job");
		paramMap.put("jobid", getID());
		paramMap.put("status", status);
		try {
			emailHelper.sendEmail(paramMap);
		} catch (Throwable t) {
			log.error("DownloadJob", t);
		}
	}

	protected void addAssessmentFile(Packager packager) {
		List<PathWrapper> fileList = new ArrayList<PathWrapper>(1);
		fileList.add(new PathWrapper(assessmentsFile.getName(), null));
		log.info("adding assessment file:" + assessmentsFile.getName() + " - "
				+ assessmentsFile);
		packager.includeFiles(config.getCacheDir(), fileList);
	}

	public void saveResults(File file) throws JobException {
	// no op not needed for this kind of job
	}

	public String[] getResultsFiles() {
		// String filename = config.getBundleName();
		String[] rfArr = new String[tarBallNames.length];
		for (int i = 0; i < tarBallNames.length; i++) {
			String filename = tarBallNames[i];
			if (!filename.endsWith(".tar.gz")) {
				filename += ".tar.gz";
			}
			String path = FileUtils.createFullPath(config.getCacheDir(), filename);
			rfArr[i] = path;
		}
		return rfArr;
	}

	public void setCaSIDBList(List<SubjectImageDataBundle> caSIDBList) {
		this.caSIDBList = caSIDBList;
	}

	public void setDbID(String dbID) {
		this.dbID = dbID;
	}

	public void setFilteredAsiList(List<AssessmentSelectionInfo> filteredAsiList) {
		this.filteredAsiList = filteredAsiList;
	}

	public EmailHelper getEmailHelper() {
		return emailHelper;
	}

	public void setEmailHelper(EmailHelper emailHelper) {
		this.emailHelper = emailHelper;
	}

	@Override
	public int getDurationType() {
		return LONG;
	}

	@Override
	public IJobFactory getJobFactory() {
		throw new UnsupportedOperationException();
	}

	@Override
	public synchronized String getStatus() {
		return status;
	}

	@Override
	public synchronized void setStatus(String status) {
		this.status = status;
	}
	
	@Override
	public void cleanup() {
		// no op
	}

	@Override
	public boolean setWFESLocation(WFESLocation location) {
		return false;
	}
}
