package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.vo.Databaseuser;
import clinical.server.vo.Tableid;
import clinical.web.ISQLDialect;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.IDBPoolService;
import clinical.web.common.ISecurityService;
import clinical.web.common.UserInfo;
import clinical.web.common.security.User;
import clinical.web.exception.BaseException;
import clinical.web.exception.DBPoolServiceException;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: AbstractServiceImpl.java 492 2011-12-06 21:50:33Z bozyurt $
 */
public abstract class AbstractServiceImpl {
	protected String theDBID;
	protected ISQLDialect sqlDialect;
	protected IDBPoolService pool = null;
	protected Log log = LogFactory.getLog(AbstractServiceImpl.class);

	public AbstractServiceImpl(String dbID) throws BaseException {
		pool = ServiceFactory.getPoolService(dbID);
		theDBID = dbID;
		sqlDialect = ServiceFactory.getSQLDialect(theDBID);
	}

	public Connection getConnection(UserInfo ui) throws DBPoolServiceException {
	   return pool.getConnection(ui.getName());
	}
	
	protected void releaseConnection(Connection con, UserInfo ui) {
		if (con == null)
			return;
		try {
			pool.releaseConnection(ui.getName(), con);
		} catch (DBPoolServiceException x) {
			log.error("Cannot release connection for user " + ui.getName(), x);
		}
	}

	protected void handleErrorAndRollBack(Connection con, String msg,
			Exception x, boolean doRollback) throws BaseException {
		if (con != null) {
			try {
				con.rollback();
			} catch (SQLException se) {
				log.error("", se);
				throw new BaseException(se);
			}
		}
		log.error(msg, x);
		if (x instanceof BaseException)
			throw (BaseException) x;
		else
			throw new BaseException(x);
	}
	
	
	protected void handleErrorAndRollBack(Connection con, String msg,
         Throwable t, boolean doRollback) throws BaseException {
      if (con != null) {
         try {
            con.rollback();
         } catch (SQLException se) {
            log.error("", se);
            throw new BaseException(se);
         }
      }
      log.error(msg, t);
      if (t instanceof BaseException)
         throw (BaseException) t;
      else
         throw new BaseException(t);
   }

	protected BigDecimal getTableID(UserInfo ui, String tableName)
			throws Exception {
		IDBCache dbCache = ServiceFactory.getDBCache(theDBID);
		Tableid tid = dbCache.getTableID(ui, tableName);
		return tid.getTableid();
	}

	protected Databaseuser getDatabaseUser(UserInfo ui, String userClass)
			throws Exception {
		ISecurityService securityService = ServiceFactory.getSecurityService();
		IDBCache dbCache = ServiceFactory.getDBCache(theDBID);

		Map<String, User> userMap = securityService.getAllUsers(theDBID);
		User u = (User) userMap.get(ui.getName());

		Databaseuser databaseUser = dbCache.getDatabaseUser(ui, u.getDbUser()
				.getName(), userClass);
		if (databaseUser == null) {
			log.info("databaseUser was null (from dbCache)");
		}
		return databaseUser;
	}
}
