package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.dao.AssessmentDAO;
import clinical.server.dao.AssessmentbooleanDAO;
import clinical.server.dao.AssessmentfloatDAO;
import clinical.server.dao.AssessmentinformantDAO;
import clinical.server.dao.AssessmentintegerDAO;
import clinical.server.dao.AssessmentitemDAO;
import clinical.server.dao.AssessmentscoreDAO;
import clinical.server.dao.AssessmenttimestampDAO;
import clinical.server.dao.AssessmentvarcharDAO;
import clinical.server.dao.ExpsegmentDAO;
import clinical.server.dao.StoredassessmentDAO;
import clinical.server.vo.Assessment;
import clinical.server.vo.Assessmentboolean;
import clinical.server.vo.Assessmentfloat;
import clinical.server.vo.Assessmentinformant;
import clinical.server.vo.Assessmentinteger;
import clinical.server.vo.Assessmentitem;
import clinical.server.vo.Assessmentscore;
import clinical.server.vo.Assessmenttimestamp;
import clinical.server.vo.Assessmentvarchar;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Expsegment;
import clinical.server.vo.Securityclassification;
import clinical.server.vo.Storedassessment;
import clinical.server.vo.Tableid;
import clinical.utils.Assertion;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.DBUtils;
import clinical.web.ISequenceHelper;
import clinical.web.ISubjectAssessmentManagement;
import clinical.web.MinimalServiceFactory;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.ISecurityService;
import clinical.web.common.UserInfo;
import clinical.web.common.security.User;
import clinical.web.exception.BaseException;
import clinical.web.exception.DBPoolServiceException;
import clinical.web.exception.SubjectAssessmentManagementException;
import clinical.web.helpers.MissingEntry;
import clinical.web.vo.AssessmentInfo;
import clinical.web.vo.AssessmentInfo.ScoreInfo;
import clinical.web.vo.AssessmentScoreValues;
import clinical.web.vo.ReconScoreValueInfo;
import clinical.web.vo.ScoreValue;

/**
 * Abstract class implementing <code>ISubjectAssessmentManagement</code>
 * interface providing database independent service methods for the subclasses.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: AbstractSubjectAssessmentManImpl.java,v 1.38 2006/07/31
 *          19:55:54 bozyurt Exp $
 */
public abstract class AbstractSubjectAssessmentManImpl extends
		AbstractServiceImpl implements ISubjectAssessmentManagement {
	/** Connection pool service interface */
	// protected IDBPoolService dbPoolService;
	// protected String theDBID;
	protected Log log = LogFactory
			.getLog(AbstractSubjectAssessmentManImpl.class);
	protected static NumberFormat numberFormat = NumberFormat.getInstance();

	public AbstractSubjectAssessmentManImpl(String dbID) throws BaseException {
		super(dbID);
	}

	public abstract List<AssessmentScoreValues> getAssessmentValuesForSubject(
			UserInfo ui, String subjectID, int experimentID,
			List<AssessmentInfo> assessments, int entryID, boolean validated)
			throws SubjectAssessmentManagementException;

	public List<AssessmentInfo> getTheAssessmentForSubject(
			UserInfo ui, String subjectID, int experimentID,
			String assessmentName) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());

			return getTheAssessmentForSubject(con, ui, subjectID, experimentID, assessmentName);
		} catch (Exception x) {
			log.error("Error in getAssessmentValuesForSubject", x);
			throw new SubjectAssessmentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
		
	}
	
	public List<AssessmentInfo> getTheAssessmentForSubject(
			Connection con, UserInfo ui, String subjectID, int experimentID,
			String assessmentName) throws Exception {
		PreparedStatement pst = null;
		try {
			StringBuilder queryBuf = new StringBuilder(200);
			List<AssessmentInfo> asList = new LinkedList<AssessmentInfo>();

			queryBuf
					.append("SELECT  distinct a.assessmentid, a.modtime, a.name, b.componentid,");
			queryBuf
					.append("b.segmentid from nc_assessment a, nc_storedassessment b ");
			queryBuf
					.append("where a.assessmentid = b.assessmentid and b.subjectid = ? ");
			queryBuf.append(" and b.nc_experiment_uniqueid = ?");
			queryBuf.append(" and a.name = ?");
			pst = con.prepareStatement(queryBuf.toString());
			log.info("*** query=" + queryBuf.toString());
			pst.clearParameters();
			pst.setString(1, subjectID);
			pst.setInt(2, experimentID);
			pst.setString(3, assessmentName);

			ResultSet rs = pst.executeQuery();
			while (rs.next()) {
				int assessmentID = rs.getInt(1);

				java.util.Date modTime = rs.getDate(2);
				String name = rs.getString(3);
				int visitID = rs.getInt(4);
				int segmentID = rs.getInt(5);
				AssessmentInfo asi = new AssessmentInfo(assessmentID, name,
						modTime, visitID, segmentID);
				asList.add(asi);
			}

			// get the scores also
			if (!asList.isEmpty()) {
				log.info("getting scores for assessments");
				SubjectAssessmentManHelper.getScores(con, asList);
			}
			return asList;
		} finally {
			if (pst != null) {
				try {
					pst.close();
				} catch (Exception x) {
				}
			}
		}
	}

	public List<AssessmentInfo> getAssessmentsForSubject(Connection con,
			UserInfo ui, String subjectID, int experimentID) throws Exception {
		PreparedStatement pst = null;
		try {
			StringBuilder queryBuf = new StringBuilder(200);
			List<AssessmentInfo> asList = new LinkedList<AssessmentInfo>();

			queryBuf
					.append("SELECT  distinct a.assessmentid, a.modtime, a.name, b.componentid,");
			queryBuf
					.append("b.segmentid from nc_assessment a, nc_storedassessment b ");
			queryBuf
					.append("where a.assessmentid = b.assessmentid and b.subjectid = ? ");
			queryBuf.append(" and b.nc_experiment_uniqueid = ?");
			pst = con.prepareStatement(queryBuf.toString());
			log.info("*** query=" + queryBuf.toString());
			pst.clearParameters();
			pst.setString(1, subjectID);
			pst.setInt(2, experimentID);

			ResultSet rs = pst.executeQuery();
			while (rs.next()) {
				int assessmentID = rs.getInt(1);

				java.util.Date modTime = rs.getDate(2);
				String name = rs.getString(3);
				int visitID = rs.getInt(4);
				int segmentID = rs.getInt(5);
				AssessmentInfo asi = new AssessmentInfo(assessmentID, name,
						modTime, visitID, segmentID);
				asList.add(asi);
			}

			// get the scores also
			if (!asList.isEmpty()) {
				log.info("getting scores for assessments");
				SubjectAssessmentManHelper.getScores(con, asList);
			}
			return asList;
		} finally {
			if (pst != null) {
				try {
					pst.close();
				} catch (Exception x) {
				}
			}
		}
	}

	public List<AssessmentInfo> getAssessmentsForSubject(UserInfo ui,
			String subjectID, int experimentID)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());

			return getAssessmentsForSubject(con, ui, subjectID, experimentID);
		} catch (Exception x) {
			log.error("Error in getAssessmentValuesForSubject", x);
			throw new SubjectAssessmentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public java.util.Date getTimeStampForAssessment(UserInfo ui,
			String subjectID, int experimentID, int visitID, int segmentID,
			int assessmentID) throws SubjectAssessmentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());

			StoredassessmentDAO dao = DAOFactory
					.createStoredassessmentDAO(this.theDBID);
			Storedassessment criteria = new Storedassessment();
			criteria.setAssessmentid(new BigDecimal(String
					.valueOf(assessmentID)));
			criteria.setComponentid(new BigDecimal(String.valueOf(visitID)));
			criteria.setSubjectid(subjectID);
			criteria.setSegmentid(new BigDecimal(String.valueOf(segmentID)));
			criteria.setNcExperimentUniqueid(new BigDecimal(String
					.valueOf(experimentID)));

			List<Storedassessment> list = dao.find(con, criteria);
			if (!list.isEmpty()) {
				Storedassessment sa = list.get(0);
				return new java.util.Date(sa.getTimeStamp().getTime());
			}
		} catch (Exception x) {
			log.error("Error in getAssessmentValuesForSubject", x);
			throw new SubjectAssessmentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
		return null;
	}

	protected Map<Integer, AssessmentInfo> convertToMap(
			List<AssessmentInfo> assessments) {
		Map<Integer, AssessmentInfo> map = new HashMap<Integer, AssessmentInfo>(
				17);
		for (Iterator<AssessmentInfo> iter = assessments.iterator(); iter
				.hasNext();) {
			AssessmentInfo asi = iter.next();
			map.put(new Integer(asi.getAssessmentID()), asi);
		}
		return map;
	}

	protected void checkForInformant(String dbID, Connection con,
			String informantRelation) throws Exception {
		AssessmentinformantDAO dao = DAOFactory
				.createAssessmentinformantDAO(dbID);
		Assessmentinformant ai = new Assessmentinformant();
		ai.setInformantrelation(informantRelation);
		log.info("checking for informant relation " + informantRelation);
		List<Assessmentinformant> aiList = dao.find(con, ai);
		if (aiList.size() != 1) {
			throw new SubjectAssessmentManagementException(
					"Cannot find the Informant relation record!");
		}
	}

	public void insertAssessmentValues(UserInfo ui, String dbID,
			AssessmentScoreValues asv,
			Map<String, Map<String, String>> dbVarMetaDataMap)
			throws SubjectAssessmentManagementException {
		insertAssessmentValues(ui, dbID, asv, null, false, dbVarMetaDataMap);
	}

	public void insertAssessmentValues(Connection con, UserInfo ui,
			String dbID, AssessmentScoreValues asv, BigDecimal entryID,
			boolean validated, Map<String, Map<String, String>> dbVarMetaDataMap)
			throws Exception {
		StoredassessmentDAO saDAO = DAOFactory.createStoredassessmentDAO(dbID);

		Storedassessment criteria = new Storedassessment();
		criteria.setAssessmentid(new BigDecimal(String.valueOf(asv
				.getAssessmentID())));
		criteria
				.setComponentid(new BigDecimal(String.valueOf(asv.getVisitID())));
		criteria
				.setSegmentid(new BigDecimal(String.valueOf(asv.getSegmentID())));
		criteria.setNcExperimentUniqueid(new BigDecimal(String.valueOf(asv
				.getExperimentID())));
		criteria.setSubjectid(asv.getSubjectID());
		List<Storedassessment> saList = saDAO.find(con, criteria);

		checkForInformant(dbID, con, asv.getInformantRelation());

		Databaseuser databaseUser = getDatabaseUser(ui, dbID,
				Constants.USERCLASS_ADMIN);
		Databaseuser keyer = getKeyer(ui, dbID);

		BigDecimal storedAsID = null;
		if (saList.isEmpty()) {

			criteria.setModuser(databaseUser.getUniqueid());
			criteria.setOwner(databaseUser.getUniqueid());
			criteria.setModtime(new java.util.Date());
			criteria.setTableid(getTableID(dbID, ui,
					Constants.STORED_ASSESSMENT_DB_TABLE));
			criteria.setTimeStamp(new Timestamp(asv.getTimeStamp().getTime()));

			storedAsID = ServiceFactory.getSequenceHelper(dbID).getNextUID(ui,
					Constants.STORED_ASSESSMENT_DB_TABLE, "uniqueid");
			criteria.setUniqueid(storedAsID);

			criteria.setIsvalidated(new Boolean(validated));

			criteria.setIstimeinterval(new Boolean(false));
			/** @todo proper handling of status field? */
			// from nc_assessmentstatus table meaning, that data quality is
			// not verified
			criteria.setStatus("000");
			// set informant metadata
			criteria.setInformantid(asv.getInformantID());
			criteria.setInformantrelation(asv.getInformantRelation());
			log.info("storedassessment record to be inserted "
					+ criteria.toString());
			saDAO.insert(con, criteria);
			log.info("inserted " + criteria.toString());
		} else {
			if (saList.size() != 1) {
				throw new SubjectAssessmentManagementException(
						"Expected only one StoredAssessment record; Found "
								+ saList.size());
			}
			Storedassessment sa = saList.get(0);
			storedAsID = sa.getUniqueid();
			// check if informant needs update
			boolean needsUpdate = false;
			needsUpdate |= !asv.getInformantID().equals(sa.getInformantid());
			needsUpdate |= !asv.getInformantRelation().equals(
					sa.getInformantrelation());
			if (needsUpdate) {
				Storedassessment newSa = new Storedassessment();
				newSa.setInformantid(asv.getInformantID());
				newSa.setInformantrelation(asv.getInformantRelation());
				saDAO.update(con, newSa, sa);
			}
		}

		int raterID = SubjectAssessmentManHelper.findClinicalRaterID(asv, con,
				dbID, ui);
		log.info("raterID=" + raterID);

		// now add assessmentxxxx records for score values
		Storedassessment sa = new Storedassessment();
		sa.setUniqueid(storedAsID);
		if ((asv != null) && (asv.getScoreValues() != null)) {
			BigDecimal missClassificationID = SubjectAssessmentManHelper
					.getClassificationID(ui, dbID,
							Constants.MISSING_VALUE_CLASSIFICATION);

			SubjectAsmContext context = new SubjectAsmContext(dbID,
					dbVarMetaDataMap, entryID, ui);
			context.setKeyerID(keyer.getUniqueid());
			context.setMissClassificationID(missClassificationID);
			context.setValidated(validated);
			context.setAsv(asv);
			context.setRaterID(raterID);

			for (ScoreValue sv : asv.getScoreValues()) {
				AssessmentInfo.ScoreInfo si = asv.findScoreInfo(sv.getName());

				insertScoreValue(context, con, sa, databaseUser, sv, si);
			}
		} else {
			System.out.println("ERROR: asv is invalid: " + asv);
		}
	}

	public void insertAssessmentValues(UserInfo ui, String dbID,
			AssessmentScoreValues asv, BigDecimal entryID, boolean validated,
			Map<String, Map<String, String>> dbVarMetaDataMap)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			log.info("in insertAssessmentValues() ");
			insertAssessmentValues(con, ui, dbID, asv, entryID, validated,
					dbVarMetaDataMap);
			con.commit();
		} catch (Exception x) {
			log.error("Error in insertAssessmentValues", x);
			try {
				con.rollback();
			} catch (Exception e) {
				throw new SubjectAssessmentManagementException(x);
			}
			if (x instanceof SubjectAssessmentManagementException) {
				throw (SubjectAssessmentManagementException) x;
			}

			throw new SubjectAssessmentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public static class Params {
		ScoreValue sv;
		AssessmentInfo.ScoreInfo si;
		BigDecimal tableID;
		int assessmentID;
		Databaseuser databaseUser;
		BigDecimal storedAssessmentID, uniqueID;
		BigDecimal entryID;
		boolean validated;
		BigDecimal missClassificationID;

		public Params(ScoreValue sv, AssessmentInfo.ScoreInfo si,
				BigDecimal tableID, int assessmentID,
				Databaseuser databaseUser, BigDecimal storedAssessmentID,
				BigDecimal uniqueID, BigDecimal entryID, boolean validated,
				BigDecimal missClassificationID) {
			this.sv = sv;
			this.si = si;
			this.tableID = tableID;
			this.assessmentID = assessmentID;
			this.databaseUser = databaseUser;
			this.storedAssessmentID = storedAssessmentID;
			this.uniqueID = uniqueID;
			this.entryID = entryID;
			this.validated = validated;
			this.missClassificationID = missClassificationID;
		}
	}

	public void persisInformantInfo(String dbID, UserInfo ui,
			String informantID, String informantRelation,
			AssessmentScoreValues asv)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			StoredassessmentDAO saDAO = DAOFactory
					.createStoredassessmentDAO(dbID);

			Storedassessment criteria = new Storedassessment();
			criteria.setAssessmentid(new BigDecimal(String.valueOf(asv
					.getAssessmentID())));
			criteria.setComponentid(new BigDecimal(String.valueOf(asv
					.getVisitID())));
			criteria.setSegmentid(new BigDecimal(String.valueOf(asv
					.getSegmentID())));
			criteria.setNcExperimentUniqueid(new BigDecimal(String.valueOf(asv
					.getExperimentID())));
			criteria.setSubjectid(asv.getSubjectID());
			List<Storedassessment> saList = saDAO.find(con, criteria);
			if (saList.size() != 1) {
				throw new SubjectAssessmentManagementException(
						"Cannot find the stored assessment record!");
			}

			AssessmentinformantDAO dao = DAOFactory
					.createAssessmentinformantDAO(dbID);
			Assessmentinformant ai = new Assessmentinformant();
			ai.setInformantrelation(informantRelation);
			List<Assessmentinformant> aiList = dao.find(con, ai);
			if (aiList.size() != 1) {
				throw new SubjectAssessmentManagementException(
						"Cannot find the Informant relation record!");
			}

			Storedassessment newSa = new Storedassessment();
			newSa.setInformantrelation(informantRelation);
			newSa.setInformantid(informantID);
			saDAO.update(con, newSa, criteria);
		} catch (Exception x) {
			log.error("Error in insertInformant", x);
			try {
				con.rollback();
			} catch (Exception e) {
				throw new SubjectAssessmentManagementException(x);
			}
			throw new SubjectAssessmentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void insertScoreValues(String dbID, UserInfo ui,
			AssessmentScoreValues asv, List<ScoreValue> scoreValues,
			Map<String, Map<String, String>> dbVarMetaDataMap)
			throws SubjectAssessmentManagementException {
		insertScoreValues(dbID, ui, asv, scoreValues, null, false,
				dbVarMetaDataMap);
	}

	public void insertScoreValues(String dbID, UserInfo ui,
			AssessmentScoreValues asv, List<ScoreValue> scoreValues,
			BigDecimal entryID, boolean validated,
			Map<String, Map<String, String>> dbVarMetaDataMap)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		try {
			log.info("in insertScoreValues()");
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			Storedassessment sa = getStoredAssessment(dbID, asv, con);

			int raterID = SubjectAssessmentManHelper.findClinicalRaterID(asv,
					con, dbID, ui);
			log.info("raterID=" + raterID);

			Databaseuser databaseUser = getDatabaseUser(ui, dbID,
					Constants.USERCLASS_ADMIN);
			Databaseuser keyer = getKeyer(ui, dbID);
			BigDecimal missClassificationID = SubjectAssessmentManHelper
					.getClassificationID(ui, dbID,
							Constants.MISSING_VALUE_CLASSIFICATION);

			SubjectAsmContext context = new SubjectAsmContext(dbID,
					dbVarMetaDataMap, entryID, ui);
			context.setKeyerID(keyer.getUniqueid());
			context.setMissClassificationID(missClassificationID);
			context.setValidated(validated);
			context.setAsv(asv);
			context.setRaterID(raterID);

			for (ScoreValue sv : scoreValues) {
				AssessmentInfo.ScoreInfo si = asv.findScoreInfo(sv.getName());

				insertScoreValue(context, con, sa, databaseUser, sv, si);
				/*
				 * insertScoreValue(dbID, ui, asv, con, sa, raterID,
				 * databaseUser, sv, si, entryID, keyer.getUniqueid(),
				 * validated, dbVarMetaDataMap, missClassificationID);
				 */
			}
		} catch (Exception x) {
			log.error("Error in insertScoreValues", x);
			try {
				con.rollback();
			} catch (Exception e) {
				throw new SubjectAssessmentManagementException(x);
			}
			throw new SubjectAssessmentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void deleteScoreValues(String dbID, UserInfo ui,
			AssessmentScoreValues asv, List<ScoreValue> scoreValues, int entryID)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		try {
			log.info("in deleteScoreValues()");
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			Storedassessment sa = findStoredAssessment(dbID, asv, con);
			if (sa != null) {
				SubjectAsmContext context = new SubjectAsmContext(dbID, null,
						GenUtils.toBigDecimal(entryID), ui);
				context.setAsv(asv);

				for (ScoreValue sv : scoreValues) {
					AssessmentInfo.ScoreInfo si = asv.findScoreInfo(sv
							.getName());
					deleteScoreValue(context, con, sa, sv, si);
				}
			}
		} catch (Exception x) {
			log.error("Error in insertScoreValues", x);
			try {
				con.rollback();
			} catch (Exception e) {
				throw new SubjectAssessmentManagementException(x);
			}
			throw new SubjectAssessmentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	private void deleteScoreValue(SubjectAsmContext context, Connection con,
			Storedassessment sa, ScoreValue sv, ScoreInfo si) throws Exception {
		AssessmentScoreValues asv = context.getAsv();
		int entryID = context.getEntryID().intValue();
		String dbID = context.getDbID();
		if (si.getType().equalsIgnoreCase("integer")) {
			AssessmentintegerDAO aiDAO = DAOFactory
					.createAssessmentintegerDAO(dbID);
			Assessmentinteger aic = SubjectAssessmentManHelper
					.prepareIntegerScoreValueCriteria(si, sa.getUniqueid(), asv
							.getAssessmentID(), entryID, sv.getScoreOrder());
			aiDAO.delete(con, aic);
		} else if (si.getType().equalsIgnoreCase("float")) {
			AssessmentfloatDAO afDAO = DAOFactory
					.createAssessmentfloatDAO(dbID);
			Assessmentfloat afc = SubjectAssessmentManHelper
					.prepareFloatScoreValueCriteria(si, sa.getUniqueid(), asv
							.getAssessmentID(), entryID, sv.getScoreOrder());
			afDAO.delete(con, afc);
		} else if (si.getType().equalsIgnoreCase("varchar")) {
			AssessmentvarcharDAO avDAO = DAOFactory
					.createAssessmentvarcharDAO(dbID);
			Assessmentvarchar avc = SubjectAssessmentManHelper
					.prepareVarcharScoreValueCriteria(si, sa.getUniqueid(), asv
							.getAssessmentID(), entryID, sv.getScoreOrder());
			avDAO.delete(con, avc);
		} else if (si.getType().equalsIgnoreCase("boolean")) {
			/**
			 * @todo since nc_assessmentboolean table is not used currently the
			 *       type conversions (if necessary) is not well defined, hence
			 *       this portion logic might not be consistent with the rest of
			 *       assessment data type tables especially in missing value
			 *       handling
			 */
			AssessmentbooleanDAO abDAO = DAOFactory
					.createAssessmentbooleanDAO(dbID);
			Assessmentboolean abc = SubjectAssessmentManHelper
					.prepareBooleanScoreValueCriteria(si, sa.getUniqueid(), asv
							.getAssessmentID(), entryID, sv.getScoreOrder());
			abDAO.delete(con, abc);
		} else if (si.getType().equalsIgnoreCase("timestamp")) {
			AssessmenttimestampDAO atDAO = DAOFactory
					.createAssessmenttimestampDAO(dbID);
			Assessmenttimestamp atc = SubjectAssessmentManHelper
					.prepareTimestampScoreValueCriteria(si, sa.getUniqueid(),
							asv.getAssessmentID());
			atDAO.delete(con, atc);
		}
	}

	private Storedassessment getStoredAssessment(String dbID,
			AssessmentScoreValues asv, Connection con) throws Exception,
			SubjectAssessmentManagementException {
		StoredassessmentDAO saDAO = DAOFactory.createStoredassessmentDAO(dbID);

		Storedassessment criteria = new Storedassessment();
		criteria.setAssessmentid(new BigDecimal(String.valueOf(asv
				.getAssessmentID())));
		criteria
				.setComponentid(new BigDecimal(String.valueOf(asv.getVisitID())));
		criteria
				.setSegmentid(new BigDecimal(String.valueOf(asv.getSegmentID())));
		criteria.setNcExperimentUniqueid(new BigDecimal(String.valueOf(asv
				.getExperimentID())));
		criteria.setSubjectid(asv.getSubjectID());
		List<Storedassessment> saList = saDAO.find(con, criteria);
		if (saList.size() != 1) {
			throw new SubjectAssessmentManagementException(
					"Cannot find the stored assessment record! "
							+ criteria.toString());
		}

		Storedassessment sa = saList.get(0);
		return sa;
	}

	private Storedassessment findStoredAssessment(String dbID,
			AssessmentScoreValues asv, Connection con) throws Exception {
		StoredassessmentDAO saDAO = DAOFactory.createStoredassessmentDAO(dbID);

		Storedassessment criteria = new Storedassessment();
		criteria.setAssessmentid(new BigDecimal(String.valueOf(asv
				.getAssessmentID())));
		criteria
				.setComponentid(new BigDecimal(String.valueOf(asv.getVisitID())));
		criteria
				.setSegmentid(new BigDecimal(String.valueOf(asv.getSegmentID())));
		criteria.setNcExperimentUniqueid(new BigDecimal(String.valueOf(asv
				.getExperimentID())));
		criteria.setSubjectid(asv.getSubjectID());
		List<Storedassessment> saList = saDAO.find(con, criteria);
		if (saList.isEmpty()) {
			return null;
		}
		assert (saList.size() == 1);
		return saList.get(0);
	}

	private String getTableName(ScoreInfo si) {
		if (si.getType().equalsIgnoreCase("integer"))
			return Constants.ASSESSMENT_INTEGER_DB_TABLE;
		if (si.getType().equalsIgnoreCase("float"))
			return Constants.ASSESSMENT_FLOAT_DB_TABLE;
		if (si.getType().equalsIgnoreCase("varchar"))
			return Constants.ASSESSMENT_VARCHAR_DB_TABLE;
		if (si.getType().equalsIgnoreCase("boolean"))
			return Constants.ASSESSMENT_BOOLEAN_DB_TABLE;
		if (si.getType().equalsIgnoreCase("timestamp"))
			return Constants.ASSESSMENT_TIMESTAMP_DB_TABLE;

		throw new RuntimeException("Unsupported score type:" + si.getType());
	}

	private void insertScoreValue(SubjectAsmContext context, Connection con,
			Storedassessment sa, Databaseuser databaseUser, ScoreValue sv,
			ScoreInfo si) throws Exception {
		String dbID = context.getDbID();
		UserInfo ui = context.getUi();
		AssessmentScoreValues asv = context.getAsv();
		int raterID = context.getRaterID();
		BigDecimal keyerID = context.getKeyerID();

		BigDecimal aiTableID = getTableID(dbID, ui,
				Constants.ASSESSMENT_INTEGER_DB_TABLE);
		BigDecimal afTableID = getTableID(dbID, ui,
				Constants.ASSESSMENT_FLOAT_DB_TABLE);
		BigDecimal avTableID = getTableID(dbID, ui,
				Constants.ASSESSMENT_VARCHAR_DB_TABLE);
		BigDecimal abTableID = getTableID(dbID, ui,
				Constants.ASSESSMENT_BOOLEAN_DB_TABLE);
		BigDecimal atTableID = getTableID(dbID, ui,
				Constants.ASSESSMENT_TIMESTAMP_DB_TABLE);

		BigDecimal uniqueID = null;
		do {
			uniqueID = ServiceFactory.getSequenceHelper(dbID).getNextUID(ui,
					getTableName(si), "nc_assessmentdatauniqueid");

			Params params = new Params(sv, si, aiTableID,
					asv.getAssessmentID(), databaseUser, sa.getUniqueid(),
					uniqueID, context.getEntryID(), context.isValidated(),
					context.getMissClassificationID());

			if (si.getType().equalsIgnoreCase("integer")) {
				AssessmentintegerDAO aiDAO = DAOFactory
						.createAssessmentintegerDAO(dbID);

				Assessmentinteger ai = SubjectAssessmentManHelper
						.prepareIntegerScoreValueRecord(params);

				if (raterID != -1) {
					ai.setRaterid(GenUtils.toBigDecimal(raterID));
				}
				ai.setKeyerid(keyerID);
				ai.setSubjectid(asv.getSubjectID());
				log.info("inserting AssessmentInteger " + ai.toString());
				aiDAO.insert(con, ai);
			} else if (si.getType().equalsIgnoreCase("float")) {
				AssessmentfloatDAO afDAO = DAOFactory
						.createAssessmentfloatDAO(dbID);
				params.tableID = afTableID;

				Assessmentfloat af = SubjectAssessmentManHelper
						.prepareFloatScoreValueRecord(params);

				if (raterID != -1) {
					af.setRaterid(GenUtils.toBigDecimal(raterID));
				}
				af.setKeyerid(keyerID);

				af.setSubjectid(asv.getSubjectID());
				log.info("inserting AssessmentFloat " + af.toString());
				afDAO.insert(con, af);
			} else if (si.getType().equalsIgnoreCase("varchar")) {
				AssessmentvarcharDAO avDAO = DAOFactory
						.createAssessmentvarcharDAO(dbID);
				params.tableID = avTableID;
				Assessmentvarchar av = SubjectAssessmentManHelper
						.prepareVarcharScoreValueRecord(params);

				if (raterID != -1) {
					av.setRaterid(new BigDecimal(String.valueOf(raterID)));
				}

				av.setKeyerid(keyerID);
				av.setSubjectid(asv.getSubjectID());
				log.info("inserting AssessmentVarchar " + av.toString());
				avDAO.insert(con, av);
			} else if (si.getType().equalsIgnoreCase("boolean")) {
				AssessmentbooleanDAO abDAO = DAOFactory
						.createAssessmentbooleanDAO(dbID);
				params.tableID = abTableID;

				Assessmentboolean ab = SubjectAssessmentManHelper
						.prepareBooleanScoreValueRecord(params);

				if (raterID != -1) {
					ab.setRaterid(GenUtils.toBigDecimal(raterID));
				}
				ab.setKeyerid(keyerID);
				ab.setSubjectid(asv.getSubjectID());
				log.info("inserting AssessmentBoolean " + ab.toString());
				abDAO.insert(con, ab);
			} else if (si.getType().equalsIgnoreCase("timestamp")) {
				AssessmenttimestampDAO atDAO = DAOFactory
						.createAssessmenttimestampDAO(dbID);
				params.tableID = atTableID;
				Assessmenttimestamp at = SubjectAssessmentManHelper
						.prepareTimestampScoreValueRecord(params, context
								.getDbVarMetaDataMap());

				if (raterID != -1) {
					at.setRaterid(new BigDecimal(String.valueOf(raterID)));
				}
				at.setKeyerid(keyerID);
				at.setSubjectid(asv.getSubjectID());
				log.info("inserting AssessmentTimestamp " + at.toString());
				atDAO.insert(con, at);
			} else {
				throw new SubjectAssessmentManagementException(
						"Unsupported score type:" + si.getType());
			}
			sv = sv.getNext();
		} while (sv != null);
	}

	public void updateAssessmentValues(UserInfo ui, String dbID,
			AssessmentScoreValues asv, List<String> scoresToBeUpdated,
			int entryID, Map<String, Map<String, String>> dbVarMetaDataMap)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		try {
			log.info("***** in updateAssessmentValues()");
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			updateAssessmentValues(con, ui, dbID, asv, scoresToBeUpdated,
					entryID, dbVarMetaDataMap);
			con.commit();
		} catch (Exception x) {
			log.error("Error in updateAssessmentValues", x);
			try {
				con.rollback();
			} catch (Exception e) {
				throw new SubjectAssessmentManagementException(x);
			}
			throw new SubjectAssessmentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void updateAssessmentValues(Connection con, UserInfo ui,
			String dbID, AssessmentScoreValues asv,
			List<String> scoresToBeUpdated, int entryID,
			Map<String, Map<String, String>> dbVarMetaDataMap) throws Exception {
		StoredassessmentDAO saDAO = DAOFactory
				.createStoredassessmentDAO(dbID);

		Storedassessment criteria = new Storedassessment();
		criteria.setAssessmentid(new BigDecimal(String.valueOf(asv
				.getAssessmentID())));
		criteria.setComponentid(new BigDecimal(String.valueOf(asv
				.getVisitID())));
		criteria.setSegmentid(new BigDecimal(String.valueOf(asv
				.getSegmentID())));
		criteria.setNcExperimentUniqueid(new BigDecimal(String.valueOf(asv
				.getExperimentID())));
		criteria.setSubjectid(asv.getSubjectID());

		List<Storedassessment> saList = saDAO.find(con, criteria);

		if (!saList.isEmpty() && saList.size() == 1) {
			Storedassessment sa = saList.get(0);
			if (log.isDebugEnabled()) {
				log.debug("found sa=" + sa.toString());
				// check if stored assessment needs update of informant info
				// if so update it
			}
			boolean needsUpdate = false;
			if (asv.getInformantID() == null && sa.getInformantid() != null) {
				asv.setInformantID(sa.getInformantid());
			}
			if (asv.getInformantRelation() == null
					&& sa.getInformantrelation() != null) {
				asv.setInformantRelation(sa.getInformantrelation());
			}

			if (asv.getInformantID() == null) {
				log.error("asv informantID=" + asv.getInformantID());
			}
			needsUpdate |= !asv.getInformantID()
					.equals(sa.getInformantid());
			needsUpdate |= !asv.getInformantRelation().equals(
					sa.getInformantrelation());
			if (needsUpdate) {
				Storedassessment newSa = new Storedassessment();
				newSa.setInformantid(asv.getInformantID());
				newSa.setInformantrelation(asv.getInformantRelation());
				saDAO.update(con, newSa, sa);
			}

			int raterID = SubjectAssessmentManHelper.findClinicalRaterID(
					asv, con, dbID, ui);
			log.info("raterID=" + raterID);

			BigDecimal missClassificationID = SubjectAssessmentManHelper
					.getClassificationID(ui, dbID,
							Constants.MISSING_VALUE_CLASSIFICATION);
			for (String scoreName : scoresToBeUpdated) {
				// log.info("updating score " + scoreName);
				ScoreValue sv = null;
				do {
					if (sv == null) {
						sv = asv.getScoreValue(scoreName, 1);
					}
					// log.info("found score value " + sv.getName() + " " +
					// sv.getValue() );
					AssessmentInfo.ScoreInfo si = asv.findScoreInfo(sv
							.getName());

					if (si.getType().equalsIgnoreCase("integer")) {
						AssessmentintegerDAO aiDAO = DAOFactory
								.createAssessmentintegerDAO(dbID);
						Assessmentinteger aic = SubjectAssessmentManHelper
								.prepareIntegerScoreValueCriteria(si, sa
										.getUniqueid(), asv
										.getAssessmentID(), entryID, sv
										.getScoreOrder());
						Assessmentinteger ai = new Assessmentinteger();

						String value = SubjectAssessmentManHelper
								.checkAndPrepareIntScoreValue(sv, ai,
										missClassificationID);

						if (raterID != -1) {
							ai.setRaterid(new BigDecimal(String
									.valueOf(raterID)));
							log.info("set raterID=" + raterID);
						}
						// update dataClassification (if necessary)
						if (!SubjectAssessmentManHelper
								.isIntValueValid(value)) {
							if (sv.getClassification() != null) {
								ai.setClassification(new BigDecimal(sv
										.getClassification().toString()));
							}
						} else {
							// to unset the classification
							unsetClassification(con,
									Constants.ASSESSMENT_INTEGER_DB_TABLE,
									sa.getUniqueid().intValue(), asv
											.getAssessmentID(), scoreName,
									entryID, sv.getScoreOrder());
						}
						aiDAO.update(con, ai, aic);
						if (log.isDebugEnabled()) {
							log.debug("updated record " + aic.toString()
									+ " with " + ai.toString());
						}
					} else if (si.getType().equalsIgnoreCase("float")) {
						AssessmentfloatDAO afDAO = DAOFactory
								.createAssessmentfloatDAO(dbID);
						Assessmentfloat afc = SubjectAssessmentManHelper
								.prepareFloatScoreValueCriteria(si, sa
										.getUniqueid(), asv
										.getAssessmentID(), entryID, sv
										.getScoreOrder());
						Assessmentfloat af = new Assessmentfloat();

						SubjectAssessmentManHelper.checkSetMissingValue(sv,
								missClassificationID);
						af.setTextvalue(sv.getUncorrectedValue());
						if (sv.getUncorrectedValue() == null) {
							af.setNull("textvalue", true);
							af.setNull("datavalue", true);
						}

						// for POSTGRES new Float(sv.getUncorrectedValue())
						// -> new Double(sv.getUncorrectedValue())
						af
								.setDatavalue(sv.getUncorrectedValue() != null ? new Float(
										sv.getUncorrectedValue())
										: null);

						log.info("###### (update) af.dataValue="
								+ af.getDatavalue());
						if (raterID != -1) {
							af.setRaterid(new BigDecimal(String
									.valueOf(raterID)));
						}

						if (!SubjectAssessmentManHelper
								.isFloatValueValid(sv.getUncorrectedValue())) {
							// log.info("!isFloatValueValid for " +
							// sv.toString());
							if (sv.getClassification() != null) {
								af.setClassification(new BigDecimal(sv
										.getClassification().toString()));
							} else {
								log.error("No classification is given for "
										+ sv.toString());
							}
						} else {
							log.info("unsetting classification for "
									+ sv.toString());
							unsetClassification(con,
									Constants.ASSESSMENT_FLOAT_DB_TABLE, sa
											.getUniqueid().intValue(), asv
											.getAssessmentID(), scoreName,
									entryID, sv.getScoreOrder());
						}
						log.info("updating " + af);
						afDAO.update(con, af, afc);

					} else if (si.getType().equalsIgnoreCase("varchar")) {
						AssessmentvarcharDAO avDAO = DAOFactory
								.createAssessmentvarcharDAO(dbID);
						Assessmentvarchar avc = SubjectAssessmentManHelper
								.prepareVarcharScoreValueCriteria(si, sa
										.getUniqueid(), asv
										.getAssessmentID(), entryID, sv
										.getScoreOrder());
						Assessmentvarchar av = new Assessmentvarchar();

						SubjectAssessmentManHelper.checkSetMissingValue(sv,
								missClassificationID);

						av.setTextvalue(sv.getUncorrectedValue());
						av.setDatavalue(sv.getUncorrectedValue());

						if (sv.getUncorrectedValue() == null) {
							av.setNull("datavalue", true);
							av.setNull("textvalue", true);
						}

						if (raterID != -1) {
							av.setRaterid(new BigDecimal(String
									.valueOf(raterID)));
						}

						if (sv.getValue() == null
								|| sv.getValue().length() == 0) {
							if (sv.getClassification() != null) {
								av.setClassification(new BigDecimal(sv
										.getClassification().toString()));
							}
						} else {
							unsetClassification(con,
									Constants.ASSESSMENT_VARCHAR_DB_TABLE,
									sa.getUniqueid().intValue(), asv
											.getAssessmentID(), scoreName,
									entryID, sv.getScoreOrder());
						}
						avDAO.update(con, av, avc);
					} else if (si.getType().equalsIgnoreCase("boolean")) {
						/**
						 * @todo since nc_assessmentboolean table is not
						 *       used currently the type conversions (if
						 *       necessary) is not well defined, hence this
						 *       portion logic might not be consistent with
						 *       the rest of assessment data type tables
						 *       especially in missing value handling
						 */
						AssessmentbooleanDAO abDAO = DAOFactory
								.createAssessmentbooleanDAO(dbID);
						Assessmentboolean abc = SubjectAssessmentManHelper
								.prepareBooleanScoreValueCriteria(si, sa
										.getUniqueid(), asv
										.getAssessmentID(), entryID, sv
										.getScoreOrder());
						Assessmentboolean ab = new Assessmentboolean();
						ab.setTextvalue(sv.getValue());
						ab.setDatavalue(new Boolean(GenUtils.toBoolean(sv
								.getValue(), false)));

						if (raterID != -1) {
							ab.setRaterid(new BigDecimal(String
									.valueOf(raterID)));
							log.info("set raterID=" + raterID);
						}
						// update dataClassification (if necessary)
						if (!SubjectAssessmentManHelper.isIntValueValid(sv
								.getValue())) {
							if (sv.getClassification() != null) {
								ab.setClassification(new BigDecimal(sv
										.getClassification().toString()));
							}
						} else {
							// to unset the classification
							unsetClassification(con,
									Constants.ASSESSMENT_BOOLEAN_DB_TABLE,
									sa.getUniqueid().intValue(), asv
											.getAssessmentID(), scoreName,
									entryID, sv.getScoreOrder());
						}
						abDAO.update(con, ab, abc);
						if (log.isDebugEnabled()) {
							log.debug("updated record " + abc.toString()
									+ " with " + ab.toString());
						}
					} else if (si.getType().equalsIgnoreCase("timestamp")) {
						AssessmenttimestampDAO atDAO = DAOFactory
								.createAssessmenttimestampDAO(dbID);
						Assessmenttimestamp atc = SubjectAssessmentManHelper
								.prepareTimestampScoreValueCriteria(si, sa
										.getUniqueid(), asv
										.getAssessmentID());
						Assessmenttimestamp at = new Assessmenttimestamp();

						SubjectAssessmentManHelper.checkSetMissingValue(sv,
								missClassificationID);
						at.setTextvalue(sv.getUncorrectedValue());

						if (sv.getUncorrectedValue() != null) {
							assert (dbVarMetaDataMap.get(sv.getName()) != null);
							Map<String, String> metaDataMap = (Map<String, String>) dbVarMetaDataMap
									.get(sv.getName());
							assert (metaDataMap.get("format") != null);
							String tsFormat = (String) metaDataMap
									.get("format");
							SimpleDateFormat df = new SimpleDateFormat(
									tsFormat);
							df.setLenient(false);
							java.util.Date d = df.parse(sv.getValue());
							at.setDatavalue(new java.sql.Timestamp(d
									.getTime()));
						} else {
							at.setDatavalue(null);
						}

						if (sv.getUncorrectedValue() == null) {
							at.setNull("textvalue", true);
							at.setNull("datavalue", true);
						}

						if (raterID != -1) {
							at.setRaterid(new BigDecimal(String
									.valueOf(raterID)));
						}

						if (sv.getClassification() != null) {
							at.setClassification(new BigDecimal(sv
									.getClassification().toString()));
						} else {
							unsetClassification(
									con,
									Constants.ASSESSMENT_TIMESTAMP_DB_TABLE,
									sa.getUniqueid().intValue(), asv
											.getAssessmentID(), scoreName,
									entryID, sv.getScoreOrder());
						}
						atDAO.update(con, at, atc);
					} else {
						throw new SubjectAssessmentManagementException(
								"Unsupported score type:" + si.getType());
					}

					sv = sv.getNext();
				} while (sv != null);

			} // iter
		}
	}

	/**
	 * 
	 * @param con
	 *            JDBC connection
	 * @param tableName
	 *            the name of the nc_assessmentXXXXX table
	 * @param storedAsID
	 *            the uniqueid of the stored assessment record
	 * @param assessmentID
	 *            the unique id of the assessment
	 * @param scoreName
	 * @param entryID
	 * @param scoreOrder
	 * @throws SQLException
	 */
	protected void unsetClassification(Connection con, String tableName,
			int storedAsID, int assessmentID, String scoreName, int entryID,
			int scoreOrder) throws SQLException {
		Statement st = null;
		try {
			st = con.createStatement();
			StringBuffer query = new StringBuffer(128);
			query.append("update ").append(tableName);
			query.append(" set classification = null where assessmentid =");
			query.append(assessmentID).append(" and storedassessmentid = ");
			query.append(storedAsID);
			query.append(" and scorename = ").append(prepareString(scoreName));
			query.append(" and entryid = ").append(entryID);
			query.append(" and scoreorder = ").append(scoreOrder);
			log.info("QUERY=" + query.toString());
			st.execute(query.toString());
		} finally {
			if (st != null) {
				try {
					st.close();
				} catch (Exception x) {
				}
			}
		}
	}

	static class StoredAssessmentInfo {
		int uniqueID;
		int visitID;
		int segmentID;

		public StoredAssessmentInfo(int uid, int visitID, int segmentID) {
			this.uniqueID = uid;
			this.visitID = visitID;
			this.segmentID = segmentID;
		}
	}

	protected List<StoredAssessmentInfo> getStoredAssessmentInfos(
			Connection con, Set<Integer> storedAsIDs) throws SQLException {
		StringBuffer sb = new StringBuffer(128);

		sb
				.append("select uniqueid, componentid, segmentid from nc_storedassessment");
		if (storedAsIDs.isEmpty()) {
			return new LinkedList<StoredAssessmentInfo>();
		} else {
			sb.append(" where uniqueid in (");
			for (Iterator<Integer> iter = storedAsIDs.iterator(); iter
					.hasNext();) {
				Integer saID = iter.next();
				sb.append(saID);
				if (iter.hasNext()) {
					sb.append(',');
				}
			}
			sb.append(")");
		}
		List<StoredAssessmentInfo> saiList = new LinkedList<StoredAssessmentInfo>();
		Statement st = null;
		try {
			st = con.createStatement();
			log.info("getStoredAssessmentInfos - query=" + sb.toString());
			ResultSet rs = st.executeQuery(sb.toString());
			while (rs.next()) {
				StoredAssessmentInfo sai = new StoredAssessmentInfo(rs
						.getInt(1), rs.getInt(2), rs.getInt(3));
				saiList.add(sai);
			}
		} finally {
			DBUtils.close(st);
		}
		return saiList;
	}

	public List<Storedassessment> getStoredAssessmentRecs(String dbID,
			UserInfo ui, String subjectID, int experimentID, int visitID,
			int segmentID, int assessmentID)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());

			StoredassessmentDAO dao = DAOFactory
					.createStoredassessmentDAO(dbID);
			Storedassessment criteria = new Storedassessment();
			criteria.setAssessmentid(new BigDecimal(String
					.valueOf(assessmentID)));
			criteria.setSubjectid(subjectID);
			criteria.setNcExperimentUniqueid(new BigDecimal(String
					.valueOf(experimentID)));
			criteria.setComponentid(new BigDecimal(String.valueOf(visitID)));
			criteria.setSegmentid(new BigDecimal(String.valueOf(segmentID)));

			List<Storedassessment> list = dao.find(con, criteria);
			return list;
		} catch (Exception x) {
			throw new SubjectAssessmentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public List<String> getDynamicQueryResults(UserInfo ui, String sqlQuery)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		Statement st = null;
		try {
			con = pool.getConnection(ui.getName());
			st = con.createStatement();
			ResultSet rs = st.executeQuery(sqlQuery);
			LinkedList<String> results = new LinkedList<String>();
			while (rs.next()) {
				results.add(rs.getString(1));
			}
			rs.close();
			return results;

		} catch (Exception x) {
			throw new SubjectAssessmentManagementException(x);
		} finally {
			DBUtils.close(st);
			releaseConnection(con, ui);
		}
	}

	public Timestamp getSegmentTimeStamp(String dbID, UserInfo ui,
			String subjectID, int experimentID, int visitID, int segmentID)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			ExpsegmentDAO dao = DAOFactory.createExpsegmentDAO(dbID);
			Expsegment criterion = new Expsegment();
			criterion.setNcExperimentUniqueid(GenUtils
					.toBigDecimal(experimentID));
			criterion.setComponentid(GenUtils.toBigDecimal(visitID));
			criterion.setSegmentid(GenUtils.toBigDecimal(segmentID));
			criterion.setSubjectid(subjectID);

			List<Expsegment> segments = dao.find(con, criterion);
			if (segments.isEmpty()) {
				// should not happen
				return null;
			}
			assert (segments.size() == 1);
			if (log.isDebugEnabled()) {
				log.debug("getSegmentTimeStamp = "
						+ (Expsegment) segments.get(0));
			}
			return ((Expsegment) segments.get(0)).getTimeStamp();
		} catch (Exception x) {
			throw new SubjectAssessmentManagementException(x);
		} finally {
			if (con != null) {
				releaseConnection(con, ui);
			}
		}
	}

	public boolean isAssessmentReconciled(UserInfo ui, String subjectID,
			int experimentID, int visitID, int segmentID, int assessmentID)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			Storedassessment criterion = new Storedassessment();
			criterion.setSubjectid(subjectID);
			criterion.setNcExperimentUniqueid(GenUtils
					.toBigDecimal(experimentID));
			criterion.setComponentid(GenUtils.toBigDecimal(visitID));
			criterion.setSegmentid(GenUtils.toBigDecimal(segmentID));
			criterion.setAssessmentid(GenUtils.toBigDecimal(assessmentID));
			StoredassessmentDAO dao = DAOFactory
					.createStoredassessmentDAO(theDBID);
			List<Storedassessment> saList = dao.find(con, criterion);
			assert (saList.size() == 1);
			Storedassessment sa = saList.get(0);
			return sa.getIsvalidated().booleanValue();
		} catch (Exception x) {
			log.error("Error in isAssessmentReconciled", x);
			throw new SubjectAssessmentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public Assessment getAssessment(UserInfo ui, String asName)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			return getAssessment(asName, con);
		} catch (Exception x) {
			log.error("Error in getAssessment", x);
			throw new SubjectAssessmentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public Assessment getAssessment(String asName, Connection con)
			throws Exception {
		AssessmentDAO asDAO = DAOFactory.createAssessmentDAO(theDBID);
		Assessment cr = new Assessment();
		cr.setName(asName);
		List<Assessment> asList = asDAO.find(con, cr);
		if (asList.isEmpty()) {
			return null;
		}
		return asList.get(0);
	}

	public Assessment addAssessment(UserInfo ui, AssessmentInfo asInfo)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		IDBCache dbCache = null;
		try {
			dbCache = ServiceFactory.getDBCache(theDBID);
			List<Securityclassification> secClassList = dbCache
					.getSecurityClassifications(ui, false);
			// FIXME currently using the first Securityclassification returned
			Assertion.assertTrue(secClassList != null
					&& !secClassList.isEmpty());
			Securityclassification secClass = secClassList.get(0);
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			ISequenceHelper sequenceHelper = MinimalServiceFactory
					.getSequenceHelper(theDBID);
			AssessmentDAO asDAO = DAOFactory.createAssessmentDAO(theDBID);
			Assessment as = new Assessment();
			as.setName(asInfo.getName());
			as.setDescription(asInfo.getDescription());
			BigDecimal uniqueid = sequenceHelper.getNextUID(con,
					Constants.ASSESSMENT_TABLE, "uniqueid");
			as.setUniqueid(uniqueid);
			as.setAssessmentid(uniqueid);
			Date now = new Date();
			as.setModtime(now);
			Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
			Assertion.assertNotNull(dbUser);

			as.setOwner(dbUser.getUniqueid());
			as.setModuser(dbUser.getUniqueid());
			as.setTableid(getTableID(ui, Constants.ASSESSMENT_TABLE));
			asDAO.insert(con, as);
			if (!asInfo.getScores().isEmpty()) {
				AssessmentscoreDAO scDAO = DAOFactory
						.createAssessmentscoreDAO(theDBID);

				for (ScoreInfo si : asInfo.getScores()) {
					Assessmentscore sc = new Assessmentscore();
					sc.setScorename(si.getName());
					sc.setAssessmentid(as.getUniqueid());
					sc.setScoretype(si.getType());
					sc.setScorelevel(new BigDecimal(si.getScoreLevel()));
					sc.setScoresequence(new BigDecimal(si.getScoreSequence()));
					sc.setOwner(dbUser.getUniqueid());
					sc.setModuser(dbUser.getUniqueid());
					sc.setModtime(now);
					BigDecimal scUniqueid = sequenceHelper.getNextUID(con,
							Constants.ASSESSMENTSCORE_TABLE, "uniqueid");

					sc.setUniqueid(scUniqueid);
					sc.setAssessmentontology(Constants.DEFAULT_ONT_SRC);
					sc.setAssessmentconcept(Constants.DEFAULT_ONT_CONCEPT);
					sc.setMinanswers(new BigDecimal(1));
					sc.setMaxanswers(new BigDecimal(1));
					sc.setTableid(getTableID(ui,
							Constants.ASSESSMENTSCORE_TABLE));
					sc.setSecurityclassification(secClass
							.getSecurityclassification());
					sc.setNullable(new Boolean(false));
					sc.setIsexcluded(new Boolean(false));
					sc.setIsmodified(new Boolean(false));
					sc.setIsnew(new Boolean(false));
					sc.setIsrequired(new Boolean(true));

					scDAO.insert(con, sc);
				}
			}

			con.commit();
			return as;
		} catch (Exception x) {
			log.error("Error in addAssessment", x);
			try {
				super.handleErrorAndRollBack(con, "Error in addAssessment", x,
						true);
			} catch (BaseException e) {
				throw new SubjectAssessmentManagementException(e);
			}

		} finally {
			releaseConnection(con, ui);
		}
		return null;
	}

	class ScoreValueStats {
		String scoreName;
		int scoreOrder;
		int entryCount = 1;
		List<Integer> entryIDs = new ArrayList<Integer>(2);

		public ScoreValueStats(String scoreName, int scoreOrder) {
			this.scoreName = scoreName;
			this.scoreOrder = scoreOrder;
		}

		public void addEntryID(int entryID) {
			entryIDs.add(new Integer(entryID));
		}
	}

	public List<MissingEntry> getMissingEntryItems(UserInfo ui,
			String subjectID, int experimentID, int visitID, int segmentID,
			int assessmentID) throws SubjectAssessmentManagementException {
		Connection con = null;
		try {
			// keyed by the scoreName
			Map<String, ScoreValueStats> map = new HashMap<String, ScoreValueStats>();
			con = pool.getConnection(ui.getName());
			Map<String, Integer> scoreNameMap = SubjectAssessmentManHelper
					.getScoreNameMap(con, assessmentID);

			prepareScoreValueStats(ui, subjectID, experimentID, visitID,
					segmentID, assessmentID, con, map);
			Map<String, List<ScoreValueStats>> nameOrderListMap = prepareOrderListMap(map);
			map = null;

			Map<String, String> qtMap = getQuestionTextMap(con, assessmentID);

			List<MissingEntry> missingEntries = new ArrayList<MissingEntry>();
			for (String scoreName : scoreNameMap.keySet()) {
				int scoreSequence = -1;
				Integer ssVal = scoreNameMap.get(scoreName);
				if (ssVal != null) {
					scoreSequence = ssVal.intValue();
				}
				List<ScoreValueStats> soList = nameOrderListMap.get(scoreName);
				if (soList != null) {
					for (ScoreValueStats svs : soList) {
						if (svs.entryCount != 2) {
							int entryID = -1;
							if (svs.entryIDs.size() == 1) {
								int id = svs.entryIDs.get(0).intValue();
								entryID = id == 1 ? 2 : 1;
							}
							MissingEntry me = new MissingEntry(svs.scoreName,
									entryID, qtMap.get(svs.scoreName),
									scoreSequence);
							missingEntries.add(me);
						}
					}
				} else {
					// no value is specified for this score
					String qt = qtMap.get(scoreName);
					MissingEntry me = new MissingEntry(scoreName, 1, qt,
							scoreSequence);
					missingEntries.add(me);
					me = new MissingEntry(scoreName, 2, qt, scoreSequence);
					missingEntries.add(me);
				}
			}

			return missingEntries;
		} catch (Exception x) {
			log.error("Error in isAssessmentReadyForReconcile", x);
			throw new SubjectAssessmentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	private Map<String, String> getQuestionTextMap(Connection con,
			int assessmentID) throws SQLException {
		Map<String, String> qtMap = new HashMap<String, String>();
		Statement st = null;
		StringBuffer buf = new StringBuffer(128);
		try {
			st = con.createStatement();
			buf
					.append("select scorename, itemleadingtext from nc_assessmentitem where assessmentid =");
			buf.append(assessmentID);

			ResultSet rs = st.executeQuery(buf.toString());
			while (rs.next()) {
				String scoreName = rs.getString(1);
				String questionText = rs.getString(2);
				qtMap.put(scoreName, questionText);
			}
			return qtMap;
		} finally {
			DBUtils.close(st);
		}
	}

	// TODO assumption: within an assessment scorenames are unique
	public boolean isAssessmentReadyForReconcile(UserInfo ui, String subjectID,
			int experimentID, int visitID, int segmentID, int assessmentID)
			throws SubjectAssessmentManagementException {

		Connection con = null;
		try {
			// keyed by the score name
			Map<String, ScoreValueStats> map = new HashMap<String, ScoreValueStats>();

			con = pool.getConnection(ui.getName());
			Map<String, Integer> scoreNameMap = SubjectAssessmentManHelper
					.getScoreNameMap(con, assessmentID);

			prepareScoreValueStats(ui, subjectID, experimentID, visitID,
					segmentID, assessmentID, con, map);

			if (scoreNameMap.size() > map.size()) {
				return false;
			}

			Map<String, List<ScoreValueStats>> nameOrderListMap = prepareOrderListMap(map);
			map = null;

			for (String scoreName : scoreNameMap.keySet()) {
				List<ScoreValueStats> soList = nameOrderListMap.get(scoreName);
				if (soList == null) {
					// means for that score no value is supplied in any of the
					// entries
					log.info("For score " + scoreName
							+ " is supplied in any of the entries");
					return false;
				}
				for (ScoreValueStats svs : soList) {
					if (svs.entryCount != 2) {
						return false;
					}
				}
			}
			return true;

		} catch (Exception x) {
			log.error("Error in isAssessmentReadyForReconcile", x);
			throw new SubjectAssessmentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	private Map<String, List<ScoreValueStats>> prepareOrderListMap(
			Map<String, ScoreValueStats> map) {
		Map<String, List<ScoreValueStats>> nameOrderListMap = new HashMap<String, List<ScoreValueStats>>();
		for (Iterator<ScoreValueStats> iter = map.values().iterator(); iter
				.hasNext();) {
			ScoreValueStats svs = iter.next();
			List<ScoreValueStats> soList = nameOrderListMap.get(svs.scoreName);
			if (soList == null) {
				soList = new LinkedList<ScoreValueStats>();
				nameOrderListMap.put(svs.scoreName, soList);
			}
			soList.add(svs);
		}
		return nameOrderListMap;
	}

	private void prepareScoreValueStats(UserInfo ui, String subjectID,
			int experimentID, int visitID, int segmentID, int assessmentID,
			Connection con, Map<String, ScoreValueStats> map)
			throws SQLException, DBPoolServiceException {
		StringBuffer buf = new StringBuffer(256);
		Statement st = null;
		try {
			// entryid in nc_assessmendata is not used
			buf
					.append("select scorename, scoreorder, entryid from nc_assessmentdata where ");
			buf.append(prepareBoolean(" isvalidated =", false)).append(" and ");
			String inQuery = prepareInnerQuery(subjectID, assessmentID,
					experimentID, visitID, segmentID);
			buf.append("nc_storedassessment_uniqueid in (").append(inQuery)
					.append(")");
			log.info("prepareScoreValueStats:: query=" + buf.toString());

			st = con.createStatement();

			ResultSet rs = st.executeQuery(buf.toString());
			while (rs.next()) {
				String scoreName = rs.getString(1);
				int scoreOrder = rs.getInt(2);
				int entryID = rs.getInt(3);
				String key = SubjectAssessmentManHelper
						.prepareScoreNameOrderKey(scoreName, scoreOrder);

				ScoreValueStats svs = map.get(key);
				if (svs == null) {
					svs = new ScoreValueStats(scoreName, scoreOrder);
					map.put(key, svs);
				} else {
					svs.entryCount++;
				}
				svs.addEntryID(entryID);
			}
			rs.close();

		} finally {
			DBUtils.close(st);
		}
	}

	public int getNumOfEntries(UserInfo ui, String subjectID, int experimentID,
			int visitID, int segmentID, int assessmentID)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		Statement st = null;
		try {
			StringBuffer buf = new StringBuffer(100);
			buf.append("select distinct ").append(prepareSO("entryid")).append(
					" from ");
			buf.append(prepareSO("nc_assessmentdata")).append(" a, ").append(
					prepareSO("nc_storedassessment"));
			buf.append(" b where ").append(
					prepareSO("a.nc_storedassessment_uniqueid"));
			buf.append(" = ").append(prepareSO("b.uniqueid")).append(" and ");
			buf.append(prepareSO("b.subjectid")).append(" = ");
			buf.append(prepareString(subjectID)).append(" and ").append(
					prepareSO("b.assessmentid"));
			buf.append(" = ").append(assessmentID).append(" and ").append(
					prepareSO("b.nc_experiment_uniqueid"));
			buf.append(" = ").append(experimentID).append(" and ");
			buf.append(prepareSO("b.componentid")).append(" = ")
					.append(visitID).append(" and ");
			buf.append(prepareSO("b.segmentid")).append(" = ")
					.append(segmentID);

			log.info(" getNumOfEntries: query=" + buf.toString());
			con = pool.getConnection(ui.getName());
			st = con.createStatement();
			ResultSet rs = st.executeQuery(buf.toString());
			int count = 0;
			while (rs.next()) {
				count++;
			}
			rs.close();
			return count;
		} catch (Exception x) {
			log.error("Error in getNumOfEntries", x);
			throw new SubjectAssessmentManagementException(x);
		} finally {
			DBUtils.close(st);
			releaseConnection(con, ui);
		}
	}

	/**
	 * 
	 * @param schemaObjName
	 *            String
	 * @return String
	 */
	protected abstract String prepareSO(String schemaObjName);

	protected abstract String prepareString(String value);

	public Map<String, ReconScoreValueInfo> getReconciliationDataForAssessment(
			UserInfo ui, String subjectID, int experimentID, int visitID,
			int segmentID, int assessmentID)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		StringBuffer buf = new StringBuffer(256);
		buf
				.append("select assessmentid, scorename, scoretype, textvalue, scoreorder,");
		buf.append("nc_storedassessment_uniqueid,entryid, keyerid, raterid,");
		buf.append(" classification from nc_assessmentdata where ");
		buf.append(prepareBoolean(" isvalidated =", false)).append(" and ");

		try {
			// keyed by the score name
			Map<String, ReconScoreValueInfo> reconMap = new LinkedHashMap<String, ReconScoreValueInfo>();

			String inQuery = prepareInnerQuery(subjectID, assessmentID,
					experimentID, visitID, segmentID);
			buf.append("nc_storedassessment_uniqueid in (").append(inQuery)
					.append(")");
			log.info("getReconciliationDataForAssessment:: query="
					+ buf.toString());

			con = pool.getConnection(ui.getName());
			Statement st = null;
			try {
				st = con.createStatement();
				ResultSet rs = st.executeQuery(buf.toString());
				while (rs.next()) {
					String scoreName = rs.getString(2);
					String scoreType = rs.getString(3);
					String textValue = rs.getString(4);
					int scoreOrder = rs.getInt(5);
					// int storeAsID = rs.getInt(6);
					BigDecimal entryID = rs.getBigDecimal(7);
					BigDecimal keyerID = rs.getBigDecimal(8);
					BigDecimal raterID = rs.getBigDecimal(9);
					BigDecimal classification = rs.getBigDecimal(10);

					ScoreValue scoreValue = new ScoreValue(scoreName,
							textValue, scoreOrder);
					scoreValue.setScoreType(scoreType);
					scoreValue.setClassification(SubjectAssessmentManHelper
							.toInteger(classification));
					scoreValue.setEntryID(SubjectAssessmentManHelper
							.toInteger(entryID));
					scoreValue.setKeyerID(SubjectAssessmentManHelper
							.toInteger(keyerID));
					scoreValue.setRaterID(SubjectAssessmentManHelper
							.toInteger(raterID));
					String key = ReconScoreValueInfo.createKey(scoreName,
							scoreOrder);
					ReconScoreValueInfo rsvi = reconMap.get(key);
					if (rsvi == null) {
						rsvi = new ReconScoreValueInfo(scoreName);
						rsvi.setAssessmentID(assessmentID);
						rsvi.setScoreOrder(scoreOrder);
						reconMap.put(key, rsvi);
					}
					rsvi.addScoreValue(scoreValue);
				}
				rs.close();
			} finally {
				DBUtils.close(st);
			}
			Map<String, List<ReconScoreValueInfo>> reconScoreNameMap = new HashMap<String, List<ReconScoreValueInfo>>();
			for (ReconScoreValueInfo rsvi : reconMap.values()) {
				List<ReconScoreValueInfo> rsviList = reconScoreNameMap.get(rsvi
						.getScoreName());
				if (rsviList == null) {
					rsviList = new LinkedList<ReconScoreValueInfo>();
					reconScoreNameMap.put(rsvi.getScoreName(), rsviList);
				}
				rsviList.add(rsvi);
			}
			// get and populate question information also (if any)
			Map<String, Assessmentitem> questionMap = getQuestionInfoForReconciliation(
					this.theDBID, con, assessmentID);
			for (Map.Entry<String, Assessmentitem> entry : questionMap
					.entrySet()) {
				String scoreName = (String) entry.getKey();
				Assessmentitem ai = (Assessmentitem) entry.getValue();
				List<ReconScoreValueInfo> rsviList = reconScoreNameMap
						.get(scoreName);
				if (rsviList != null) {
					for (ReconScoreValueInfo rsvi : rsviList) {
						rsvi.setLeadingText(ai.getItemleadingtext());
						rsvi.setTrailingText(ai.getItemtrailingtext());
					}
				}
			}

			return reconMap;
		} catch (Exception x) {
			log.error("Error in getReconciliationDataForAssessment", x);
			throw new SubjectAssessmentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void saveReconciledAssessmentData(String dbID, UserInfo ui,
			AssessmentScoreValues asv,
			Map<String, ReconScoreValueInfo> reconMap,
			Map<String, Map<String, String>> dbVarMetaDataMap)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		/** only one storedassessment record for all entries (for double entry) */
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			log.info("in saveReconciledAssessmentData() ");
			StoredassessmentDAO saDAO = DAOFactory
					.createStoredassessmentDAO(dbID);

			Storedassessment criteria = new Storedassessment();
			criteria.setAssessmentid(GenUtils.toBigDecimal(asv
					.getAssessmentID()));
			criteria.setComponentid(GenUtils.toBigDecimal(asv.getVisitID()));
			criteria.setSegmentid(GenUtils.toBigDecimal(asv.getSegmentID()));
			criteria.setNcExperimentUniqueid(GenUtils.toBigDecimal(asv
					.getExperimentID()));
			criteria.setSubjectid(asv.getSubjectID());
			List<Storedassessment> saList = saDAO.find(con, criteria);
			assert (saList.size() == 1);
			Storedassessment sa = saList.get(0);

			int storedAsID = sa.getUniqueid().intValue();
			int raterID = SubjectAssessmentManHelper.findClinicalRaterID(ui,
					dbID, con, asv.getClinicalRater());
			log.info("raterID=" + raterID);
			Databaseuser databaseUser = getDatabaseUser(ui, dbID,
					Constants.USERCLASS_ADMIN);

			// get keyer
			Databaseuser keyer = getKeyer(ui, dbID);

			// keyed by the score type
			Map<String, List<String>> sv2UpdateMap = new HashMap<String, List<String>>(
					5);

			List<ScoreValue> reconSvList = new LinkedList<ScoreValue>();
			int maxEntryID = -1;
			for (ReconScoreValueInfo rsvi : reconMap.values()) {
				if (rsvi.getReconciledValue() == null) {
					List<String> scoreNames = sv2UpdateMap.get(rsvi
							.getScoreType());
					if (scoreNames == null) {
						scoreNames = new LinkedList<String>();
						sv2UpdateMap.put(rsvi.getScoreType(), scoreNames);
					}
					if (!scoreNames.contains(rsvi.getScoreName())) {
						scoreNames.add(rsvi.getScoreName());
					}
				} else {
					maxEntryID = rsvi.getMaxEntryID();
					reconSvList.add(rsvi.getReconciledValue());
				}
			}

			// update the isvalidated fields for first entry of the matching
			// score values
			for (Map.Entry<String, List<String>> entry : sv2UpdateMap
					.entrySet()) {
				String scoreType = (String) entry.getKey();
				List<String> scoreNames = (List<String>) entry.getValue();
				updateValidationFlag(con, storedAsID, asv.getAssessmentID(), 1,
						scoreNames, scoreType, true);
			}

			BigDecimal missClassificationID = SubjectAssessmentManHelper
					.getClassificationID(ui, dbID,
							Constants.MISSING_VALUE_CLASSIFICATION);
			// insert a new entry entry per reconciled values
			BigDecimal entryID = GenUtils.toBigDecimal(maxEntryID + 1);

			SubjectAsmContext context = new SubjectAsmContext(dbID,
					dbVarMetaDataMap, entryID, ui);
			context.setKeyerID(keyer.getUniqueid());
			context.setMissClassificationID(missClassificationID);
			context.setValidated(true);
			context.setAsv(asv);
			context.setRaterID(raterID);

			for (Iterator<ScoreValue> iter = reconSvList.iterator(); iter
					.hasNext();) {
				ScoreValue sv = iter.next();
				AssessmentInfo.ScoreInfo si = asv.findScoreInfo(sv.getName());

				insertScoreValue(context, con, sa, databaseUser, sv, si);

			}
			// now its time set the validated flag for the stored assessment
			// also
			/** @todo proper handling of status field? */
			Storedassessment updateBean = new Storedassessment();
			updateBean.setIsvalidated(new Boolean(true));
			saDAO.update(con, updateBean, criteria);

			con.commit();
		} catch (Exception x) {
			log.error("Error in insertAssessmentValues", x);
			try {
				con.rollback();
			} catch (Exception e) {
				throw new SubjectAssessmentManagementException(x);
			}
			if (x instanceof SubjectAssessmentManagementException) {
				throw (SubjectAssessmentManagementException) x;
			}

			throw new SubjectAssessmentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	protected void updateValidationFlag(Connection con, int storedAsID,
			int assessmentID, int entryID, List<String> scoreNames,
			String scoreType, boolean validated) throws Exception {
		StringBuffer buf = new StringBuffer(200);

		buf.append("update ").append(
				SubjectAssessmentManHelper
						.findAssessmentDataTablename(scoreType));
		buf.append(" set ");
		buf.append(prepareBoolean(" isvalidated =", validated));
		buf.append(" where storedassessmentid = ").append(storedAsID).append(
				" and ");
		buf.append("assessmentid = ").append(assessmentID).append(
				" and entryid =");
		buf.append(entryID).append(" and scorename in (");
		for (Iterator<String> iter = scoreNames.iterator(); iter.hasNext();) {
			String scoreName = iter.next();
			buf.append(prepareString(scoreName));
			if (iter.hasNext()) {
				buf.append(',');
			}
		}
		buf.append(")");
		log.info("updateValidationFlag query:" + buf.toString());
		Statement st = null;
		try {
			st = con.createStatement();
			st.execute(buf.toString());
		} finally {
			DBUtils.close(st);
		}
	}

	protected Map<String, Assessmentitem> getQuestionInfoForReconciliation(
			String dbID, Connection con, int assessmentID) throws Exception {

		Map<String, Assessmentitem> questionMap = new HashMap<String, Assessmentitem>();
		AssessmentitemDAO dao = DAOFactory.createAssessmentitemDAO(dbID);
		Assessmentitem criterion = new Assessmentitem();
		criterion.setAssessmentid(GenUtils.toBigDecimal(assessmentID));
		List<Assessmentitem> aiList = dao.find(con, criterion);
		for (Assessmentitem ai : aiList) {
			questionMap.put(ai.getScorename(), ai);
		}
		return questionMap;
	}

	public List<AssessmentScoreValues> getAssessmentValuesForSubject(
			UserInfo ui, String subjectID, int experimentID, int visitID,
			int segmentID, List<AssessmentInfo> assessments, int theEntryID,
			boolean validated) throws SubjectAssessmentManagementException {

		if (assessments == null || assessments.isEmpty()) {
			return new ArrayList<AssessmentScoreValues>(0);
		}

		Statement st = null;
		StringBuffer buf = new StringBuffer(256);
		buf
				.append("select assessmentid, scorename, scoretype, textvalue, scoreorder, ");
		buf.append("nc_storedassessment_uniqueid,entryid, keyerid, raterid, ");
		buf.append("classification from nc_assessmentdata where ");
		if (theEntryID > 0) {
			buf.append(" entryid = ").append(theEntryID).append(" and ");
		}
		buf.append(prepareBoolean(" isvalidated =", validated)).append(" and ");

		Connection con = null;
		try {
			Map<Integer, String> reasonMap = DBUtils.getMissingValueReasonMap(
					ui, this.theDBID);

			String inQuery = SubjectAssessmentManHelper.prepareInnerQuery(
					subjectID, assessments, experimentID, visitID, segmentID);
			buf.append("nc_storedassessment_uniqueid in (").append(inQuery);
			buf.append(")");

			log
					.info("getAssessmentValuesForSubject -  query="
							+ buf.toString());

			con = pool.getConnection(ui.getName());
			st = con.createStatement();

			int[] as_ids = new int[assessments.size()];

			int idx = 0;
			for (Iterator<AssessmentInfo> iter = assessments.iterator(); iter
					.hasNext();) {
				AssessmentInfo asi = iter.next();
				as_ids[idx++] = asi.getAssessmentID();
			}

			Map<Integer, AssessmentInfo> asiMap = convertToMap(assessments);
			Set<Integer> storedAsIds = new TreeSet<Integer>();
			ResultSet rs = st.executeQuery(buf.toString());

			Map<String, AssessmentScoreValues> asvMap = new LinkedHashMap<String, AssessmentScoreValues>(
					17);
			while (rs.next()) {
				int asID = rs.getInt(1);
				String scoreName = rs.getString(2);
				// String scoreType = rs.getString(3);
				String textValue = rs.getString(4);
				int scoreOrder = rs.getInt(5);
				int storeAsID = rs.getInt(6);
				BigDecimal entryID = rs.getBigDecimal(7);
				BigDecimal keyerID = rs.getBigDecimal(8);
				BigDecimal raterID = rs.getBigDecimal(9);
				BigDecimal classification = rs.getBigDecimal(10);

				storedAsIds.add(new Integer(storeAsID));

				Integer asIDObject = new Integer(asID);
				// FIXME
				String asvKey = getAsvMapKey(asID, storeAsID);
				AssessmentScoreValues asv = asvMap.get(asvKey);
				if (asv == null) {
					AssessmentInfo asi = asiMap.get(asIDObject);

					asv = new AssessmentScoreValues(asi);
					asv.setSubjectID(subjectID);
					asv.setExperimentID(experimentID);
					if (visitID != -1) {
						asv.setVisitID(visitID);
					}
					if (segmentID != -1) {
						asv.setSegmentID(segmentID);
					}
					asv.setStoredAsID(storeAsID);

					if (visitID != -1 && segmentID != -1) {
						java.util.Date timeStamp = getTimeStampForAssessment(
								ui, subjectID, experimentID, visitID,
								segmentID, asi.getAssessmentID());
						asv.setTimeStamp(timeStamp);
					}

					asvMap.put(asvKey, asv);
				}
				/** @todo add score hierarchy */

				ScoreValue scoreValue = new ScoreValue(scoreName, textValue,
						scoreOrder);
				Integer cid = SubjectAssessmentManHelper
						.toInteger(classification);
				scoreValue.setClassification(cid);
				if (cid != null) {
					scoreValue.setReason((String) reasonMap.get(cid));
				}

				scoreValue.setEntryID(SubjectAssessmentManHelper
						.toInteger(entryID));
				scoreValue.setKeyerID(SubjectAssessmentManHelper
						.toInteger(keyerID));
				scoreValue.setRaterID(SubjectAssessmentManHelper
						.toInteger(raterID));

				asv.addScoreValue(scoreValue);

				if (log.isDebugEnabled()) {
					log.debug(rs.getString(2) + ", " + rs.getString(3) + ", "
							+ rs.getString(4));
				}
			} // while

			if (visitID == -1 || segmentID == -1) {
				// set the apropriate visitIDs and segmentIDs for each
				// AssessmentScoreValues object
				List<StoredAssessmentInfo> saiList = getStoredAssessmentInfos(
						con, storedAsIds);
				for (Iterator<StoredAssessmentInfo> iter = saiList.iterator(); iter
						.hasNext();) {
					StoredAssessmentInfo sai = iter.next();
					for (Iterator<AssessmentScoreValues> it = asvMap.values()
							.iterator(); it.hasNext();) {
						AssessmentScoreValues asv = it.next();
						if (asv.getStoredAsID() == sai.uniqueID) {
							asv.setVisitID(sai.visitID);
							asv.setSegmentID(sai.segmentID);
						}
					}
				}
			}

			return new ArrayList<AssessmentScoreValues>(asvMap.values());
		} catch (Exception x) {
			log.error("Error in getAssessmentValuesForSubject", x);
			throw new SubjectAssessmentManagementException(x);
		} finally {
			DBUtils.close(st);
			releaseConnection(con, ui);
		}
	}

	/**
	 * 
	 * @param prefix
	 *            column-name &lt;comparsion-op&gt;
	 * @param value
	 *            boolean database dialect dependent boolean value
	 * @return String
	 */
	abstract protected String prepareBoolean(String prefix, boolean value);

	protected String getAsvMapKey(int asID, int storedAsID) {
		StringBuffer sb = new StringBuffer();
		sb.append(asID).append(':').append(storedAsID);
		return sb.toString();
	}

	protected Databaseuser getDatabaseUser(UserInfo ui, String dbID,
			String userClass) throws Exception {
		ISecurityService securityService = ServiceFactory.getSecurityService();
		IDBCache dbCache = ServiceFactory.getDBCache(dbID);

		Map<String, User> userMap = securityService.getAllUsers(dbID);
		User u = (User) userMap.get(ui.getName());

		Databaseuser databaseUser = dbCache.getDatabaseUser(ui, u.getDbUser()
				.getName(), userClass);
		if (databaseUser == null) {
			log.info("databaseUser was null (from dbCache)");
		} else {
			log.info(databaseUser.toString());
		}
		return databaseUser;
	}

	protected Databaseuser getKeyer(UserInfo ui, String dbID) throws Exception {
		IDBCache dbCache = ServiceFactory.getDBCache(dbID);
		Databaseuser databaseUser = dbCache.getDatabaseUser(ui, ui.getName(),
				Constants.USERCLASS_DATA_PROCESSING);
		if (databaseUser == null) {
			log.info("databaseUser was null (from dbCache) for keyer "
					+ ui.getName());
		} else {
			log.info("keyer=" + databaseUser.toString());
		}
		return databaseUser;
	}

	protected BigDecimal getTableID(String dbID, UserInfo ui, String tableName)
			throws Exception {
		IDBCache dbCache = ServiceFactory.getDBCache(dbID);
		Tableid tid = dbCache.getTableID(ui, tableName);
		return tid.getTableid();
	}

	/**
	 * FIXME use TSQLProcessor?
	 * 
	 * @param subjectID
	 * @param assessmentID
	 * @param experimentID
	 * @param visitID
	 * @param segmentID
	 * @return
	 */
	protected String prepareInnerQuery(String subjectID, int assessmentID,
			int experimentID, int visitID, int segmentID) {
		StringBuffer buf = new StringBuffer(128);
		buf
				.append("select uniqueid from nc_storedassessment where subjectid = ");
		buf.append(prepareString(subjectID));

		if (experimentID >= 0) {
			buf.append(" and nc_experiment_uniqueid = ").append(experimentID);
		}
		if (visitID >= 0) {
			buf.append(" and componentid = ").append(visitID);
		}
		if (segmentID >= 0) {
			buf.append(" and segmentid = ").append(segmentID);

		}
		buf.append(" and assessmentid = ").append(assessmentID);

		return buf.toString();
	}
}
