package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.Timestamp;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheManager;

import clinical.cache.CacheUtils;
import clinical.server.vo.Assessment;
import clinical.server.vo.Humansubject;
import clinical.utils.Assertion;
import clinical.utils.DateTimeUtils;
import clinical.web.Constants;
import clinical.web.IDBUserManService;
import clinical.web.ISubjectAssessmentManagement;
import clinical.web.ISubjectVisitManagement;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBPoolService;
import clinical.web.common.UserInfo;
import clinical.web.common.query.SearchCriteria;
import clinical.web.common.query.SearchPredicate;
import clinical.web.common.query.SearchPredicateList;
import clinical.web.exception.BaseException;
import clinical.web.exception.SubjectAssessmentManagementException;
import clinical.web.vo.AssessmentInfo;
import clinical.web.vo.AssessmentScoreValues;
import clinical.web.vo.ScoreValue;
import clinical.web.vo.Visit;
import clinical.web.vo.AssessmentInfo.ScoreInfo;
import clinical.web.vo.upload.AssessmentType;
import clinical.web.vo.upload.ScoreType;
import clinical.web.vo.upload.VisitInfo;

public class AssessmentEntryServiceImpl extends AbstractServiceImpl implements IAssessmentEntryService {

	public AssessmentEntryServiceImpl(String dbID) throws BaseException {
		super(dbID);
	}

	/* (non-Javadoc)
	 * @see clinical.web.services.IAssessmentEntryService#saveAssessmentMetadata(clinical.web.common.UserInfo, clinical.web.vo.upload.AssessmentType)
	 */
	@Override
	public Assessment saveAssessmentMetadata(UserInfo ui, AssessmentType at)
			throws Exception {
		Connection con = null;
		IDBPoolService dbPoolService = null;
		try {
			ISubjectAssessmentManagement isam = ServiceFactory
					.getSubjectAssessmentManagement(theDBID);
			dbPoolService = ServiceFactory.getPoolService(this.theDBID);
			con = dbPoolService.getConnection(ui.getName());
			con.setAutoCommit(false);
			Assessment assessment = isam.getAssessment(at.getName(), con);

			if (assessment == null) {
				assessment = addAssessment(ui, at, isam);
			}

			return assessment;
		} catch (Throwable t) {
			t.printStackTrace();
			try {
				con.rollback();
			} catch (Exception e) {
				log.error("rollback error", e);
			}
			throw new Exception(t.getMessage());
		} finally {
			if (dbPoolService != null) {
				dbPoolService.releaseConnection(ui.getName(), con);
			}
		}
	}

	/* (non-Javadoc)
	 * @see clinical.web.services.IAssessmentEntryService#saveAssessmentData(clinical.web.common.UserInfo, clinical.web.vo.upload.VisitInfo)
	 */
	@Override
	public void saveAssessmentData(UserInfo ui, VisitInfo visitInfo)
			throws Exception {
		Connection con = null;
		IDBPoolService dbPoolService = null;
		int visitId = 1;
		int startSegId = 1;
		try {
			dbPoolService = ServiceFactory.getPoolService(this.theDBID);
			con = dbPoolService.getConnection(ui.getName());
			con.setAutoCommit(false);

			ISubjectVisitManagement isvm = ServiceFactory
					.getSubjectVisitManagement(this.theDBID);
			List<Visit> visitList = isvm.getAllVisitsForSubject(ui,
					visitInfo.getSubjectID());

			// filter out visits not belonging to the visitInfo's experiment
			for (Iterator<Visit> it = visitList.iterator(); it.hasNext();) {
				Visit v = it.next();
				if (v.getExperimentID() != visitInfo.getExpId()) {
					it.remove();
				}
			}
			Assertion.assertFalse(visitList.isEmpty());
			Visit theVisit = findVisit(visitInfo, visitList);
			Assertion.assertNotNull(theVisit);

			if (!visitInfo.getAssessments().isEmpty()) {
				for (AssessmentType at : visitInfo.getAssessments()) {
					handleAssessment(con, ui, at, visitInfo, visitId,
							startSegId);
				}
			}

		} catch (Throwable t) {
			t.printStackTrace();
			try {
				con.rollback();
			} catch (Exception e) {
				log.error("rollback error", e);
			}
			throw new Exception(t.getMessage());
		} finally {
			if (dbPoolService != null) {
				dbPoolService.releaseConnection(ui.getName(), con);
			}
		}
	}

	protected void handleAssessment(Connection con, UserInfo ui,
			AssessmentType at, VisitInfo visitInfo, int visitId, int segmentId)
			throws Exception {
		ISubjectAssessmentManagement isam = ServiceFactory
				.getSubjectAssessmentManagement(theDBID);

		ISubjectVisitManagement isvm = ServiceFactory
				.getSubjectVisitManagement(theDBID);
		SearchPredicateList spList = new SearchPredicateList();
		SearchPredicate sp = new SearchPredicate("subjectid",
				visitInfo.getSubjectID(), SearchPredicate.EQUAL,
				SearchPredicate.STRING);
		spList.addSearchPredicate(sp, SearchPredicateList.NONE);

		SearchCriteria sc = new SearchCriteria(spList);
		List<Humansubject> matchingSubjects = isvm.getMatchingSubjects(ui, sc);
		Humansubject theHS = matchingSubjects.get(0);

		String asName = at.getName();

		Assessment assessment = isam.getAssessment(asName, con);

		if (assessment == null) {
			assessment = addAssessment(ui, at, isam);
		}
		AssessmentInfo asInfo = prepAssessmentInfo(at, assessment);
		AssessmentScoreValues asv = new AssessmentScoreValues(asInfo);
		asv.setSubjectID(visitInfo.getSubjectID());
		asv.setExperimentID(visitInfo.getExpId());
		asv.setVisitID(visitId);
		asv.setSegmentID(segmentId);
		asv.setTimeStamp(visitInfo.getVisitDate());
		// informant and clinical rater information
		asv.setInformantID(visitInfo.getSubjectID());
		asv.setInformantRelation(Constants.SELF);
		String clinicalRater = ui.getName();
		asv.setClinicalRater(clinicalRater);

		List<AssessmentInfo> aiList = isam.getAssessmentsForSubject(con, ui,
				visitInfo.getSubjectID(), visitInfo.getExpId());
		boolean foundAsEntry = false;
		for (AssessmentInfo ai : aiList) {
			if (ai.getAssessmentID() == asInfo.getAssessmentID()) {
				if (visitId == ai.getVisitID()
						&& segmentId == ai.getSegmentID()) {
					foundAsEntry = true;
					break;
				}
			}
		}

		if (foundAsEntry) {
			throw new Exception("An entry of the assessment '"
					+ asInfo.getName() + "' for visit with ID " + visitId
					+ " and segment(episode) ID " + segmentId
					+ " already exists!");
		}
		IDBUserManService dbusm = ServiceFactory.getDBUserManService(theDBID);
		dbusm.ensureClinicalRater(con, ui, clinicalRater);
		Map<String, Map<String, String>> dbVarMetaDataMap = new HashMap<String, Map<String, String>>(
				3);
		for (clinical.web.vo.upload.ScoreValue sv : at.getSvList()) {
			String value = sv.getValue();
			// special handling for age score
			if (sv.getName().equalsIgnoreCase("age")) {
				int age = DateTimeUtils.calculateAge(theHS.getBirthdate(),
						visitInfo.getVisitDate());
				if (age <= 0) {
					throw new Exception(
							"Age should be greater than 0! Please check subject's birthdate!");
				}
				value = String.valueOf(age);
			}

			// FIXME no date type is supported for upload assessments
			ScoreValue scoreValue = new ScoreValue(sv.getName(), value, 1);
			asv.addScoreValue(scoreValue);
		}
		BigDecimal entryID = new BigDecimal(1);
		isam.insertAssessmentValues(con, ui, theDBID, asv, entryID, true,
				dbVarMetaDataMap);
	}

	protected Assessment addAssessment(UserInfo ui, AssessmentType asInfo,
			ISubjectAssessmentManagement isam)
			throws SubjectAssessmentManagementException {
		AssessmentInfo ai = new AssessmentInfo(-1, asInfo.getName(), null, -1,
				-1);
		if (asInfo.getDescr() != null) {
			ai.setDescription(asInfo.getDescr());
		}
		int i = 1;
		for (ScoreType st : asInfo.getSiList()) {
			ScoreInfo si = new ScoreInfo(st.getName(),
					getScoreType(st.getType()), 1, -1, null, i);
			ai.addScore(si);
			i++;
		}

		// make sure that any assessment cache is dropped.
		String siteID = CacheUtils.getDBID2SiteIDMap().get(this.theDBID);
		Assertion.assertNotNull(siteID);
		String key = siteID + ":asiScores";

		CacheManager man = CacheManager.getInstance();
		Cache cache = man.getCache("remote");
		if (cache != null)
			cache.remove(key);

		return isam.addAssessment(ui, ai);
	}

	protected AssessmentInfo prepAssessmentInfo(AssessmentType asType,
			Assessment assessment) {
		AssessmentInfo asInfo = new AssessmentInfo(assessment.getUniqueid()
				.intValue(), assessment.getName(), null, -1, -1);
		int i = 1;
		for (ScoreType st : asType.getSiList()) {
			ScoreInfo si = new ScoreInfo(st.getName(),
					getScoreType(st.getType()), 1, -1, null, i);
			asInfo.addScore(si);
			i++;
		}
		return asInfo;
	}

	public static String getScoreType(String stType) {
		if (stType.equals(ScoreType.INT)) {
			return "integer";
		} else if (stType.equals(ScoreType.FLOAT)) {
			return stType;
		} else if (stType.equals(ScoreType.STRING)) {
			return "varchar";
		}
		return null;
	}

	public static Visit findVisit(VisitInfo vi, List<Visit> visitList) {
		Timestamp ts = new Timestamp(vi.getVisitDate().getTime());
		for (Visit v : visitList) {
			if (v.getTs().equals(ts)) {
				return v;
			}
		}
		return null;
	}

}
