package clinical.web.services;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.json.JSONArray;
import org.json.JSONObject;

import clinical.server.vo.Deriveddata;
import clinical.server.vo.Expcomponent;
import clinical.server.vo.Experiment;
import clinical.server.vo.Humansubject;
import clinical.server.vo.Jobs;
import clinical.server.vo.Researchgroup;
import clinical.server.vo.Subjexperiment;
import clinical.utils.Assertion;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.utils.Profiler;
import clinical.web.CBFBIRNConstants;
import clinical.web.ICBFProcReportQueryService;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.common.vo.CBFProcessReportRec;
import clinical.web.common.vo.CBFProcessReportRec.QualityMeasureType;
import clinical.web.common.vo.CBFProcessReportRecListWrapper;
import clinical.web.exception.BaseException;
import clinical.web.helpers.CBFBirnHelper;
import clinical.web.scheduler.JobInfo;
import clinical.web.vo.JobProvenanceInfo;
import clinical.web.vo.JobProvenanceInfo.JobProvenanceParamInfo;
import clinical.web.vo.JobProvenanceInfo.JobProvenanceParamType;
import clinical.web.vo.upload.SegmentInfo;
import clinical.web.vo.upload.SeriesHeaderInfo;
import clinical.web.vo.upload.VisitInfo;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: CBFProcReportQueryService.java 569 2012-03-21 01:01:36Z
 *          bozyurt$
 */

public final class CBFProcReportQueryService extends AbstractServiceImpl
		implements ICBFProcReportQueryService {
	static Profiler profiler;

	static {
		profiler = Profiler.getInstance("QO");
		profiler.setProfiler(true);
	}

	public CBFProcReportQueryService(String dbID) throws BaseException {
		super(dbID);
	}

	public List<Deriveddata> getDerivedDataForJob(UserInfo ui, String jobID)
			throws Exception {
		Connection con = null;
		try {
			con = this.pool.getConnection(ui.getName());

			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			StringBuilder sb = new StringBuilder(128);
			sb.append("select a.* from Deriveddata as a, ");
			sb.append("Jobs as b, VisitJob as c where ");
			sb.append("c.jobUniqueId = b.uniqued and a.subjectid = c.subjectid ");
			sb.append("and a.ncExperimentUniqueid = c.expId and ");
			sb.append("a.componentid = c.componentId and ");
			sb.append("b.jobid = '").append(jobID).append("'");

			List<?> results = tsp.executeQuery(con, sb.toString());
			List<Deriveddata> ddList = new ArrayList<Deriveddata>(
					results.size());
			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				Deriveddata dd = (Deriveddata) it.next();
				ddList.add(dd);
			}
			return ddList;
		} finally {
			releaseConnection(con, ui);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.ICBFProcReportQueryService#getCBFProcessRecords
	 * (clinical.web.common.UserInfo, java.util.List)
	 */
	public CBFProcessReportRecListWrapper getCBFProcessRecords(UserInfo ui,
			List<Experiment> allowedExps) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			CBFProcessReportRecListWrapper wrapper = getCBFProcessRecords(con,
					allowedExps);
			con.commit();
			return wrapper;
		} finally {
			releaseConnection(con, ui);
		}
	}

	private List<Jobs> getJobs(Connection con, Set<Integer> allowedExpIdSet)
			throws Exception {
		TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
		StringBuilder qBuf = new StringBuilder(128);
		if (allowedExpIdSet.isEmpty()) {
			qBuf.append("select a.* from Jobs as a where ");
			qBuf.append("a.jobtype = '").append(
					CBFBIRNConstants.CBF_PROCESSING_WORKFLOW);
			qBuf.append("' and a.jobstatus <> '").append(JobInfo.CANCELED)
					.append("'");
		} else {
			qBuf.append("select a.* from Jobs as a, VisitJob as b ");
			qBuf.append("where a.uniqueid = b.jobUniqueId and ");
			qBuf.append("a.jobtype = '").append(
					CBFBIRNConstants.CBF_PROCESSING_WORKFLOW);
			qBuf.append("' and a.jobstatus <> '").append(JobInfo.CANCELED)
					.append("'");
			qBuf.append(" and b.expId in (");
			boolean first = true;
			for (Integer expId : allowedExpIdSet) {
				if (!first) {
					qBuf.append(',');
				}
				qBuf.append(expId);
				first = false;
			}
			qBuf.append(')');
		}

		List<?> results = tsp.executeQuery(con, qBuf.toString());
		List<Jobs> jobList = new ArrayList<Jobs>(results.size());
		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			Jobs job = (Jobs) it.next();
			jobList.add(job);
		}
		return jobList;
	}

	private CBFProcessReportRecListWrapper getCBFProcessRecords(
			final Connection con, final List<Experiment> allowedExps)
			throws Exception {

		profiler.entryPoint("getCBFProcessRecords");
		Set<Integer> allowedExpIdSet = new HashSet<Integer>(17);
		for (Experiment exp : allowedExps) {
			allowedExpIdSet.add(exp.getUniqueid().intValue());
		}

		profiler.entryPoint("getJobs");
		List<Jobs> jobList = getJobs(con, allowedExpIdSet);
		profiler.exitPoint("getJobs");

		if (jobList.isEmpty()) {
			return new CBFProcessReportRecListWrapper(
					new ArrayList<CBFProcessReportRec>(0));
		}

		// remove any job record with status REMOVED and has no jobsize implying
		// no output
		for (Iterator<Jobs> it = jobList.iterator(); it.hasNext();) {
			Jobs job = it.next();
			String jobStatus = job.getJobstatus();
			if ((jobStatus.equalsIgnoreCase(JobInfo.REMOVED) && job
					.getJobsize() == null)
					|| jobStatus.equalsIgnoreCase(JobInfo.FINISHED_WITH_ERR)
					|| job.getJobsize() == null) {
				it.remove();
			}
		}

		IJobProvenanceService jps = ServiceFactory
				.getJobProvenanceService(theDBID);

		profiler.entryPoint("filterJobs");

		List<Jobs> filteredJobs = new ArrayList<Jobs>(jobList.size());
		Map<Jobs, VisitInfo> jobs2VisitMap = new HashMap<Jobs, VisitInfo>();
		Set<String> subjectIdSet = new HashSet<String>();
		Map<String, Date> latestJobMap = new HashMap<String, Date>();
		for (Jobs job : jobList) {
			// skip canceled jobs;
			if (job.getJobstatus().equalsIgnoreCase(JobInfo.CANCELED)) {
				continue;
			}
			JSONObject js = new JSONObject(job.getJobContext());
			JSONArray jsonArr = js.getJSONArray("viList");
			js = jsonArr.getJSONObject(0);
			profiler.entryPoint("vi.initializeFromJSON");
			VisitInfo vi = VisitInfo.initializeFromJSON(js);
			profiler.exitPoint("vi.initializeFromJSON");

			if (!allowedExpIdSet.isEmpty()) {
				if (allowedExpIdSet.contains(vi.getExpId())) {
					handleRetained(filteredJobs, jobs2VisitMap, subjectIdSet,
							latestJobMap, job, vi);
				}

			} else {
				handleRetained(filteredJobs, jobs2VisitMap, subjectIdSet,
						latestJobMap, job, vi);
			}
		}

		profiler.exitPoint("filterJobs");

		// no matching subjects no result
		if (subjectIdSet.isEmpty()) {
			return new CBFProcessReportRecListWrapper(
					new ArrayList<CBFProcessReportRec>(0));
		}

		jobList = filteredJobs;

		profiler.entryPoint("findJobProvenances");
		List<JobProvenanceInfo> jpiList = jps.findJobProvenances(con, jobList);
		profiler.exitPoint("findJobProvenances");

		Map<BigDecimal, JobProvenanceInfo> jpiMap = new HashMap<BigDecimal, JobProvenanceInfo>();
		for (JobProvenanceInfo jpi : jpiList) {
			jpiMap.put(jpi.getJob().getUniqueid(), jpi);
		}

		filteredJobs = new ArrayList<Jobs>(jobList.size());

		// now leave only the latest jobs and jobs with provenance for each
		// subject visit
		Map<String, Jobs> visitLatestJobMap = new HashMap<String, Jobs>();
		for (Jobs job : jobList) {
			JobProvenanceInfo jpi = jpiMap.get(job.getUniqueid());
			if (jpi != null) {
				filteredJobs.add(job);
				continue;
			}
			VisitInfo vi = jobs2VisitMap.get(job);
			String visitKey = getVisitKey(vi);
			Jobs latestJob = visitLatestJobMap.get(visitKey);
			if (latestJob == null) {
				visitLatestJobMap.put(visitKey, job);
			} else {
				if (latestJob.getJobstartdate().before(job.getJobstartdate())) {
					jobs2VisitMap.remove(latestJob);
					visitLatestJobMap.put(visitKey, job);
				} else {
					jobs2VisitMap.remove(job);
				}
			}
		}

		for (Jobs job : visitLatestJobMap.values()) {
			filteredJobs.add(job);
		}

		jobList = null;
		visitLatestJobMap.clear();

		log.info("filteredJobs.size =" + filteredJobs.size());

		profiler.entryPoint("getConditions4Subjects");
		Map<String, SubjectRecord> sid2CondMap = getConditions4Subjects(con,
				subjectIdSet);
		profiler.exitPoint("getConditions4Subjects");

		Map<Integer, String> expId2NameMap = new HashMap<Integer, String>(17);
		for (Experiment exp : allowedExps) {
			expId2NameMap.put(exp.getUniqueid().intValue(), exp.getName());
		}

		List<CBFProcessReportRec> recList = new ArrayList<CBFProcessReportRec>(
				filteredJobs.size());

		profiler.entryPoint("getVisitNames");
		Map<String, VisitInfoWrapper> viwMap = getVisitNames(con, filteredJobs,
				jobs2VisitMap, allowedExpIdSet);
		profiler.exitPoint("getVisitNames");

		profiler.entryPoint("prepCBFProcessRecords1");
		for (Jobs job : filteredJobs) {
			VisitInfo vi = jobs2VisitMap.get(job);
			String key = vi.getSubjectID() + ":" + vi.getExpId();
			SubjectRecord sr = sid2CondMap.get(key);
			String viKey = buildVisitKey(vi.getSubjectID(), vi.getExpId(),
					vi.getVisitId());
			VisitInfoWrapper viw = viwMap.get(viKey);
			String visitName = viw != null ? viw.getVisitName() : null;

			profiler.entryPoint("prepCBFProcessRecords4Job");
			List<CBFProcessReportRec> records4Job = prepCBFProcessRecords4Job(
					job, vi, sr, expId2NameMap, jpiMap, visitName);
			profiler.exitPoint("prepCBFProcessRecords4Job");

			if (records4Job != null) {
				recList.addAll(records4Job);
			}
		}
		profiler.exitPoint("prepCBFProcessRecords1");

		// sort by project name - subject and visit ID
		Collections.sort(recList, new Comparator<CBFProcessReportRec>() {
			public int compare(CBFProcessReportRec o1, CBFProcessReportRec o2) {
				int diff = o1.getProjectName().compareTo(o2.getProjectName());
				if (diff == 0) {
					diff = o1.getSubjectID().compareTo(o2.getSubjectID());
					if (diff == 0) {
						diff = o1.getVisitId().compareTo(o2.getVisitId());
					}
				}
				return diff;
			}
		});
		CBFProcessReportRecListWrapper wrapper = new CBFProcessReportRecListWrapper(
				recList);
		// prepare data structures for comparison table
		profiler.entryPoint("populateOthers");
		wrapper.populateOthers();
		profiler.exitPoint("populateOthers");
		profiler.exitPoint("getCBFProcessRecords");

		profiler.showStats();

		return wrapper;
	}

	protected Map<String, VisitInfoWrapper> getVisitNames(final Connection con,
			List<Jobs> filteredJobs, Map<Jobs, VisitInfo> jobs2VisitMap,
			Set<Integer> allowedExpIdSet) throws Exception {
		Set<String> subjectIDSet = new HashSet<String>();
		Map<String, VisitInfo> viMap = new HashMap<String, VisitInfo>();

		for (Jobs job : filteredJobs) {
			VisitInfo vi = jobs2VisitMap.get(job);
			subjectIDSet.add(vi.getSubjectID());
			String key = buildVisitKey(vi.getSubjectID(), vi.getExpId(),
					vi.getVisitId());
			viMap.put(key, vi);
		}

		TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
		StringBuilder qBuf = new StringBuilder(128);
		qBuf.append("select a.* from Expcomponent as a where ");
		qBuf.append("a.subjectid in (");
		boolean first = true;
		for (String subjectID : subjectIDSet) {
			if (!first) {
				qBuf.append(',');
			} else {
				first = false;
			}
			qBuf.append('\'').append(subjectID).append('\'');
		}
		qBuf.append(')');

		List<?> results = tsp.executeQuery(con, qBuf.toString());
		Map<String, VisitInfoWrapper> viwMap = new HashMap<String, CBFProcReportQueryService.VisitInfoWrapper>();

		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			Expcomponent visit = (Expcomponent) it.next();
			String key = buildVisitKey(visit.getSubjectid(), visit
					.getNcExperimentUniqueid().intValue(), visit
					.getComponentid().intValue());
			VisitInfo vi = viMap.get(key);
			if (vi != null) {
				viwMap.put(key, new VisitInfoWrapper(vi, visit.getName()));
			}
		}
		return viwMap;
	}

	public static String buildVisitKey(String subjectID, int expID, int visitID) {
		StringBuilder sb = new StringBuilder();
		sb.append(subjectID).append(':').append(expID).append(':')
				.append(visitID);
		return sb.toString();
	}

	private static final class PngFileFilter implements FileFilter {
		@Override
		public boolean accept(File pathname) {
			return pathname.getName().endsWith(".png");
		}
	}

	public static class VisitInfoWrapper {
		final VisitInfo vi;
		final String visitName;

		public VisitInfoWrapper(VisitInfo vi, String visitName) {
			super();
			this.vi = vi;
			this.visitName = visitName;
		}

		public VisitInfo getVi() {
			return vi;
		}

		public String getVisitName() {
			return visitName;
		}
	}// ;

	protected List<CBFProcessReportRec> prepCBFProcessRecords4Job(Jobs job,
			VisitInfo vi, SubjectRecord sr, Map<Integer, String> expId2NameMap,
			Map<BigDecimal, JobProvenanceInfo> jpiMap, String visitName)
			throws Exception {
		JobProvenanceInfo jpi = jpiMap.get(job.getUniqueid());
		String expName = expId2NameMap.get(vi.getExpId());
		Assertion.assertNotNull(expName);
		List<CBFProcessReportRec> list = new ArrayList<CBFProcessReportRec>(1);
		for (SegmentInfo si : vi.getSiList()) {
			if (isCBFProcessable(si)) {
				if (log.isDebugEnabled()) {
					log.debug("prepProcessRecord " + si);
				}
				CBFProcessReportRec rec = prepProcessRecord(job, vi, si, sr,
						expName, jpi, visitName);
				if (rec == null) {
					return null;
				} else {
					list.add(rec);
				}
			}
		}

		return list;
	}

	protected List<CBFProcessReportRec> prepCBFProcessRecords4Job(Jobs job,
			VisitInfo vi, SubjectRecord sr, Map<Integer, String> expId2NameMap)
			throws Exception {
		String expName = expId2NameMap.get(vi.getExpId());
		Assertion.assertNotNull(expName);
		List<CBFProcessReportRec> list = new ArrayList<CBFProcessReportRec>(1);
		for (SegmentInfo si : vi.getSiList()) {
			if (isCBFProcessable(si)) {
				CBFProcessReportRec rec = prepProcessRecord(job, vi, si, sr,
						expName);
				if (rec == null) {
					return null;
				} else {
					list.add(rec);
				}
			}
		}
		return list;
	}

	CBFProcessReportRec prepProcessRecord(Jobs job, VisitInfo vi,
			SegmentInfo si, SubjectRecord sr, String expName,
			JobProvenanceInfo jpi, String visitName) {
		profiler.entryPoint("prepDerivedDirContents");
		DerivedDataDirContents contents = prepDerivedDirContents(si, jpi);
		profiler.exitPoint("prepDerivedDirContents");

		if (contents == null) {
			log.info("prepProcessRecord:: contents was null");
			return null;
		}
		CBFProcessReportRec rec = new CBFProcessReportRec();
		rec.setSubjectID(vi.getSubjectID());
		rec.setProjectName(expName);
		if (jpi != null) {
			String lastPart = new File(jpi.getDataURI()).getName();
			rec.setTag(si.getProtocolId() + " (" + lastPart + ")");

			// add provenance info to the display record if any
			if (jpi.getNumOfParams() > 0) {
				for (JobProvenanceParamInfo pi : jpi.getParams()) {
					// skip any quality measure comments/reasons
					String paramName = pi.getName();
					if (paramName
							.equals(CBFBIRNConstants.CBF_PROC_QUALITY_REASONS_NAME)
							|| paramName
									.equals(CBFBIRNConstants.CBF_PROC_QUALITY_COMMENT_NAME)
							|| paramName
									.equals(CBFBIRNConstants.SEG_QUALITY_REASONS_NAME)) {
						continue;
					}
					if (paramName
							.equals(CBFBIRNConstants.CBF_PROC_QUALITY_RATING_NAME)) {
						if (rec.getQmType() == QualityMeasureType.SEGMENTATION) {
							rec.setQmType(QualityMeasureType.CBF_AND_SEG);
						} else {
							rec.setQmType(QualityMeasureType.CBF);
						}
					} else if (paramName
							.equals(CBFBIRNConstants.SEG_QUALITY_RATING_NAME)) {
						if (rec.getQmType() == QualityMeasureType.CBF) {
							rec.setQmType(QualityMeasureType.CBF_AND_SEG);
						} else {
							rec.setQmType(QualityMeasureType.SEGMENTATION);
						}
					}

					JobProvenanceParamType type = pi.getTypeFromValue();
					CBFProcessReportRec.Param p = null;
					if (type == JobProvenanceParamType.STRING) {
						p = new CBFProcessReportRec.Param(pi.getName(),
								pi.getValue(),
								CBFProcessReportRec.ParamType.STRING);
					} else if (type == JobProvenanceParamType.BOOL) {
						p = new CBFProcessReportRec.Param(pi.getName(),
								pi.getValue(),
								CBFProcessReportRec.ParamType.BOOL);
					} else if (type == JobProvenanceParamType.DATE) {
						p = new CBFProcessReportRec.Param(pi.getName(),
								pi.getValue(),
								CBFProcessReportRec.ParamType.DATE);
					} else {
						p = new CBFProcessReportRec.Param(pi.getName(),
								pi.getValue(),
								CBFProcessReportRec.ParamType.NUMBER);
					}
					if (p != null) {
						rec.addProvenanceParam(p);
					}
				}
			}

		} else {
			rec.setTag(si.getProtocolId());
		}

		rec.setJobID(job.getJobid());
		rec.setJobUniqueID(job.getUniqueid().intValue());

		rec.setVisitDate(vi.getVisitDate());
		rec.setVisitId(String.valueOf(vi.getVisitId()));
		rec.setCondition(sr.condition);
		rec.setProcessed(contents.matFile != null);
		rec.setReportContent(contents.reportContent);
		for (String imgFile : contents.summaryImages) {
			File f = new File(imgFile);
			String name = f.getName();
			if (name.equals(CBFBIRNConstants.CBFHIST_IMG)) {
				rec.setHistogramImagePath(imgFile);
			} else if (name.equals(CBFBIRNConstants.CBFMAP_IMG)) {
				rec.setCbfImagePath(imgFile);
			} else if (name.equals(CBFBIRNConstants.MOTION_IMG)) {
				rec.setMotionImagePath(imgFile);
			}
		}
		for (String imgFile : contents.additionalImages) {
			String name = new File(imgFile).getName();
			if (name.equals(CBFBIRNConstants.CSF_MAP_PV_IMG)) {
				rec.setCsfMapPVImagePath(imgFile);
			} else if (name.equals(CBFBIRNConstants.CSF_MAP_TOP5_IMG)) {
				rec.setCsfMapTop5ImagePath(imgFile);
			} else if (name.equals(CBFBIRNConstants.CSF_ROI_PV_IMG)) {
				rec.setCsfRoiPVImagePath(imgFile);
			} else if (name.equals(CBFBIRNConstants.CSF_ROI_TOP5_IMG)) {
				rec.setCsfRoiTop5ImagePath(imgFile);
			}
		}
		for (String imgFile : contents.segmentationImages) {
			File f = new File(imgFile);
			if (f.getName().equals(CBFBIRNConstants.CSF_MASKS_IMG)) {
				rec.setCsfMasksImagePath(imgFile);
			} else if (f.getName().equals(CBFBIRNConstants.CSF_SLICES_IMG)) {
				rec.setCsfSlicesImagePath(imgFile);
			}
		}
		rec.setLocalID(sr.localid);

		if (!contents.summaryImages.isEmpty()) {
			extractGMValue(contents, rec);
		}
		if (visitName != null) {
			rec.setVisitName(visitName);
		}
		return rec;
	}

	CBFProcessReportRec prepProcessRecordForPostProcess(Jobs job, VisitInfo vi,
			SegmentInfo si, String expName) {
		CBFProcessReportRec rec = new CBFProcessReportRec();
		rec.setSubjectID(vi.getSubjectID());
		rec.setProjectName(expName);
		rec.setTag(si.getProtocolId());
		rec.setVisitDate(vi.getVisitDate());
		rec.setVisitId(String.valueOf(vi.getVisitId()));
		rec.setJobID(job.getJobid());
		return rec;
	}

	CBFProcessReportRec prepProcessRecord(Jobs job, VisitInfo vi,
			SegmentInfo si, SubjectRecord sr, String expName) {
		DerivedDataDirContents contents = findSummaryImages(si);
		if (contents == null) {
			return null;
		}
		CBFProcessReportRec rec = new CBFProcessReportRec();
		rec.setSubjectID(vi.getSubjectID());
		rec.setProjectName(expName);
		rec.setTag(si.getProtocolId());
		rec.setVisitDate(vi.getVisitDate());
		rec.setVisitId(String.valueOf(vi.getVisitId()));
		rec.setCondition(sr.condition);
		rec.setProcessed(contents.matFile != null);
		for (String imgFile : contents.summaryImages) {
			File f = new File(imgFile);
			if (f.getName().equals(CBFBIRNConstants.CBFHIST_IMG)) {
				rec.setHistogramImagePath(imgFile);
			} else if (f.getName().equals(CBFBIRNConstants.CBFMAP_IMG)) {
				rec.setCbfImagePath(imgFile);
			}
		}
		rec.setLocalID(sr.localid);
		if (!contents.summaryImages.isEmpty()) {
			extractGMValue(contents, rec);
		}
		return rec;
	}

	protected void extractGMValue(DerivedDataDirContents contents,
			CBFProcessReportRec rec) {
		File dir = new File(contents.summaryImages.get(0)).getParentFile();
		File gmFile = new File(dir, "gm.txt");
		if (gmFile.exists()) {
			String content = null;
			try {
				content = FileUtils.loadTextFile(gmFile.getAbsolutePath());
				if (content != null) {
					double value = GenUtils.toDouble(content.trim(),
							Double.NEGATIVE_INFINITY);
					if (value != Double.NEGATIVE_INFINITY) {
						rec.setGmCBF(value);
					}
				}
			} catch (IOException e) {
				log.warn("", e);
			}
		}
	}

	enum TriCondition {
		UNKNOWN, YES, NO
	};

	DerivedDataDirContents prepDerivedDirContents(SegmentInfo si,
			JobProvenanceInfo jpi) {
		List<String> images = si.getShi().getImages();
		String anImg = images.get(0);
		String[] parts = anImg.split("\\/");
		int visitLocIdx = -1;
		for (int i = 0; i < parts.length; i++) {
			if (parts[i].startsWith("Visit_")) {
				visitLocIdx = i;
				break;
			}
		}
		if (visitLocIdx == -1) {
			log.info("no visitLocIdx: " + anImg);
			return null;
		}
		StringBuilder sb = new StringBuilder(128);

		for (int i = 0; i <= visitLocIdx; i++) {
			sb.append('/').append(parts[i]);
		}
		File dir = new File(sb.toString());
		if (!dir.exists()) {
			log.info("prepDerivedDirContents: dir does not exists" + dir);
			return null;
		}

		TriCondition automated = TriCondition.UNKNOWN;
		File rootDerivedDir = null;
		if (jpi != null) {
			rootDerivedDir = new File(jpi.getDataURI());
			JobProvenanceParamInfo jpiParam = jpi.findParam("automated");
			if (jpiParam != null && jpiParam.getValue().equals("yes")) {
				automated = TriCondition.YES;
			} else {
				automated = TriCondition.NO;
			}
		} else {
			rootDerivedDir = new File(dir, CBFBIRNConstants.DERIVED);
		}
		// check if there are cbf summary images in the derived directory
		DerivedDataDirContents contents = new DerivedDataDirContents();

		profiler.entryPoint("findDerivedDataDir");
		File derivedDir = findDerivedDataDir(dir, si, rootDerivedDir, parts);
		profiler.exitPoint("findDerivedDataDir");

		if (derivedDir == null) {
			log.info("rootDerivedDir:" + rootDerivedDir + " parts:" + parts);
		}
		if (derivedDir != null) {
			File[] files = derivedDir.listFiles();
			for (File f : files) {
				if (f.isDirectory()
						&& (automated == TriCondition.UNKNOWN || automated == TriCondition.YES)) {

					if (f.getName().equals(CBFBIRNConstants.INTERMEDIATE)) {
						File[] intFiles = f.listFiles(new PngFileFilter());
						for (File intFile : intFiles) {
							String name = intFile.getName();
							if (name.equals(CBFBIRNConstants.CSF_MASKS_IMG)
									|| name.equals(CBFBIRNConstants.CSF_SLICES_IMG)) {
								contents.segmentationImages.add(intFile
										.getAbsolutePath());
							} else if (name
									.equals(CBFBIRNConstants.CSF_MAP_PV_IMG)
									|| name.equals(CBFBIRNConstants.CSF_MAP_TOP5_IMG)
									|| name.equals(CBFBIRNConstants.CSF_ROI_PV_IMG)
									|| name.equals(CBFBIRNConstants.CSF_ROI_TOP5_IMG)) {
								contents.additionalImages.add(intFile
										.getAbsolutePath());
							}
						}
					}
				} else if (f.isDirectory()
						&& f.getName().equals(CBFBIRNConstants.INTERMEDIATE)) {
					File[] intFiles = f.listFiles(new PngFileFilter());
					for (File intFile : intFiles) {
						String name = intFile.getName();
						if (name.equals(CBFBIRNConstants.CSF_MAP_PV_IMG)
								|| name.equals(CBFBIRNConstants.CSF_MAP_TOP5_IMG)
								|| name.equals(CBFBIRNConstants.CSF_ROI_PV_IMG)
								|| name.equals(CBFBIRNConstants.CSF_ROI_TOP5_IMG)) {
							contents.additionalImages.add(intFile
									.getAbsolutePath());
						}
					}
				} else {
					String name = f.getName();
					if (name.equals(CBFBIRNConstants.CBFHIST_IMG)
							|| name.equals(CBFBIRNConstants.CBFMAP_IMG)
							|| name.equals(CBFBIRNConstants.MOTION_IMG)) {
						contents.summaryImages.add(f.getAbsolutePath());
					} else if (name
							.equals(CBFBIRNConstants.FMRI_STRUCT_MAT_FILE)) {
						contents.matFile = f.getAbsolutePath();
					} else if (name.equals(CBFBIRNConstants.CBF_SUMMARY_FILE)) {
						try {
							contents.reportContent = FileUtils.loadTextFile(f
									.getAbsolutePath());
						} catch (IOException e) {
							log.error(e);
							contents.reportContent = null;
						}
					} else if ((automated == TriCondition.UNKNOWN || automated == TriCondition.YES)) {
						// for Siemens processing putting segmentation images in
						// wrong place bug workaround (The bug is fixed
						// on 2/7/2012)
						if (name.equals(CBFBIRNConstants.CSF_MASKS_IMG)
								|| name.equals(CBFBIRNConstants.CSF_SLICES_IMG)) {
							contents.segmentationImages
									.add(f.getAbsolutePath());
						}
					}
				}
			}
		}
		return contents;
	}

	/**
	 * 
	 * @param si
	 * @return If return value is null that means there is not even raw data.
	 */
	DerivedDataDirContents findSummaryImages(SegmentInfo si) {
		List<String> images = si.getShi().getImages();
		String anImg = images.get(0);
		String[] parts = anImg.split("\\/");
		int visitLocIdx = -1;
		for (int i = 0; i < parts.length; i++) {
			if (parts[i].startsWith("Visit_")) {
				visitLocIdx = i;
				break;
			}
		}
		if (visitLocIdx == -1) {
			return null;
		}
		StringBuilder sb = new StringBuilder(128);

		for (int i = 0; i <= visitLocIdx; i++) {
			sb.append('/').append(parts[i]);
		}
		File dir = new File(sb.toString());
		if (dir.exists()) {
			// check if there are cbf summary images in the derived directory
			DerivedDataDirContents contents = new DerivedDataDirContents();
			File derivedDir = findDerivedDataDir(dir, si, parts);
			if (derivedDir != null) {
				File[] files = derivedDir.listFiles();
				for (File f : files) {
					if (!f.isFile()) {
						continue;
					}
					if (f.getName().equals("cbfhist.png")
							|| f.getName().equals("cbfmap.png")) {
						contents.summaryImages.add(f.getAbsolutePath());
					} else if (f.getName().equals("fmri_struct.mat")) {
						contents.matFile = f.getAbsolutePath();
					}
				}
			}

			return contents;
		}
		return null;
	}

	public static class DerivedDataDirContents {
		List<String> summaryImages = new ArrayList<String>(2);
		List<String> segmentationImages = new ArrayList<String>(2);
		List<String> additionalImages = new ArrayList<String>(4);
		String matFile;
		String reportContent;

	}// ;

	File findDerivedDataDir(File dir, SegmentInfo si, File derivedDir,
			String[] parts) {
		String imageType = si.getShi().getImageType();
		if (!derivedDir.isDirectory()) {
			return null;
		}
		File[] files = derivedDir.listFiles();
		List<File> dirs = new ArrayList<File>(files.length);
		for (File file : files) {
			if (file.isDirectory()) {
				dirs.add(file);
			}
		}
		if (dirs.isEmpty()) {
			return null;
		}

		if (imageType.equals(SeriesHeaderInfo.PFILE)) {
			String pfile = parts[parts.length - 1];
			pfile = pfile.replaceFirst("\\.[^\\.]+$", "");
			for (File dd : dirs) {
				if (dd.getName().indexOf(pfile) != -1) {
					return dd;
				}
			}
		} else if (imageType.equals(SeriesHeaderInfo.AFNI)) {
			String brikFile = parts[parts.length - 1];
			brikFile = brikFile.replaceFirst("\\.\\w+$", "");
			brikFile = brikFile.replaceFirst("(\\+\\w+)+$", "");
			for (File dd : dirs) {
				if (dd.getName().indexOf(brikFile) != -1) {
					return dd;
				}
			}
		} else if (imageType.equals(SeriesHeaderInfo.DICOM)) {
			// for Siemens Data
			String dicomDirname = parts[parts.length - 1];

			for (File dd : dirs) {
				if (dd.getName().indexOf(dicomDirname) != -1) {
					return dd;
				}
			}
		} else if (imageType.equals(SeriesHeaderInfo.ANALYZE)) {
			String analyzeFile = parts[parts.length - 1];
			analyzeFile = analyzeFile.replaceFirst("\\.\\w+$", "");
			for (File dd : dirs) {
				if (dd.getName().indexOf(analyzeFile) != -1) {
					return dd;
				}
			}
		}
		return null;
	}

	File findDerivedDataDir(File dir, SegmentInfo si, String[] parts) {
		String imageType = si.getShi().getImageType();
		File derivedDir = new File(dir, CBFBIRNConstants.DERIVED);
		if (!derivedDir.isDirectory()) {
			return null;
		}
		File[] files = derivedDir.listFiles();
		List<File> dirs = new ArrayList<File>(files.length);
		for (File file : files) {
			if (file.isDirectory()) {
				dirs.add(file);
			}
		}
		if (dirs.isEmpty()) {
			return null;
		}

		if (imageType.equals(SeriesHeaderInfo.PFILE)) {
			String pfile = parts[parts.length - 1];
			pfile = pfile.replaceFirst("\\.[^\\.]+$", "");
			for (File dd : dirs) {
				if (dd.getName().indexOf(pfile) != -1) {
					return dd;
				}
			}
		} else if (imageType.equals(SeriesHeaderInfo.AFNI)) {
			String brikFile = parts[parts.length - 1];
			brikFile = brikFile.replaceFirst("\\.\\w+$", "");
			brikFile = brikFile.replaceFirst("(\\+\\w+)+$", "");
			for (File dd : dirs) {
				if (dd.getName().indexOf(brikFile) != -1) {
					return dd;
				}
			}
		} else if (imageType.equals(SeriesHeaderInfo.DICOM)) {
			// for Siemens Data
			String dicomDirname = parts[parts.length - 1];

			for (File dd : dirs) {
				if (dd.getName().indexOf(dicomDirname) != -1) {
					return dd;
				}
			}
		}
		return null;
	}

	public static boolean isCBFProcessable(SegmentInfo si) {
		/*
		 * not valid for Siemens data and for the time being unnecessary if
		 * (si.getShi().getImageType().equals(SeriesHeaderInfo.DICOM)) { return
		 * false; }
		 */
		String protocolID = si.getProtocolId();
		return CBFBirnHelper.isCBFProcessable(protocolID);
		/*
		 * return (!protocolID.equals(ANATOMICAL) && !protocolID.equals(CSF) &&
		 * !protocolID.equals(MINCON) && !protocolID.equals(FIELD_MAP));
		 */
	}

	protected void handleRetained(List<Jobs> filteredJobs,
			Map<Jobs, VisitInfo> jobs2VisitMap, Set<String> subjectIdSet,
			Map<String, Date> latestJobMap, Jobs job, VisitInfo vi) {
		jobs2VisitMap.put(job, vi);
		subjectIdSet.add(vi.getSubjectID());
		filteredJobs.add(job);
		String visitKey = getVisitKey(vi);
		Date jobStartDate = latestJobMap.get(visitKey);
		if (jobStartDate == null) {
			latestJobMap.put(visitKey, job.getJobstartdate());
		} else {
			if (job.getJobstartdate().after(jobStartDate)) {
				latestJobMap.put(visitKey, job.getJobstartdate());
			}
		}
	}

	protected Map<String, SubjectRecord> getConditions4Subjects(Connection con,
			Set<String> subjectIdSet) throws Exception {
		Map<String, SubjectRecord> sid2CondMap = new HashMap<String, SubjectRecord>();
		TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
		StringBuilder qBuf = new StringBuilder(128);

		qBuf.append("select a.subjectid, a.ncExperimentUniqueid, b.name, ");
		qBuf.append("c.localid ");
		qBuf.append("from Subjexperiment as a, Researchgroup as b, ");
		qBuf.append("Humansubject as c where ");
		qBuf.append("a.ncResearchgroupUniqueid = b.uniqueid and ");
		qBuf.append("a.subjectid = c.subjectid and a.subjectid in (");
		for (Iterator<String> it = subjectIdSet.iterator(); it.hasNext();) {
			String subjectId = it.next();
			qBuf.append("'").append(subjectId).append("'");
			if (it.hasNext()) {
				qBuf.append(',');
			}
		}
		qBuf.append(')');
		List<?> results = tsp.executeQuery(con, qBuf.toString());
		for (Iterator<?> iter = results.iterator(); iter.hasNext();) {
			Object[] row = (Object[]) iter.next();
			Subjexperiment se = (Subjexperiment) row[0];
			Researchgroup rg = (Researchgroup) row[1];
			Humansubject hs = (Humansubject) row[2];
			SubjectRecord sr = new SubjectRecord(rg.getName(), hs.getLocalid());
			String key = se.getSubjectid() + ":" + se.getNcExperimentUniqueid();
			sid2CondMap.put(key, sr);
		}
		return sid2CondMap;
	}

	static class SubjectRecord {
		String condition;
		String localid;

		public SubjectRecord(String condition, String localid) {
			super();
			this.condition = condition;
			this.localid = localid;
		}
	}// ;

	public static String getVisitKey(VisitInfo vi) {
		StringBuilder sb = new StringBuilder();
		sb.append(vi.getSubjectID()).append(':');
		sb.append(vi.getExpId()).append(':');
		sb.append(vi.getVisitId());
		return sb.toString();
	}
}
