package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.LogFactory;

import clinical.server.vo.Collectionequipment;
import clinical.server.vo.Dataobject;
import clinical.server.vo.Dataobjecttype;
import clinical.server.vo.Expcomponent;
import clinical.server.vo.Experiment;
import clinical.server.vo.Expsegment;
import clinical.server.vo.Rawdata;
import clinical.utils.Assertion;
import clinical.web.ICBFWFManagementService;
import clinical.web.ServiceFactory;
import clinical.web.common.IAuthorizationService;
import clinical.web.common.IDBCache;
import clinical.web.common.IDBPoolService;
import clinical.web.common.UserInfo;
import clinical.web.common.IAuthorizationService.PrivilegeLabel;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.exception.BaseException;
import clinical.web.vo.upload.SegmentInfo;
import clinical.web.vo.upload.SeriesHeaderInfo;
import clinical.web.vo.upload.VisitInfo;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: CBFWFManagementService.java 798 2013-04-23 23:37:47Z bozyurt $
 */

public class CBFWFManagementService extends AbstractServiceImpl implements
		ICBFWFManagementService {
	protected IDBCache dbCache;
	protected static Set<String> siteSet = new HashSet<String>(23);

	static {
		siteSet.add("duke");
		siteSet.add("bwh");
		siteSet.add("mgh");
		siteSet.add("ucla");
		siteSet.add("uci");
		siteSet.add("ucsd");
		siteSet.add("ucsf");
		siteSet.add("unm");
		siteSet.add("mrn");
		siteSet.add("uiowa");
		siteSet.add("umn");
		siteSet.add("yale");
	}
	
	public static boolean isAKnownSite(String siteStr) {
		return siteSet.contains(siteStr);
	}

	public CBFWFManagementService(String dbID) throws BaseException {
		super(dbID);
		log = LogFactory.getLog(CBFWFManagementService.class);
		dbCache = ServiceFactory.getDBCache(dbID);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.ICBFWFManagementService#getScanVisits(clinical.
	 * web.common.UserInfo)
	 */
	public Map<String, List<VisitInfo>> getScanVisits(UserInfo ui)
			throws Exception {
		IAuthorizationService authService = ServiceFactory
				.getAuthorizationService();

		Map<String, List<VisitInfo>> exp2VisitInfoMap = new HashMap<String, List<VisitInfo>>(
				7);
		List<Experiment> allExps = dbCache.getExperiments(ui, false);
		List<Experiment> userExps = new ArrayList<Experiment>(allExps.size());
		Map<Integer, Experiment> expMap = new HashMap<Integer, Experiment>(7);
		for (Experiment exp : allExps) {
			if (authService.isAuthorized(ui, theDBID, PrivilegeLabel.UPDATE, exp
					.getUniqueid().intValue())) {
				userExps.add(exp);
				expMap.put(exp.getUniqueid().intValue(), exp);
			}
		}
		allExps = null;

		// no experiments -> no results
		if (userExps.isEmpty()) {
			return exp2VisitInfoMap;
		}

		Connection con = null;
		IDBPoolService dbPoolService = null;

		try {

			dbPoolService = ServiceFactory.getPoolService(this.theDBID);
			con = dbPoolService.getConnection(ui.getName());
			con.setAutoCommit(false);
			// TODO also filter the visits already processed
			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			StringBuilder sb = new StringBuilder(128);
			sb.append("select distinct b.timeStamp, a.subjectid, ");
			sb.append("a.ncExperimentUniqueid, ");
			sb.append("a.componentid from Rawdata as a, Expcomponent as b ");
			sb.append("where a.subjectid = b.subjectid and ");
			sb.append("a.componentid = b.componentid and ");
			sb.append("a.ncExperimentUniqueid in (");
			for (Iterator<Experiment> it = userExps.iterator(); it.hasNext();) {
				Experiment exp = it.next();
				sb.append(exp.getUniqueid());
				if (it.hasNext())
					sb.append(',');
			}
			sb.append(") order by a.subjectid, a.ncExperimentUniqueid");

			List<?> results = tsp.executeQuery(con, sb.toString());
			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				Object[] row = (Object[]) it.next();
				Rawdata rd = (Rawdata) row[1];
				Expcomponent ec = (Expcomponent) row[0];
				int visitId = rd.getComponentid().intValue();
				int expId = rd.getNcExperimentUniqueid().intValue();
				Date visitDate = new Date(ec.getTimeStamp().getTime());
				Experiment exp = expMap.get(expId);
				List<VisitInfo> list = exp2VisitInfoMap.get(exp.getName());
				if (list == null) {
					list = new ArrayList<VisitInfo>();
					exp2VisitInfoMap.put(exp.getName(), list);
				}

				VisitInfo vi = new VisitInfo(visitId, null, null, rd
						.getSubjectid(), null, expId, visitDate, null);

				// FIXME hack for getting site id information used for FBIRN
				// Siemens processing
				int idx = exp.getName().lastIndexOf('_');
				if (idx != -1) {
					String suffix = exp.getName().substring(idx + 1);
					if (siteSet.contains(suffix)) {
						vi.setSiteName(suffix);
					}
				}
				list.add(vi);
			}
			populateMinimalSegmentInfo(ui, con, exp2VisitInfoMap);

			con.commit();

			for (List<VisitInfo> list : exp2VisitInfoMap.values()) {
				for (VisitInfo vi : list) {
					vi.determineIfHasAnat();
					vi.determineIfHasFieldMap();
					// IBO 7/13/2015
					vi.determineIfIs3DPCASL();
				}
			}
			return exp2VisitInfoMap;
		} catch (Exception x) {
			if (con != null) {
				con.rollback();
			}
			throw x;
		} finally {
			if (dbPoolService != null) {
				dbPoolService.releaseConnection(ui.getName(), con);
			}
		}
	}

	protected void populateMinimalSegmentInfo(UserInfo ui, Connection con,
			Map<String, List<VisitInfo>> exp2VisitInfoMap) throws Exception {
		Map<String, VisitInfo> viMap = new HashMap<String, VisitInfo>();
		Set<String> subjectIdSet = new HashSet<String>();
		for (List<VisitInfo> viList : exp2VisitInfoMap.values()) {
			for (VisitInfo vi : viList) {
				String key = prepKey(vi.getExpId(), vi.getSubjectID(), vi
						.getVisitId());
				viMap.put(key, vi);
				subjectIdSet.add(vi.getSubjectID());
			}
		}

		if (subjectIdSet.isEmpty()) {
			return;
		}

		TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
		StringBuilder sb = new StringBuilder(128);
		sb.append("select a.segmentid, a.name, a.componentid, "
				+ "a.ncExperimentUniqueid,"
				+ "a.subjectid from Expsegment as a where a.subjectid in (");
		for (Iterator<String> it = subjectIdSet.iterator(); it.hasNext();) {
			String subjectId = it.next();
			sb.append("'").append(subjectId).append("'");
			if (it.hasNext())
				sb.append(',');
		}
		sb.append(")");
		log.info("q:" + sb.toString());
		List<?> results = tsp.executeQuery(con, sb.toString());
		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			Expsegment seg = (Expsegment) it.next();
			int expId = seg.getNcExperimentUniqueid().intValue();
			int visitId = seg.getComponentid().intValue();
			String key = prepKey(expId, seg.getSubjectid(), visitId);
			VisitInfo vi = viMap.get(key);
			if (vi != null) {
				SegmentInfo si = new SegmentInfo(seg.getName(), seg
						.getDescription(), -1, null, null, seg.getTimeStamp(),
						null);
				si.setSegmentId(seg.getSegmentid().intValue());
				vi.addSI(si);
			}
		}

		tsp = new TSQLProcessor(this.sqlDialect);
		sb = new StringBuilder(128);
		sb.append("select distinct a.ncColequipmentUniqueid, "
				+ "a.subjectid, a.ncExperimentUniqueid from Rawdata "
				+ "as a where a.subjectid in (");
		for (Iterator<String> it = subjectIdSet.iterator(); it.hasNext();) {
			String subjectId = it.next();
			sb.append("'").append(subjectId).append("'");
			if (it.hasNext())
				sb.append(',');
		}
		sb.append(")");
		log.info("q:" + sb.toString());
		results = tsp.executeQuery(con, sb.toString());

		List<Collectionequipment> equipments = dbCache.getCollectionEquipments(
				ui, true);
		Map<BigDecimal, String> makeMap = new HashMap<BigDecimal, String>();
		for (Collectionequipment ce : equipments) {
			makeMap.put(ce.getUniqueid(), ce.getMake());
		}
		equipments = null;

		Map<String, List<VisitInfo>> expSubjViMap = new HashMap<String, List<VisitInfo>>();
		for (List<VisitInfo> viList : exp2VisitInfoMap.values()) {
			for (VisitInfo vi : viList) {
				String key = prepKey(vi.getExpId(), vi.getSubjectID());
				List<VisitInfo> list = expSubjViMap.get(key);
				if (list == null) {
					list = new ArrayList<VisitInfo>(1);
					expSubjViMap.put(key, list);
				}
				list.add(vi);
			}
		}

		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			Rawdata rd = (Rawdata) it.next();
			int expId = rd.getNcExperimentUniqueid().intValue();
			String key = prepKey(expId, rd.getSubjectid());
			String make = makeMap.get(rd.getNcColequipmentUniqueid());
			Assertion.assertNotNull(make);
			List<VisitInfo> list = expSubjViMap.get(key);
			for (VisitInfo vi : list) {
				vi.setScannerType(make);
			}
		}
	}

	static String prepKey(int expId, String subjectid) {
		StringBuilder sb = new StringBuilder();
		sb.append(subjectid).append(':').append(expId);
		return sb.toString();
	}

	static String prepKey(int expId, String subjectId, int visitId) {
		StringBuilder sb = new StringBuilder();
		sb.append(expId).append(':').append(subjectId).append(':');
		sb.append(visitId);
		return sb.toString();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.ICBFWFManagementService#getVisitInfosWithScans(
	 * clinical.web.common.UserInfo, java.util.List)
	 */
	public List<VisitInfo> getVisitInfosWithScans(UserInfo ui,
			List<VisitInfo> viList) throws Exception {
		Connection con = null;
		IDBPoolService dbPoolService = null;
		IDBCache dbCache = null;

		try {
			dbCache = ServiceFactory.getDBCache(theDBID);
			List<Dataobjecttype> dotList = dbCache.getDataObjectTypes(ui, true);
			Map<BigDecimal, Dataobjecttype> dotMap = new HashMap<BigDecimal, Dataobjecttype>();
			for (Dataobjecttype dot : dotList) {
				dotMap.put(dot.getUniqueid(), dot);
			}
			dotList = null;

			dbPoolService = ServiceFactory.getPoolService(this.theDBID);
			con = dbPoolService.getConnection(ui.getName());

			Set<String> subjectIdSet = new HashSet<String>();
			Map<String, VisitInfo> viMap = new HashMap<String, VisitInfo>();
			for (VisitInfo vi : viList) {
				subjectIdSet.add(vi.getSubjectID());
				String key = prepVisitKey(vi.getSubjectID(), vi.getExpId(), vi
						.getVisitId());
				viMap.put(key, vi);
			}

			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			StringBuilder sb = new StringBuilder(128);
			sb.append("select b.datauri, a.segmentid, a.subjectid,"
					+ "a.protocolid, a.timeStamp, a.componentid,"
					+ "a.ncExperimentUniqueid, a.name, c.objecttypeid "
					+ "from Expsegment as a, Rawdata as b, "
					+ "Dataobject as c "
					+ "where a.subjectid = b.subjectid and "
					+ "a.ncExperimentUniqueid = b.ncExperimentUniqueid and "
					+ "a.componentid = b.componentid and "
					+ "a.segmentid = b.segmentid and "
					+ "c.dataid = b.uniqueid and a.subjectid in (");

			for (Iterator<String> it = subjectIdSet.iterator(); it.hasNext();) {
				String subjectId = it.next();
				sb.append("'").append(subjectId).append("'");
				if (it.hasNext())
					sb.append(',');
			}
			sb.append(")");
			log.info("q:" + sb.toString());

			Map<String, SegmentInfo> siMap = new HashMap<String, SegmentInfo>();
			List<?> results = tsp.executeQuery(con, sb.toString());
			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				Object[] row = (Object[]) it.next();
				Rawdata rd = (Rawdata) row[0];
				Expsegment segment = (Expsegment) row[1];
				Dataobject dataObj = (Dataobject) row[2];
				String key = prepVisitKey(segment.getSubjectid(), segment
						.getNcExperimentUniqueid().intValue(), segment
						.getComponentid().intValue());

				VisitInfo vi = viMap.get(key);
				if (vi != null) {
					Date segmentTs = new Date(segment.getTimeStamp().getTime());

					String segKey = prepSegmentKey(segment.getSubjectid(),
							segment.getNcExperimentUniqueid().intValue(),
							segment.getComponentid().intValue(), segment
									.getSegmentid().intValue());

					SegmentInfo si = siMap.get(segKey);
					if (si == null) {
						SeriesHeaderInfo shi = new SeriesHeaderInfo();
						Dataobjecttype dot = dotMap.get(dataObj
								.getObjecttypeid());
						Assertion.assertNotNull(dot);
						String seriesImgType = getSeriesImageType(dot);
						Assertion.assertNotNull(seriesImgType);
						shi.setImageType(seriesImgType);

						shi.addImage(rd.getDatauri());

						si = new SegmentInfo(segment.getName(), null, 1,
								segment.getProtocolid(), -1, segmentTs, shi);
						si.setSegmentId(segment.getSegmentid().intValue());

						vi.addSI(si);
						siMap.put(segKey, si);
					} else {
						si.getShi().addImage(rd.getDatauri());
					}
				}
			}
		} finally {
			if (dbPoolService != null) {
				dbPoolService.releaseConnection(ui.getName(), con);
			}
		}
		return viList;
	}

	public static String getSeriesImageType(Dataobjecttype dot) {
		String objectType = dot.getObjecttype();
		if (objectType.endsWith("AFNI")) {
			return SeriesHeaderInfo.AFNI;
		} else if (objectType.endsWith("DICOM")) {
			return SeriesHeaderInfo.DICOM;
		} else if (objectType.endsWith("PFILE")) {
			return SeriesHeaderInfo.PFILE;
		} else if (objectType.endsWith("ANALYZE")) {
			return SeriesHeaderInfo.ANALYZE;
		}
		return null;
	}

	public static String prepVisitKey(String subjectID, int expID, int visitID) {
		StringBuilder sb = new StringBuilder();
		sb.append(subjectID).append(':').append(expID);
		sb.append(':').append(visitID);
		return sb.toString();
	}

	public static String prepSegmentKey(String subjectID, int expID,
			int visitID, int segmentId) {
		StringBuilder sb = new StringBuilder();
		sb.append(subjectID).append(':').append(expID);
		sb.append(':').append(visitID).append(':');
		sb.append(segmentId);
		return sb.toString();
	}

}
