package clinical.web.services;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.dao.JobsDAO;
import clinical.server.dao.VisitJobDAO;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Jobs;
import clinical.server.vo.Tableid;
import clinical.server.vo.VisitJob;
import clinical.utils.Assertion;
import clinical.utils.DateTimeUtils;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.IAppConfigService;
import clinical.web.IJobManagementService;
import clinical.web.ISQLDialect;
import clinical.web.ISequenceHelper;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBPoolService;
import clinical.web.common.ISecurityService;
import clinical.web.common.UserInfo;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.common.security.User;
import clinical.web.exception.BaseException;
import clinical.web.exception.DBPoolServiceException;
import clinical.web.exception.JobManServiceException;
import clinical.web.scheduler.IJob;
import clinical.web.scheduler.IJobFactory;
import clinical.web.scheduler.JobInfo;
import clinical.web.scheduler.JobRecord;
import clinical.web.scheduler.JobScheduler;
import clinical.web.scheduler.JobVisitContext;
import clinical.web.scheduler.JobScheduler.NullJobFactory;

/**
 * @author I. Burak Ozyurt
 * @version $Id: JobManagementServiceImpl.java,v 1.5 2008/01/12 01:00:33 bozyurt
 *          Exp $
 */
public class JobManagementServiceImpl implements IJobManagementService {
	/** Connection pool service interface */
	protected IDBPoolService dbPoolService;
	protected String dbType;
	protected ISecurityService securityService;

	/**
	 * interface for retrieving and caching nearly static data from the
	 * database.
	 */
	protected DBCache dbCache;
	/** unique number generator for primary keys */
	protected ISequenceHelper seqHelper;
	protected String dbID;
	private Log log = LogFactory.getLog(JobManagementServiceImpl.class);
	private ISQLDialect sqlDialect;

	public JobManagementServiceImpl(String dbID) throws BaseException {
		dbPoolService = ServiceFactory.getPoolService(dbID);
		dbCache = DBCache.getInstance(dbID);
		seqHelper = ServiceFactory.getSequenceHelper(dbID);
		this.dbID = dbID;
		securityService = ServiceFactory.getSecurityService();
		this.dbType = securityService.getDBType(dbID);
		sqlDialect = ServiceFactory.getSQLDialect(dbID);
	}

	public List<Jobs> getAllJobs(UserInfo ui) throws JobManServiceException {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			JobsDAO dao = DAOFactory.createJobsDAO(dbID);
			Jobs criteriaBean = new Jobs();
			return dao.find(con, criteriaBean);
		} catch (Exception x) {
			log.error("Error in getAllJobs", x);
			throw new JobManServiceException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public Jobs getTheJob(UserInfo ui, String jobUser, String jobID)
			throws JobManServiceException {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			JobsDAO dao = DAOFactory.createJobsDAO(dbID);
			Jobs criteriaBean = new Jobs();
			criteriaBean.setJobuser(jobUser);
			criteriaBean.setJobid(jobID);
			List<Jobs> jobList = dao.find(con, criteriaBean);
			if (jobList.isEmpty()) {
				return null;
			}

			return jobList.get(0);
		} catch (Exception x) {
			log.error("Error in getTheJob", x);
			throw new JobManServiceException(x);
		} finally {
			releaseConnection(con, ui);
		}

	}

	public List<Jobs> getJobsByStatus(UserInfo ui, String status)
			throws JobManServiceException {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			JobsDAO dao = DAOFactory.createJobsDAO(dbID);
			Jobs criteriaBean = new Jobs();
			criteriaBean.setJobstatus(status);
			return dao.find(con, criteriaBean);
		} catch (Exception x) {
			log.error("Error in getJobsByStatus", x);
			throw new JobManServiceException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public VisitJob findVisitJob(UserInfo ui, String jobID)
			throws JobManServiceException {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			StringBuilder sb = new StringBuilder(128);
			sb.append("select a.* from VisitJob as a, ");
			sb.append("Jobs as b where a.jobUniqueId = b.uniqueid ");
			sb.append("and b.jobid = '").append(jobID).append("'");
			List<?> results = tsp.executeQuery(con, sb.toString());
			if (results.isEmpty()) {
				return null;
			}
			return (VisitJob) results.get(0);
		} catch (Exception x) {
			log.error("Error in findVisitJob", x);
			throw new JobManServiceException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void removeJob(UserInfo ui, String jobID)
			throws JobManServiceException {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			JobsDAO dao = DAOFactory.createJobsDAO(dbID);
			Jobs criteriaBean = new Jobs();
			criteriaBean.setJobid(jobID);
			List<Jobs> jobList = dao.find(con, criteriaBean);
			Assertion.assertTrue(jobList.size() == 1);
			VisitJobDAO vjDAO = DAOFactory.createVisitJobDAO(dbID);
			VisitJob vjCR = new VisitJob();
			vjCR.setComponentId(jobList.get(0).getUniqueid());
			vjDAO.delete(con, vjCR);

			dao.delete(con, criteriaBean);
			con.commit();
		} catch (Exception x) {
			try {
				con.rollback();
			} catch (SQLException e) {
				log.error("rollback", e);
			}
			log.error("Error in removeJob", x);
			throw new JobManServiceException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void expireJob(UserInfo ui, String jobID)
			throws JobManServiceException {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			JobsDAO dao = DAOFactory.createJobsDAO(dbID);
			Jobs criteriaBean = new Jobs();
			criteriaBean.setJobid(jobID);
			List<Jobs> jl = dao.find(con, criteriaBean);
			if (!jl.isEmpty()) {
				Jobs job = jl.get(0);
				Jobs updateBean = new Jobs();
				updateBean.setJobstatus(job.getJobstatus() + "_expired");
				dao.update(con, updateBean, criteriaBean);
				con.commit();
			}
		} catch (Exception x) {
			try {
				con.rollback();
			} catch (SQLException e) {
				log.error("rollback", e);
			}
			log.error("Error in removeJob", x);
			throw new JobManServiceException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void removeOrphanJobs(UserInfo ui) throws JobManServiceException {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			con.setAutoCommit(false);
			JobsDAO dao = DAOFactory.createJobsDAO(dbID);

			IAppConfigService configService = ServiceFactory
					.getAppConfigService();

			String cacheRoot = configService
					.getParamValue("download.cacheroot");
			String dataRoot = configService.getParamValue("cbfbirn.data.root");

			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			StringBuilder sb = new StringBuilder(128);
			sb.append("select a.uniqueid, a.jobstatus, "
					+ "a.savedresultfile, a.jobuser, "
					+ "a.jobid from Jobs as a where a.jobstatus in ('"
					+ JobInfo.WAITING + "','" + JobInfo.RUNNING + "','"
					+ JobInfo.SHELVED + "')");

			List<?> mjList = tsp.executeQuery(con, sb.toString());
			JobScheduler scheduler = JobScheduler.getInstance();

			Map<String, List<String>> affectedUserMap = new HashMap<String, List<String>>();
			for (Object o : mjList) {
				Jobs job = (Jobs) o;

				if (job.getJobstatus().equals(JobInfo.WAITING)
						|| job.getJobstatus().equals(JobInfo.SHELVED)) {

					JobRecord jr = findJob(ui, job.getJobuser(), job.getJobid());
					if (jr != null) {
						if (job.getJobstatus().equals(JobInfo.SHELVED)) {
							Date jobstartdate = job.getJobstartdate();
							if (jobstartdate == null) {
								removeJob(ui, job.getJobid());
								continue;
							}
							int daysPassed = DateTimeUtils.calculateDayDiff(jobstartdate, new Date());
							if (daysPassed > 7) {
								// anything shelved older than a week is removed from the system. (IBO  3/17/2014)
								removeJob(ui, job.getJobid());
								continue;
							}
						}

						IJobFactory jobFactory = scheduler.getJobFactory(jr
								.getType());
						if (jobFactory != null
								|| !(jobFactory instanceof NullJobFactory)) {
							IJob job2Resume = jobFactory.create(jr);
							scheduler.recoverJob(job2Resume);
							continue;
						}

					}
				}

				String[] savedResultFiles = JobManagementHelper
						.fromDBSavedResultsFileString(job.getSavedresultfile());
				File f = new File(savedResultFiles[0]);
				boolean canRemove = f.getParentFile().isDirectory();
				canRemove &= f.getParentFile().getParentFile()
						.getAbsolutePath().equals(cacheRoot);
				if (canRemove) {
					FileUtils.deleteRecursively(f.getParentFile());
					log.info("cleaned up directory contents:"
							+ f.getParentFile());
				}
				List<String> jobIdList = affectedUserMap.get(job.getJobuser());
				if (jobIdList == null) {
					jobIdList = new ArrayList<String>(1);
					affectedUserMap.put(job.getJobuser(), jobIdList);
				}
				jobIdList.add(job.getJobid());
				Jobs jobCR = new Jobs();
				jobCR.setUniqueid(job.getUniqueid());

				dao.delete(con, jobCR);
			}
			con.commit();
			// build effected user list
			prepChrashReport(dataRoot, affectedUserMap);

		} catch (Exception x) {
			log.error("Error in findJob", x);
			try {
				con.rollback();
			} catch (SQLException e) {
				e.printStackTrace();
			}
			throw new JobManServiceException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	protected void prepChrashReport(String dataRoot,
			Map<String, List<String>> affectedUserMap) {
		if (affectedUserMap.isEmpty()) {
			return;
		}
		List<String> affectedUserList = new ArrayList<String>(
				affectedUserMap.keySet());
		Collections.sort(affectedUserList);
		SimpleDateFormat sdf = new SimpleDateFormat("MM_dd_yyyy_hh_mm_ss");
		String ts = sdf.format(new Date());

		File crashReportFile = new File(dataRoot, "crash_" + ts + ".report");
		BufferedWriter out = null;

		try {
			out = new BufferedWriter(new FileWriter(crashReportFile));
			out.write("Affected Jobs");
			out.newLine();
			out.write("-------------");
			out.newLine();
			for (String jobUser : affectedUserList) {
				StringBuilder sb = new StringBuilder(100);
				List<String> list = affectedUserMap.get(jobUser);
				for (Iterator<String> it = list.iterator(); it.hasNext();) {
					String jobID = it.next();
					sb.append(jobID);
					if (it.hasNext())
						sb.append(',');
				}
				sb.append(" (").append(jobUser).append(')');
				out.write(sb.toString());
				out.newLine();
			}
			out.write("Affected Users");
			out.newLine();
			out.write("-------------");
			out.newLine();

			for (String jobUser : affectedUserList) {
				out.write(jobUser);
				out.newLine();
			}
			log.info("wrote crash report:" + crashReportFile);
		} catch (IOException iox) {
			log.error("prepChrashReport", iox);
		} finally {
			FileUtils.close(out);
		}
	}

	public JobRecord findJob(UserInfo ui, String jobUser, String jobID)
			throws JobManServiceException {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			JobsDAO dao = DAOFactory.createJobsDAO(dbID);
			Jobs criteriaBean = new Jobs();
			criteriaBean.setJobuser(jobUser);
			criteriaBean.setJobid(jobID);
			List<Jobs> jobList = dao.find(con, criteriaBean);
			if (jobList.isEmpty()) {
				return null;
			}
			if (jobList.size() > 1) {
				throw new Exception("The criteria jobUser:" + jobUser
						+ " and jobID:" + jobID
						+ " returned more than one record!");
			}

			Jobs job = jobList.get(0);
			if (JobInfo.REMOVED.equals(job.getJobstatus())
					|| JobInfo.CANCELED.equals(job.getJobstatus())) {
				return null;
			}
			JobRecord jr = new JobRecord(job.getJobuser(), job.getJobtype(),
					job.getJobid(), job.getJobstartdate(), job.getJobstatus(),
					job.getDescription(),
					JobManagementHelper.fromDBSavedResultsFileString(job
							.getSavedresultfile()), false);
			jr.setContext(job.getJobContext());
			int downloadCount = job.getDownloadcount() != null ? job
					.getDownloadcount().intValue() : 0;
			jr.setDownloadCount(downloadCount);

			return jr;
		} catch (Exception x) {
			log.error("Error in findJob", x);
			throw new JobManServiceException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public List<JobRecord> getSuccesfulJobsForVisit(UserInfo ui, int expId,
			String subjectId, int visitId) throws JobManServiceException {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			StringBuilder sb = new StringBuilder(128);
			sb.append("select a.* from Jobs as a, VisitJob as b ");
			sb.append("where a.uniqueid = b.jobUniqueId and ");
			sb.append("b.expId =").append(expId).append(" and ");
			sb.append("b.subjectid = '").append(subjectId).append("' and ");
			sb.append("b.componentId = ").append(visitId);
			List<?> results = tsp.executeQuery(con, sb.toString());
			List<JobRecord> jrList = new ArrayList<JobRecord>(results.size());
			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				Jobs job = (Jobs) it.next();
				if (!JobInfo.FINISHED.equals(job.getJobstatus())
						&& !JobInfo.REMOVED.equals(job.getJobstatus())) {
					continue;
				}

				if (job.getJobenddate() == null) {
					continue;
				}

				JobRecord jr = new JobRecord(job.getJobuser(),
						job.getJobtype(), job.getJobid(),
						job.getJobstartdate(), job.getJobstatus(),
						job.getDescription(),
						JobManagementHelper.fromDBSavedResultsFileString(job
								.getSavedresultfile()), false);

				jr.setDateFinished(job.getJobenddate());

				if (job.getStatusDetail() != null) {
					jr.setStatusDetail(job.getStatusDetail());
				}
				if (job.getJobsize() != null
						&& job.getJobsize().longValue() > 0) {
					jr.setJobSize(job.getJobsize().longValue());
				}
				if (job.getDownloadcount() != null
						&& job.getDownloadcount().intValue() > 0) {
					jr.setDownloadCount(job.getDownloadcount().intValue());
				}
				if (job.getErrormsg() != null) {
					jr.setErrorMsg(job.getErrormsg());
				}
				jrList.add(jr);
			}
			return jrList;

		} catch (Exception x) {
			log.error("Error in getJobsForVisit", x);
			throw new JobManServiceException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public List<JobRecord> getJobsForUser(UserInfo ui, String jobUser)
			throws JobManServiceException {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			JobsDAO dao = DAOFactory.createJobsDAO(dbID);
			Jobs criteriaBean = new Jobs();
			criteriaBean.setJobuser(jobUser);

			List<Jobs> jobList = dao.find(con, criteriaBean);
			List<JobRecord> jrList = new ArrayList<JobRecord>(jobList.size());
			for (Iterator<Jobs> iterator = jobList.iterator(); iterator
					.hasNext();) {
				Jobs job = iterator.next();
				if (JobInfo.REMOVED.equals(job.getJobstatus())
						|| JobInfo.CANCELED.equals(job.getJobstatus())) {
					continue;
				}

				JobRecord jr = new JobRecord(job.getJobuser(),
						job.getJobtype(), job.getJobid(),
						job.getJobstartdate(), job.getJobstatus(),
						job.getDescription(),
						JobManagementHelper.fromDBSavedResultsFileString(job
								.getSavedresultfile()), false);
				if (job.getStatusDetail() != null) {
					jr.setStatusDetail(job.getStatusDetail());
				}
				if (job.getJobsize() != null
						&& job.getJobsize().longValue() > 0) {
					jr.setJobSize(job.getJobsize().longValue());
				}
				if (job.getDownloadcount() != null
						&& job.getDownloadcount().intValue() > 0) {
					jr.setDownloadCount(job.getDownloadcount().intValue());
				}
				if (job.getErrormsg() != null) {
					String errMsg = org.springframework.web.util.HtmlUtils
							.htmlEscape(job.getErrormsg());
					jr.setErrorMsg(errMsg);
				}
				if (job.getJobenddate() != null) {
					jr.setDateFinished(job.getJobenddate());
				}
				jrList.add(jr);
			}
			return jrList;
		} catch (Exception x) {
			log.error("Error in getJobsForUser", x);
			throw new JobManServiceException(x);
		} finally {
			releaseConnection(con, ui);
		}

	}

	public void updateJob(UserInfo ui, JobRecord jobRec)
			throws JobManServiceException {
		log.info("updateJob " + jobRec);
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			con.setAutoCommit(false);
			Jobs criteriaBean = new Jobs();
			criteriaBean.setJobid(jobRec.getJobID());
			criteriaBean.setJobuser(jobRec.getUser());

			Jobs updateBean = new Jobs();
			updateBean.setJobstatus(jobRec.getStatus());
			if (jobRec.getDateFinished() != null) {
				updateBean.setJobenddate(jobRec.getDateFinished());
			}
			if (jobRec.getJobSize() > 0) {
				updateBean.setJobsize(new BigDecimal(jobRec.getJobSize()));
			}
			if (jobRec.getDownloadCount() > 0) {
				updateBean.setDownloadcount(new BigDecimal(jobRec
						.getDownloadCount()));
			}
			if (jobRec.getLastDownloadTime() != null) {
				updateBean.setLastdownloadtime(jobRec.getLastDownloadTime());
			}
			if (jobRec.getSavedResultFiles() != null) {
				String s = JobManagementHelper
						.buildSavedResultStringForDB(jobRec
								.getSavedResultFiles());
				updateBean.setSavedresultfile(s);
			}
			if (jobRec.getErrorMsg() != null) {
				String stripped = jobRec.getErrorMsg().replaceAll("\\p{Cntrl}",
						"");
				if (stripped.length() > 2048) {
					log.error("Truncated error message:!" + stripped);
					stripped = stripped.substring(0, 2048);
				}
				updateBean.setErrormsg(stripped);
			}
			if (jobRec.getStatusDetail() != null) {
				updateBean.setStatusDetail(jobRec.getStatusDetail());
			}
			if (JobInfo.WAITING.equals(jobRec.getStatus())) {
				updateBean.setJobContext(jobRec.getContext());
			}

			updateBean.setModtime(new Date());
			JobsDAO dao = DAOFactory.createJobsDAO(dbID);
			dao.update(con, updateBean, criteriaBean);
			con.commit();
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in updateJob", x, true);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void addJob(UserInfo ui, JobRecord jobRec)
			throws JobManServiceException {
		log.info("addJob " + jobRec);
		Connection con = null;
		try {
			if (securityService == null) {
				securityService = ServiceFactory.getSecurityService();
			}
			con = dbPoolService.getConnection(ui.getName());
			con.setAutoCommit(false);

			VisitJobDAO vjDAO = DAOFactory.createVisitJobDAO(this.dbID);

			Jobs job = new Jobs();
			job.setJobuser(jobRec.getUser());
			job.setJobtype(jobRec.getType());
			job.setJobid(jobRec.getJobID());
			job.setJobstartdate(jobRec.getDate());
			job.setJobstatus(jobRec.getStatus());
			// TODO job size
			job.setDescription(jobRec.getDescription());

			String srfStr = JobManagementHelper
					.buildSavedResultStringForDB(jobRec.getSavedResultFiles());

			job.setSavedresultfile(srfStr);
			if (jobRec.getErrorMsg() != null) {
				String stripped = jobRec.getErrorMsg().replaceAll("\\p{Cntrl}",
						"");
				job.setErrormsg(stripped);
			}

			Map<String, User> userMap = securityService.getAllUsers(this.dbID);
			User u = userMap.get(ui.getName());

			Databaseuser databaseUser = dbCache.getDatabaseUser(ui, u
					.getDbUser().getName(), Constants.USERCLASS_ADMIN);
			job.setModuser(databaseUser.getUniqueid());
			job.setOwner(databaseUser.getUniqueid());
			Tableid tid = dbCache.getTableID(ui, Constants.JOBS_DB_TABLE);

			job.setTableid(tid.getUniqueid());
			job.setUniqueid(seqHelper.getNextUID(ui, Constants.JOBS_DB_TABLE,
					"uniqueid"));
			job.setModtime(new Date());

			if (jobRec.getContext() != null) {
				job.setJobContext(jobRec.getContext());
			}

			JobsDAO dao = DAOFactory.createJobsDAO(dbID);

			dao.insert(con, job);

			// save any job-visit context info
			if (!jobRec.getJvcList().isEmpty()) {
				for (JobVisitContext jvc : jobRec.getJvcList()) {
					VisitJob vj = new VisitJob();
					vj.setJobUniqueId(job.getUniqueid());
					vj.setSubjectid(jvc.getSubjectID());
					vj.setExpId(GenUtils.toBigDecimal(jvc.getExpID()));
					vj.setComponentId(GenUtils.toBigDecimal(jvc.getVisitID()));
					vj.setModtime(new Date());
					vjDAO.insert(con, vj);
				}
			}

			con.commit();
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in addJob", x, true);
		} finally {
			releaseConnection(con, ui);
		}

	}

	protected void handleErrorAndRollBack(Connection con, String msg,
			Exception x, boolean doRollback) throws JobManServiceException {
		if (con != null) {
			try {
				con.rollback();
			} catch (SQLException se) {
				log.error("", se);
				throw new JobManServiceException(se.getMessage());
			}
		}
		log.error(msg, x);
		throw new JobManServiceException(x);
	}

	protected void releaseConnection(Connection con, UserInfo ui) {
		if (con == null) {
			return;
		}
		try {
			dbPoolService.releaseConnection(ui.getName(), con);
		} catch (DBPoolServiceException x) {
			log.error("Cannot release connection for user " + ui.getName(), x);
		}
	}
}
