package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.LogFactory;

import clinical.server.dao.JobResultDAO;
import clinical.server.dao.JobResultGroupDAO;
import clinical.server.dao.JobResultGroupTypesDAO;
import clinical.server.dao.JobResultTypeDAO;
import clinical.server.dao.StoredJobResultGroupDAO;
import clinical.server.vo.JobResult;
import clinical.server.vo.JobResultGroup;
import clinical.server.vo.JobResultGroupTypes;
import clinical.server.vo.JobResultType;
import clinical.server.vo.StoredJobResultGroup;
import clinical.utils.Assertion;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.DBUtils;
import clinical.web.ISequenceHelper;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.common.query.JobResultQueryBuilder;
import clinical.web.common.query.Operator;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.exception.BaseException;
import clinical.web.vo.JobResultGroupInfo;
import clinical.web.vo.JobResultInfo;
import clinical.web.vo.JobResultTypeInfo;
import clinical.web.vo.JobResultTypeInfo.ResultType;
import clinical.web.vo.JobResultValueSummary;
import clinical.web.vo.StoredJobResultGroupInfo;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class JobResultManService extends AbstractServiceImpl implements
		IJobResultService {
	private ISequenceHelper seqHelper;

	public JobResultManService(String dbID) throws BaseException {
		super(dbID);
		log = LogFactory.getLog(JobProvenanceManServiceImpl.class);
		// dbCache = ServiceFactory.getDBCache(dbID);
		seqHelper = ServiceFactory.getSequenceHelper(dbID);
	}

	public List<JobResultGroupInfo> getJobResultGroups(Connection con)
			throws Exception {
		JobResultGroupDAO jrgDAO = DAOFactory
				.createJobResultGroupDAO(this.theDBID);
		List<JobResultGroup> jrgList = jrgDAO.find(con, new JobResultGroup());
		if (jrgList.isEmpty()) {
			return new ArrayList<JobResultGroupInfo>(0);
		}
		List<JobResultGroupInfo> jrgiList = new ArrayList<JobResultGroupInfo>(
				jrgList.size());
		Map<Integer, JobResultGroupInfo> jrgiMap = new HashMap<Integer, JobResultGroupInfo>(
				7);
		for (JobResultGroup jrg : jrgList) {
			JobResultGroupInfo jrgi = new JobResultGroupInfo(jrg.getName(),
					jrg.getDescription(), jrg.getUniqueid().intValue());
			jrgiList.add(jrgi);
			jrgiMap.put(jrgi.getId(), jrgi);
		}

		TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
		StringBuilder sb = new StringBuilder(100);
		sb.append("select a.*, b.jobResultGroupId ");
		sb.append("from JobResultType as a, JobResultGroupTypes as b ");
		sb.append("where a.uniqueid = b.jobResultTypeId and ");
		sb.append("b.jobResultGroupId  in (");
		boolean first = true;
		for (JobResultGroup jrg : jrgList) {
			if (!first) {
				sb.append(',');
			}
			sb.append(jrg.getUniqueid());
			first = false;
		}
		sb.append(") order by b.jobResultGroupId");

		List<?> results = tsp.executeQuery(con, sb.toString());
		for (Iterator<?> iter = results.iterator(); iter.hasNext();) {
			Object[] row = (Object[]) iter.next();
			JobResultType jrt = (JobResultType) row[0];
			JobResultGroupTypes jrgt = (JobResultGroupTypes) row[1];
			Integer jrGroupId = jrgt.getJobResultGroupId().intValue();
			JobResultGroupInfo jrgi = jrgiMap.get(jrGroupId);
			ResultType type = ResultType.valueOf(jrt.getResultType());
			Assertion.assertNotNull(type);
			JobResultTypeInfo jrti = new JobResultTypeInfo(jrt.getName(),
					jrt.getGroupName(), jrt.getDescription(), jrt.getUnit(),
					type);
			jrgi.addResultType(jrti);
		}

		return jrgiList;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.IJobResultService#saveJobResultGroup(clinical.web
	 * .common.UserInfo, clinical.web.vo.JobResultGroupInfo)
	 */
	@Override
	public void saveJobResultGroup(UserInfo ui, JobResultGroupInfo jrgi)
			throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			saveJobResultGroup(con, jrgi);
			con.commit();
		} catch (Exception x) {
			log.error("Error in saveJobResultGroup", x);
			try {
				con.rollback();
			} catch (SQLException e) {
				log.error("rollback:", e);
			}
			throw x;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void saveJobResultGroup(Connection con, JobResultGroupInfo jrgi)
			throws Exception {
		JobResultGroup jrg = new JobResultGroup();
		JobResultGroupDAO jrgDAO = DAOFactory
				.createJobResultGroupDAO(this.theDBID);

		JobResultGroup jrgCR = new JobResultGroup();
		jrgCR.setName(jrgi.getName());
		if (!jrgDAO.find(con, jrgCR).isEmpty()) {
			throw new Exception("A job result group with name '"
					+ jrgi.getName() + "' already exists!");
		}

		jrg.setName(jrgi.getName());
		jrg.setDescription(jrgi.getDescription());
		BigDecimal jrgUniqueID = seqHelper.getNextUID(con,
				Constants.JOB_RESULT_GROUP_DB_TABLE, "uniqueid");
		jrg.setUniqueid(jrgUniqueID);

		jrgDAO.insert(con, jrg);
		JobResultGroupTypesDAO jrgTypesDAO = DAOFactory
				.createJobResultGroupTypesDAO(theDBID);
		JobResultTypeDAO jtDAO = DAOFactory.createJobResultTypeDAO(theDBID);
		for (JobResultTypeInfo jrti : jrgi.getResultTypes()) {
			BigDecimal typeUniqueId = seqHelper.getNextUID(con,
					Constants.JOB_RESULT_TYPE_DB_TABLE, "uniqueid");
			JobResultType jrt = new JobResultType();
			jrt.setName(jrti.getName());
			jrt.setGroupName(jrti.getGroup());
			jrt.setDescription(jrti.getDescription());
			jrt.setResultType(jrti.getType().toString());
			jrt.setUnit(jrti.getUnit());
			jrt.setUniqueid(typeUniqueId);

			jtDAO.insert(con, jrt);
			JobResultGroupTypes rec = new JobResultGroupTypes();
			rec.setJobResultGroupId(jrgUniqueID);
			rec.setJobResultTypeId(typeUniqueId);
			jrgTypesDAO.insert(con, rec);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.IJobResultService#deleteJobResultGroup(clinical
	 * .web.common.UserInfo, java.lang.String, boolean)
	 */
	@Override
	public void deleteJobResultGroup(UserInfo ui, String jobResultGroupName,
			boolean deleteJobResultTypes) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			JobResultGroupDAO jrgDAO = DAOFactory
					.createJobResultGroupDAO(this.theDBID);
			JobResultGroupTypesDAO jrgTypesDAO = DAOFactory
					.createJobResultGroupTypesDAO(theDBID);
			JobResultTypeDAO jtDAO = DAOFactory.createJobResultTypeDAO(theDBID);
			JobResultGroup cr = new JobResultGroup();

			cr.setName(jobResultGroupName);

			List<JobResultGroup> jrgList = jrgDAO.find(con, cr);
			if (jrgList.isEmpty()) {
				// nothing to delete
				return;
			}
			JobResultGroup jobResultGroup = jrgList.get(0);

			JobResultGroupTypes jrgtCr = new JobResultGroupTypes();
			jrgtCr.setJobResultGroupId(jobResultGroup.getUniqueid());

			if (deleteJobResultTypes) {
				List<JobResultGroupTypes> list = jrgTypesDAO.find(con, jrgtCr);
				JobResultType jtCr = new JobResultType();
				for (JobResultGroupTypes jrgt : list) {
					jtCr.setUniqueid(jrgt.getJobResultTypeId());
					jtDAO.delete(con, jtCr);
				}
			}

			jrgTypesDAO.delete(con, jrgtCr);
			con.commit();
		} catch (Exception x) {
			log.error("Error in deleteJobResultTypes", x);
			try {
				con.rollback();
			} catch (SQLException e) {
				log.error("rollback:", e);
			}
			throw x;
		} finally {
			releaseConnection(con, ui);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.IJobResultService#deleteStoredJobResultGroup(clinical
	 * .web.common.UserInfo, java.lang.Integer)
	 */
	@Override
	public void deleteStoredJobResultGroup(UserInfo ui,
			Integer storedJobResultGroupId) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			deleteStoredJobResultGroup(con, storedJobResultGroupId);
			con.commit();
		} catch (Exception x) {
			log.error("Error in deleteStoredJobResultGroup", x);
			try {
				con.rollback();
			} catch (SQLException e) {
				log.error("rollback:", e);
			}
			throw x;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void deleteStoredJobResultGroup(Connection con,
			Integer storedJobResultGroupId) throws Exception {
		Assertion.assertNotNull(storedJobResultGroupId);
		StoredJobResultGroup sjrgCR = new StoredJobResultGroup();
		StoredJobResultGroupDAO groupDAO = DAOFactory
				.createStoredJobResultGroupDAO(theDBID);
		JobResultDAO jrDAO = DAOFactory.createJobResultDAO(theDBID);
		JobResult cr = new JobResult();
		cr.setStoredJobResultGroupId(new BigDecimal(storedJobResultGroupId));

		jrDAO.delete(con, cr);
		sjrgCR.setUniqueid(new BigDecimal(storedJobResultGroupId));
		groupDAO.delete(con, sjrgCR);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.IJobResultService#saveStoredJobResultGroup(clinical
	 * .web.common.UserInfo, clinical.web.vo.StoredJobResultGroupInfo,
	 * java.lang.Integer)
	 */
	@Override
	public void saveStoredJobResultGroup(UserInfo ui,
			StoredJobResultGroupInfo sjrgi, Integer jobProvenanceId)
			throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			saveStoredJobResultGroup(con, sjrgi, jobProvenanceId);
			con.commit();
		} catch (Exception x) {
			log.error("Error in saveStoredJobResultGroup", x);
			try {
				con.rollback();
			} catch (SQLException e) {
				log.error("rollback:", e);
			}
			throw x;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void saveStoredJobResultGroup(Connection con,
			StoredJobResultGroupInfo sjrgi, Integer jobProvenanceId)
			throws Exception {
		Assertion.assertNotNull(jobProvenanceId);
		StoredJobResultGroup sjrg = new StoredJobResultGroup();
		StoredJobResultGroupDAO groupDAO = DAOFactory
				.createStoredJobResultGroupDAO(theDBID);
		JobResultDAO jrDAO = DAOFactory.createJobResultDAO(theDBID);

		sjrg.setJobProvenanceId(new BigDecimal(jobProvenanceId));
		sjrg.setJobUniqueid(sjrgi.getJob().getUniqueid());
		sjrg.setModtime(new Date());
		if (sjrgi.getExpID() != null) {
			sjrg.setExpId(new BigDecimal(sjrgi.getExpID()));
		}
		sjrg.setSubjectid(sjrgi.getSubjectID());
		if (sjrgi.getVisitID() != null) {
			sjrg.setVisitId(new BigDecimal(sjrgi.getVisitID()));
		}

		BigDecimal jobResultGroupInfoId = getJobResultGroupInfoId(con, sjrgi
				.getJobResultGroup().getName());
		Assertion.assertNotNull(jobResultGroupInfoId);
		sjrg.setJobGroupId(jobResultGroupInfoId);
		BigDecimal sjrgId = seqHelper.getNextUID(con,
				Constants.STORED_JOB_RESULT_GROUP_DB_TABLE, "uniqueid");

		sjrg.setUniqueid(sjrgId);
		groupDAO.insert(con, sjrg);

		Map<String, JobResultType> map = new HashMap<String, JobResultType>();
		for (JobResultInfo jri : sjrgi.getResults()) {
			JobResult jr = new JobResult();
			jr.setName(jri.getName());
			jr.setValue(jri.getValue());
			if (jri.getNumericValue() != null) {
				jr.setNumericValue(jri.getNumericValue().doubleValue());
			}
			jr.setStoredJobResultGroupId(jobResultGroupInfoId);
			String key = jri.getTypeInfo().getGroup() + ":"
					+ jri.getTypeInfo().getName();
			JobResultType jrt = map.get(key);
			if (jrt == null) {
				jrt = findJobResultType(con, jri.getTypeInfo());
				Assertion.assertNotNull(jrt);
				map.put(key, jrt);
			}
			jr.setResultTypeId(jrt.getUniqueid());
			BigDecimal uniqueid = seqHelper.getNextUID(con,
					Constants.JOB_RESULT_DB_TABLE, "uniqueid");
			jr.setUniqueid(uniqueid);

			jrDAO.insert(con, jr);
		}
	}

	public JobResultType findJobResultType(Connection con,
			JobResultTypeInfo jrti) throws Exception {
		JobResultType cr = new JobResultType();
		JobResultTypeDAO jtDAO = DAOFactory.createJobResultTypeDAO(theDBID);
		cr.setName(jrti.getName());
		cr.setGroupName(jrti.getGroup());

		List<JobResultType> list = jtDAO.find(con, cr);
		return list.isEmpty() ? null : list.get(0);
	}

	public BigDecimal getJobResultGroupInfoId(Connection con, String groupName)
			throws Exception {
		TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
		StringBuilder sb = new StringBuilder(100);
		sb.append("select a.uniqueid from JobResultGroup as a ");
		sb.append("where a.name='").append(groupName).append("'");
		List<?> results = tsp.executeQuery(con, sb.toString());
		if (results.isEmpty()) {
			return null;
		}
		JobResultGroup jrg = (JobResultGroup) results.get(0);
		return jrg.getUniqueid();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.IJobResultService#queryForJobResults(clinical.web
	 * .common.UserInfo, clinical.web.common.query.Operator, java.util.List,
	 * java.util.List)
	 */
	@Override
	public List<JobResultValueSummary> queryForJobResults(UserInfo ui,
			Operator rootOp, List<JobResultGroupInfo> jrgiList,
			List<Integer> experimentIds) throws Exception {

		JobResultQueryBuilder jrqb = new JobResultQueryBuilder(sqlDialect,
				jrgiList, experimentIds);

		List<JobResultValueSummary> resultList = new LinkedList<JobResultValueSummary>();
		Connection con = null;
		Statement st = null;
		ResultSet rs = null;
		try {
			con = pool.getConnection(ui.getName());
			st = con.createStatement();
			jrqb.visit(rootOp);

			log.info("queryForJobResults = " + jrqb.getQuery());
			rs = st.executeQuery(jrqb.getQuery());

			while (rs.next()) {
				JobResultValueSummary jrvs = prepareJobResultValueSummary(rs);
				resultList.add(jrvs);
			}

		} finally {
			DBUtils.close(st, rs);
			releaseConnection(con, ui);
		}
		return resultList;
	}

	/*
	 * buf.append("select b.subjectid, a.value, a.name,");
	 * buf.append("b.job_group_id, b.exp_id, b.visit_id, b.modtime, c.result_type "
	 * ); buf.append("from nc_job_result a, nc_stored_job_result_group b,");
	 * buf.append("nc_job_result_type where ");
	 */
	private JobResultValueSummary prepareJobResultValueSummary(ResultSet rs)
			throws SQLException {
		String subjectID = rs.getString(1);
		String value = rs.getString(2);
		String varName = rs.getString(3);
		Integer jobGroupID = getValue(rs, 4);
		Integer expID = getValue(rs, 5);
		Integer visitID = getValue(rs, 6);
		Date modTime = rs.getDate(7);
		String resultType = rs.getString(8);
		JobResultValueSummary jrvs = new JobResultValueSummary();
		jrvs.setSubjectID(subjectID);
		jrvs.setName(varName);
		jrvs.setValue(value);
		jrvs.setGroupID(jobGroupID);
		jrvs.setExperimentID(expID);
		jrvs.setVisitID(visitID);
		jrvs.setModTime(modTime);
		ResultType rt = ResultType.valueOf(resultType);
		jrvs.setResultType(rt);

		return jrvs;
	}

	public static Integer getValue(ResultSet rs, int columIndex)
			throws SQLException {
		int x = rs.getInt(columIndex);
		if (rs.wasNull()) {
			return null;
		} else {
			return new Integer(x);
		}
	}
}
