package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.json.JSONArray;
import org.json.JSONObject;

import clinical.server.dao.AssessmentscoreDAO;
import clinical.server.dao.ScoreSynonymExperimentDAO;
import clinical.server.dao.ScoreSynonymsDAO;
import clinical.server.vo.Assessment;
import clinical.server.vo.Assessmentscore;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Experiment;
import clinical.server.vo.ScoreSynonymExperiment;
import clinical.server.vo.ScoreSynonyms;
import clinical.utils.Assertion;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.IDBPoolService;
import clinical.web.common.UserInfo;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.exception.BaseException;
import clinical.web.vo.SynonymInfo;

public class SynonymExpansionServiceImpl extends AbstractServiceImpl implements
		SynonymExpansionService {

	public SynonymExpansionServiceImpl(String dbID) throws BaseException {
		super(dbID);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.SynonymExpansionService#getExperimentsUsingAssessment
	 * (clinical.web.common.UserInfo, java.lang.String)
	 */
	@Override
	public JSONObject getExperimentsUsingAssessment(UserInfo ui, String asID)
			throws Exception {
		JSONObject js = new JSONObject();
		js.put("asID", asID);
		JSONArray jsArr = new JSONArray();
		js.put("expList", jsArr);
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			StringBuilder qb = new StringBuilder();
			qb.append("select distinct e.name, e.uniqueid from Storedassessment as s, Experiment as e ");
			qb.append(
					"where s.ncExperimentUniqueid = e.uniqueid and s.assessmentid = ")
					.append(asID);
			List<?> results = tsp.executeQuery(con, qb.toString());
			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				Experiment exp = (Experiment) it.next();
				JSONObject expJS = new JSONObject();
				expJS.put("id", exp.getUniqueid().intValue());
				expJS.put("name", exp.getName());
				jsArr.put(expJS);
			}
		} finally {
			releaseConnection(con, ui);
		}

		return js;
	}

	public List<SynonymInfo> getAllSynonymInfos(UserInfo ui) throws Exception {
		Connection con = null;
		IDBPoolService dbPoolService = null;
		try {
			con = pool.getConnection(ui.getName());
			ScoreSynonymsDAO ssDAO = DAOFactory.createScoreSynonymsDAO(theDBID);

			List<ScoreSynonyms> ssList = ssDAO.find(con, new ScoreSynonyms());
			List<SynonymInfo> siList = new ArrayList<SynonymInfo>(ssList.size());
			if (ssList.isEmpty()) {
				return siList;
			}
			IDBCache dbCache = ServiceFactory.getDBCache(theDBID);

			List<Assessment> assessments = dbCache.getAssessments(ui, false);
			Map<BigDecimal, String> asMap = new HashMap<BigDecimal, String>();
			for (Assessment as : assessments) {
				asMap.put(as.getUniqueid(), as.getName());
			}

			Map<Integer, SynonymInfo> siMap = new HashMap<Integer, SynonymInfo>();
			for (ScoreSynonyms ss : ssList) {
				Integer ssId = ss.getUniqueid().intValue();
				String assessmentName = asMap.get(ss.getAssessmentId());
				SynonymInfo si = new SynonymInfo(ssId, ss.getScoreName(),
						assessmentName, ss.getSynonyms(),
						ss.getCanonicalValue());
				siList.add(si);
				siMap.put(si.getScoreSynonymId(), si);
			}
			List<Experiment> experiments = dbCache.getExperiments(ui, false);
			Map<BigDecimal, String> expNameMap = new HashMap<BigDecimal, String>();
			for (Experiment exp : experiments) {
				expNameMap.put(exp.getUniqueid(), exp.getName());
			}

			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			StringBuilder qb = new StringBuilder();
			qb.append("select a.* from ScoreSynonymExperiment as a where ");
			qb.append("a.scoreSynonymId in (");
			for (Iterator<SynonymInfo> iter = siList.iterator(); iter.hasNext();) {
				SynonymInfo si = iter.next();
				qb.append(si.getScoreSynonymId());
				if (iter.hasNext()) {
					qb.append(',');
				}
			}
			qb.append(')');
			List<?> results = tsp.executeQuery(con, qb.toString());
			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				ScoreSynonymExperiment sse = (ScoreSynonymExperiment) it.next();
				String expName = expNameMap.get(sse.getExperimentId());
				SynonymInfo si = siMap.get(sse.getScoreSynonymId().intValue());
				if (si != null) {
					si.addExpName(expName);
				}
			}
			return siList;
		} finally {
			if (dbPoolService != null) {
				dbPoolService.releaseConnection(ui.getName(), con);
			}
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.SynonymExpansionService#saveScoreSynonyms(clinical
	 * .web.common.UserInfo, clinical.server.vo.ScoreSynonyms, java.util.List)
	 */
	@Override
	public ScoreSynonyms saveScoreSynonyms(UserInfo ui, ScoreSynonyms ss,
			List<Integer> expIds) throws Exception {
		Connection con = null;
		IDBPoolService dbPoolService = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			ScoreSynonymsDAO ssDAO = DAOFactory.createScoreSynonymsDAO(theDBID);
			ScoreSynonyms ssCR = new ScoreSynonyms();
			ssCR.setAssessmentId(ss.getAssessmentId());
			ssCR.setScoreName(ss.getScoreName());
			ssCR.setCanonicalValue(ss.getCanonicalValue());

			List<ScoreSynonyms> list = ssDAO.find(con, ssCR);
			if (!list.isEmpty()) {
				throw new Exception("Duplicate Score Synonym Entry");
			}

			// get score ID
			Assessmentscore score = findScore(con, ss);
			Assertion.assertNotNull(score);
			ss.setScoreId(score.getUniqueid());

			ss.setModtime(new Date());
			Databaseuser databaseUser = getDatabaseUser(ui,
					Constants.USERCLASS_ADMIN);
			Assertion.assertNotNull(databaseUser);
			ss.setModuser(databaseUser.getUniqueid());
			BigDecimal uniqueid = ServiceFactory
					.getSequenceHelper(this.theDBID).getNextUID(ui,
							Constants.SCORE_SYNONYMS_DB_TABLE, "uniqueid");
			ss.setUniqueid(uniqueid);

			ssDAO.insert(con, ss);
			if (!ss.getAppliesglobal() && !expIds.isEmpty()) {
				ScoreSynonymExperimentDAO sseDAO = DAOFactory
						.createScoreSynonymExperimentDAO(theDBID);
				for (Integer expId : expIds) {
					ScoreSynonymExperiment sse = new ScoreSynonymExperiment(
							uniqueid, new BigDecimal(expId));
					sseDAO.insert(con, sse);
				}
			}
			con.commit();
			return ss;
		} catch (Throwable t) {
			handleErrorAndRollBack(con, t.getMessage(), t, true);
			return null;
		} finally {
			if (dbPoolService != null) {
				if (con != null) {
					con.setAutoCommit(true);
				}
				dbPoolService.releaseConnection(ui.getName(), con);
			}
		}
	}

	Assessmentscore findScore(Connection con, ScoreSynonyms ss)
			throws Exception {
		AssessmentscoreDAO scoreDAO = DAOFactory
				.createAssessmentscoreDAO(theDBID);
		Assessmentscore cr = new Assessmentscore();
		cr.setScorename(ss.getScoreName());
		cr.setAssessmentid(ss.getAssessmentId());
		List<Assessmentscore> list = scoreDAO.find(con, cr);

		if (list.isEmpty()) {
			return null;
		}
		return list.get(0);

	}

	@Override
	public void deleteScoreSynonyms(UserInfo ui, Integer scoreSynonymId)
			throws Exception {
		Connection con = null;
		IDBPoolService dbPoolService = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			ScoreSynonymsDAO ssDAO = DAOFactory.createScoreSynonymsDAO(theDBID);
			ScoreSynonymExperimentDAO sseDAO = DAOFactory
					.createScoreSynonymExperimentDAO(theDBID);
			ScoreSynonymExperiment sseCR = new ScoreSynonymExperiment();
			sseCR.setScoreSynonymId(new BigDecimal(scoreSynonymId));

			sseDAO.delete(con, sseCR);

			ScoreSynonyms ssCR = new ScoreSynonyms();
			ssCR.setUniqueid(new BigDecimal(scoreSynonymId));

			ssDAO.delete(con, ssCR);
			con.commit();
		} catch (Throwable t) {
			handleErrorAndRollBack(con, t.getMessage(), t, true);
		} finally {
			if (dbPoolService != null) {
				if (con != null) {
					con.setAutoCommit(true);
				}
				dbPoolService.releaseConnection(ui.getName(), con);
			}
		}

	}

}
