package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import clinical.server.dao.ConfWebuserDAO;
import clinical.server.dao.PublicationDAO;
import clinical.server.dao.UsageDataDAO;
import clinical.server.vo.ConfWebuser;
import clinical.server.vo.Experiment;
import clinical.server.vo.Publication;
import clinical.server.vo.UsageData;
import clinical.utils.Assertion;
import clinical.web.DAOFactory;
import clinical.web.ISequenceHelper;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.exception.BaseException;
import clinical.web.vo.UsageInfo;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: UsageManagementService.java 420 2011-08-03 00:42:39Z bozyurt $
 */
public class UsageManagementService extends AbstractServiceImpl implements
		IUsageManagementService {
	protected ISequenceHelper seqHelper;

	public UsageManagementService(String dbID) throws BaseException {
		super(dbID);
		seqHelper = ServiceFactory.getSequenceHelper(theDBID);
	}

	public Map<ConfWebuser, List<UsageInfo>> getAllUsages(UserInfo ui)
			throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			Map<ConfWebuser, List<UsageInfo>> userUsageMap = new HashMap<ConfWebuser, List<UsageInfo>>();

			UsageDataDAO udDAO = DAOFactory.createUsageDataDAO(theDBID);
			UsageData udCR = new UsageData();
			List<UsageData> udList = udDAO.find(con, udCR);
			if (udList.isEmpty()) {
				return userUsageMap;
			}

			IDBCache dbCache = ServiceFactory.getDBCache(theDBID);
			List<Experiment> experiments = dbCache.getExperiments(ui, false);
			Map<BigDecimal, Experiment> expMap = new HashMap<BigDecimal, Experiment>();
			for (Experiment exp : experiments) {
				expMap.put(exp.getUniqueid(), exp);
			}

			ConfWebuserDAO wuDAO = DAOFactory.createConfWebuserDAO(theDBID);
			List<ConfWebuser> wuList = wuDAO.find(con, new ConfWebuser());
			Map<BigDecimal, ConfWebuser> wuMap = new HashMap<BigDecimal, ConfWebuser>();
			for (ConfWebuser wu : wuList) {
				wuMap.put(wu.getUniqueId(), wu);
			}

			Map<BigDecimal, UsageInfo> usageInfoMap = new HashMap<BigDecimal, UsageInfo>();

			List<UsageInfo> usageInfoList = new ArrayList<UsageInfo>(udList
					.size());
			for (UsageData ud : udList) {
				Experiment exp = expMap.get(ud.getExpId());
				UsageInfo usageInfo = new UsageInfo(exp);
				usageInfo.setUsageData(ud);
				usageInfoList.add(usageInfo);
				usageInfoMap.put(ud.getUniqueid(), usageInfo);
			}

			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			StringBuilder sb = new StringBuilder(128);
			sb.append("select a.uniqueid, b.* from UsageData as a, ");
			sb.append("Publication as b ");
			sb.append("where a.uniqueid = b.udId");
			sb.append(" order by a.uniqueid");

			List<?> results = tsp.executeQuery(con, sb.toString());
			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				Object[] row = (Object[]) it.next();
				UsageData ud = (UsageData) row[0];
				Publication pub = (Publication) row[1];
				UsageInfo usageInfo = usageInfoMap.get(ud.getUniqueid());
				usageInfo.addReference(pub);
			}
			for (UsageInfo usageInfo : usageInfoList) {
				ConfWebuser wu = wuMap.get(usageInfo.getUsageData()
						.getWebuserId());
				List<UsageInfo> uiList = userUsageMap.get(wu);
				if (uiList == null) {
					uiList = new ArrayList<UsageInfo>(1);
					userUsageMap.put(wu, uiList);
				}
				uiList.add(usageInfo);
			}

			UsageInfoComparator comparator = new UsageInfoComparator();
			for (List<UsageInfo> uiList : userUsageMap.values()) {
				Collections.sort(uiList, comparator);
			}
			con.commit();
			return userUsageMap;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "getAllUsages", x, true);
		} finally {
			releaseConnection(con, ui);
		}
		return null;
	}

	private final class UsageInfoComparator implements Comparator<UsageInfo> {
		public int compare(UsageInfo o1, UsageInfo o2) {
			return o1.getExperiment().getName().compareTo(
					o2.getExperiment().getName());
		}
	}

	public List<UsageInfo> findUsageData(UserInfo ui) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			ConfWebuser webUser = getWebUser(ui, con);

			UsageDataDAO udDAO = DAOFactory.createUsageDataDAO(theDBID);
			UsageData udCR = new UsageData();
			udCR.setWebuserId(webUser.getUniqueId());
			List<UsageData> udList = udDAO.find(con, udCR);
			if (udList.isEmpty()) {
				return new ArrayList<UsageInfo>(0);
			}

			IDBCache dbCache = ServiceFactory.getDBCache(theDBID);
			List<Experiment> experiments = dbCache.getExperiments(ui, false);
			Map<BigDecimal, Experiment> expMap = new HashMap<BigDecimal, Experiment>();
			for (Experiment exp : experiments) {
				expMap.put(exp.getUniqueid(), exp);
			}

			Map<BigDecimal, UsageInfo> usageInfoMap = new HashMap<BigDecimal, UsageInfo>();

			List<UsageInfo> usageInfoList = new ArrayList<UsageInfo>(udList
					.size());
			for (UsageData ud : udList) {
				Experiment exp = expMap.get(ud.getExpId());
				UsageInfo usageInfo = new UsageInfo(exp);
				usageInfo.setUsageData(ud);
				usageInfoList.add(usageInfo);
				usageInfoMap.put(ud.getUniqueid(), usageInfo);
			}

			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			StringBuilder sb = new StringBuilder(128);
			sb
					.append("select a.uniqueid, b.* from UsageData as a, Publication as b ");
			sb.append("where a.uniqueid = b.udId and ");
			sb.append("a.webuserId = ");
			sb.append(webUser.getUniqueId());
			sb.append(" order by a.uniqueid");

			List<?> results = tsp.executeQuery(con, sb.toString());
			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				Object[] row = (Object[]) it.next();
				UsageData ud = (UsageData) row[0];
				Publication pub = (Publication) row[1];
				UsageInfo usageInfo = usageInfoMap.get(ud.getUniqueid());
				usageInfo.addReference(pub);
			}
			con.commit();

			return usageInfoList;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "findUsageData", x, true);
		} finally {
			releaseConnection(con, ui);
		}
		return null;
	}

	public void saveUsageData(UserInfo ui, UsageInfo usage) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			ConfWebuser webUser = getWebUser(ui, con);

			usage.getUsageData().setExpId(usage.getExperiment().getUniqueid());
			usage.getUsageData().setWebuserId(webUser.getUniqueId());

			UsageDataDAO udDAO = DAOFactory.createUsageDataDAO(theDBID);
			UsageData udCR = new UsageData();
			udCR.setExpId(usage.getExperiment().getUniqueid());
			udCR.setWebuserId(webUser.getUniqueId());

			List<UsageData> udList = udDAO.find(con, udCR);
			if (udList.isEmpty()) {
				BigDecimal nextUID = seqHelper.getNextUID(ui, "nc_usage_data",
						"uniqueid");
				usage.getUsageData().setUniqueid(nextUID);
				udDAO.insert(con, usage.getUsageData());
				udList = udDAO.find(con, udCR);
				Assertion.assertTrue(udList.size() == 1);
				usage.getUsageData().setUniqueid(udList.get(0).getUniqueid());
			} else {
				String persistentGrants = udList.get(0).getGrants();
				String persistentPapers = udList.get(0).getPapers();

				if (!persistentGrants.equals(usage.getUsageData().getGrants())
						|| hasChange(persistentPapers, usage.getUsageData()
								.getPapers())) {
					udDAO.update(con, usage.getUsageData(), udCR);
				}
			}

			con.commit();
		} catch (Exception x) {
			handleErrorAndRollBack(con, "saveUsageData", x, true);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public static boolean hasChange(String original, String newContent) {
		if (newContent != null && newContent.length() > 0) {
			if (original == null || !original.equals(newContent)) {
				return true;
			}
		}
		return false;
	}

	public void addPublication(UserInfo ui, UsageInfo usage, Publication pub)
			throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			ConfWebuser webUser = getWebUser(ui, con);
			usage.getUsageData().setExpId(usage.getExperiment().getUniqueid());
			usage.getUsageData().setWebuserId(webUser.getUniqueId());

			UsageDataDAO udDAO = DAOFactory.createUsageDataDAO(theDBID);
			UsageData udCR = new UsageData();
			udCR.setExpId(usage.getExperiment().getUniqueid());
			udCR.setWebuserId(webUser.getUniqueId());
			List<UsageData> udList = udDAO.find(con, udCR);

			if (udList.isEmpty()) {
				BigDecimal nextUID = seqHelper.getNextUID(ui, "nc_usage_data",
						"uniqueid");
				usage.getUsageData().setUniqueid(nextUID);
				udDAO.insert(con, usage.getUsageData());
				udList = udDAO.find(con, udCR);
				Assertion.assertTrue(udList.size() == 1);
				usage.getUsageData().setUniqueid(udList.get(0).getUniqueid());
			} else {
				String persistentGrants = udList.get(0).getGrants();
				if (!persistentGrants.equals(usage.getUsageData().getGrants())) {
					udDAO.update(con, usage.getUsageData(), udCR);
				}
			}

			addPublication(con, ui, pub, usage.getUsageData());
			usage.addReference(pub);
			con.commit();
		} catch (Exception x) {
			handleErrorAndRollBack(con, "addPublication", x, true);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void addPublication(Connection con, UserInfo ui, Publication pub,
			UsageData ud) throws Exception {
		PublicationDAO dao = DAOFactory.createPublicationDAO(theDBID);
		Publication cr = new Publication();
		cr.setReference(pub.getReference());
		cr.setUdId(ud.getUniqueid());

		List<Publication> list = dao.find(con, cr);
		pub.setUdId(ud.getUniqueid());

		if (list.isEmpty()) {
			BigDecimal nextUID = seqHelper.getNextUID(ui, "nc_publication",
					"uniqueid");
			pub.setUniqueid(nextUID);
			dao.insert(con, pub);
			list = dao.find(con, pub);
			Assertion.assertTrue(list.size() == 1);
			pub.setUniqueid(list.get(0).getUniqueid());
		} else {
			dao.update(con, pub, cr);
		}
	}

	protected ConfWebuser getWebUser(UserInfo ui, Connection con)
			throws Exception {
		ConfWebuserDAO wuDAO = DAOFactory.createConfWebuserDAO(theDBID);
		ConfWebuser wuCR = new ConfWebuser();
		wuCR.setName(ui.getPerceivedName());
		List<ConfWebuser> wuList = wuDAO.find(con, wuCR);
		Assertion.assertTrue(wuList.size() == 1);
		ConfWebuser webUser = wuList.get(0);
		return webUser;
	}
}
