package clinical.web.services;

import java.io.File;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.vo.Jobs;
import clinical.utils.FileUtils;
import clinical.web.Constants;
import clinical.web.IJobManagementService;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.exception.BaseException;
import clinical.web.scheduler.JobInfo;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class WFFileCacheCleanupService implements IPeriodicService {
	protected String cacheRoot;
	protected String dbID;
	protected long cacheCheckInterval = 3600000l; // every hour;
	protected long uploadCacheMaxAge = 24 * 3600000l; // a day
	protected long wfCacheMaxAge = 7 * 24 * 3600000l; // 7 days
	private static Log log = LogFactory.getLog(WFFileCacheCleanupService.class);

	public WFFileCacheCleanupService(String cacheRoot, String dbID)
			throws BaseException {
		this.cacheRoot = cacheRoot;
		this.dbID = dbID;
	}

	@Override
	public long getPeriod() {
		return cacheCheckInterval;
	}

	@Override
	public void service() throws Exception {
		File cacheRootDir = new File(cacheRoot);
		IJobManagementService js = ServiceFactory.getJobManagementService(dbID);

		List<Jobs> shelvedJobs = js.getJobsByStatus(new UserInfo(
				Constants.ADMIN_USER, null, null), JobInfo.SHELVED);
		Set<String> shelvedJobIdSet = new HashSet<String>();
		for (Jobs job : shelvedJobs) {
			shelvedJobIdSet.add(job.getJobid());
		}
		File[] files = cacheRootDir.listFiles();
		for (File file : files) {
			if (file.isDirectory()) {
				if (file.getName().equals("upload")) {
					cleanupUploadCache(file);
				} else {
					handleWFCacheCleanup(file, shelvedJobIdSet);
				}
			}
		}
	}

	protected void handleWFCacheCleanup(File dir, Set<String> shelvedJobIdSet) {
		try {
			if (isPastAge(dir) && 
					!isShelved(dir, shelvedJobIdSet)) {
				File[] files = dir.listFiles();
				for (File f : files) {
					if (f.isDirectory()) {
						FileUtils.deleteRecursively(f);
					} else if (f.isFile()) {
						if (!f.getName().endsWith(".gz")) {
							f.delete();
						}
					}
				}
			}
		} catch (Throwable t) {
			log.error("handleWFCacheCleanup:" + t.getMessage());
		}
	}

	protected boolean isShelved(File jobDir, Set<String> shelvedJobIdSet) {
		String name = jobDir.getName();
		int idx = name.lastIndexOf('_');
		if (idx == -1) {
			return false;
		}
		String jobId = name.substring(idx + 1);
		boolean flag = shelvedJobIdSet.contains(jobId);
		if (flag) {
			log.info("detecting a shelved job directory:" + jobDir);
		}
		return flag;
	}

	protected boolean isPastAge(File dir) {
		long latestLM = dir.lastModified();
		File[] files = dir.listFiles();
		for (File f : files) {
			long lastModified = f.lastModified();
			if (lastModified > latestLM) {
				latestLM = lastModified;
			}
		}
		long age = System.currentTimeMillis() - latestLM;
		if (age > wfCacheMaxAge) {
			return true;
		}
		return false;
	}

	protected void cleanupUploadCache(File uploadCacheRoot) {
		File[] files = uploadCacheRoot.listFiles();
		for (File file : files) {
			if (file.isDirectory()) {
				try {
					long lastModified = file.lastModified();
					long age = System.currentTimeMillis() - lastModified;
					if (age > uploadCacheMaxAge) {
						FileUtils.deleteRecursively(file);
					}
				} catch (Throwable t) {
					log.error("cleanupUploadCache:" + t.getMessage());
				}
			}
		}
	}

}
