package clinical.web.vo;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import clinical.utils.Assertion;
import clinical.web.common.query.BinaryOperator;
import clinical.web.common.query.Operator;
import clinical.web.common.query.QueryPartInfo;
import clinical.web.common.query.QueryUtils;
import clinical.web.common.query.SearchPredicate;
import clinical.web.common.query.UnaryOperator;
import clinical.web.common.vo.AsScoreInfo;
import clinical.web.common.vo.AssessmentSelectionInfo;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: AssessmentQueryInfo.java,v 1.1.2.1 2008/07/15 00:58:50 bozyurt
 *          Exp $
 */
public class AssessmentQueryInfo implements Serializable {
	private static final long serialVersionUID = 1L;
	protected List<QueryPartInfo> qpiList;
	protected List<AssessmentSelectionInfo> asiList = new ArrayList<AssessmentSelectionInfo>(
			7);

	public AssessmentQueryInfo() {
	}

	public AssessmentQueryInfo(List<AssessmentSelectionInfo> asiList) {
		super();
		this.asiList = asiList;
	}

	public AssessmentQueryInfo(AssessmentQueryInfo other) {
		asiList = other.asiList;
		qpiList = new ArrayList<QueryPartInfo>(other.qpiList.size());
		for (QueryPartInfo qp : other.qpiList) {
			qpiList.add(new QueryPartInfo(qp));
		}
	}
	
	public int getQueryPartSize() {
	   return qpiList != null ? qpiList.size() : 0;	
	}

	public void resetAllSelected() {
		for (AssessmentSelectionInfo asi : asiList) {
			asi.setSelected(false);
			if (asi.getScores() == null)
				continue;
			for (AsScoreInfo si : asi.getScores())
				si.setSelected(false);
		}
	}

	public AssessmentSelectionInfo findAsi(BigDecimal asID) {
		for (AssessmentSelectionInfo asi : asiList)
			if (asi.getAssessmentID().equals(asID))
				return asi;
		return null;
	}

	public List<QueryPartInfo> getOPIList4Site(
			List<AssessmentSelectionInfo> newAsiList) {
		List<QueryPartInfo> list = new ArrayList<QueryPartInfo>(qpiList.size());
		Map<String, AssessmentSelectionInfo> newAsiMap = new HashMap<String, AssessmentSelectionInfo>(
				17);
		for (AssessmentSelectionInfo newAsi : newAsiList) {
			newAsiMap.put(newAsi.getName(), newAsi);
		}
		for (QueryPartInfo qpi : this.qpiList) {
			AssessmentSelectionInfo newAsi = newAsiMap.get(qpi.getAsi()
					.getName());
			if (newAsi == null) {
				return null;
			}
			AsScoreInfo newSi = newAsi.findScoreInfo(qpi.getScoreInfo()
					.getName());
			if (newSi == null)
				return null;
			QueryPartInfo newQpi = new QueryPartInfo(qpi, newAsi, newSi);
			list.add(newQpi);
		}
		return list;
	}

	public Operator prepareOpTree() {
		Operator root = null;
		BinaryOperator last = null;
		int noQParts = qpiList.size();
		boolean first = true;
		for (Iterator<QueryPartInfo> it = qpiList.iterator(); it.hasNext();) {
			QueryPartInfo qp = it.next();
			String connector = qp.getConnector();
			SearchPredicate.Range range = null;
			SearchPredicate.DateRange dateRange = null;
			SearchPredicate sp = null;

			int logOp = qp.getOperator();
			int spType = QueryUtils.getSearchPredicateType(qp.getScoreInfo()
					.getType());
			if (qp.getLowBound() != null && qp.getLowBound().length() > 0
					&& qp.getUppBound() != null
					&& qp.getUppBound().length() > 0) {

				Object v = QueryUtils.convertToType(qp.getLowBound(), spType);
				if (v instanceof Date) {
					Date lowBound = (Date) v;
					Date uppBound = (Date) QueryUtils.convertToType(
							qp.getUppBound(), spType);
					dateRange = new SearchPredicate.DateRange(lowBound,
							uppBound);
				} else {
					Number lowBound = (Number) QueryUtils.convertToType(
							qp.getLowBound(), spType);
					Number uppBound = (Number) QueryUtils.convertToType(
							qp.getUppBound(), spType);

					range = new SearchPredicate.Range(lowBound, uppBound);
				}
			}

			Object value = null;
			if (range == null && dateRange == null) {
				if (qp.getRhs().trim().equals("*")) {
					value = qp.getRhs().trim();
				} else {
					value = QueryUtils.convertToType(qp.getRhs(), spType);
				}
			}
			Object attribute = qp.getScoreInfo();
			if (range != null) {
				sp = new SearchPredicate(attribute, range, logOp, spType);
			} else if (dateRange != null) {
				sp = new SearchPredicate(attribute, dateRange, logOp, spType);
			} else {
				sp = new SearchPredicate(attribute, value, logOp, spType);
			}

			if (noQParts == 1) {
				root = new UnaryOperator(sp, Operator.NONE);
			} else {
				if (first) {
					UnaryOperator uo = new UnaryOperator(sp, Operator.NONE);
					root = new BinaryOperator();
					((BinaryOperator) root).setLhs(uo);
					last = (BinaryOperator) root;
					first = false;
				} else {
					// do we have more query parts?
					if (it.hasNext()) {
						last.setLogicalOp(getSelectedConnector(connector));
						BinaryOperator bop = new BinaryOperator();
						UnaryOperator lhs = new UnaryOperator(sp, Operator.NONE);
						bop.setLhs(lhs);
						last.setRhs(bop);
						last = bop;
					} else {
						// the last one
						UnaryOperator uo = new UnaryOperator(sp, Operator.NONE);
						last.setLogicalOp(getSelectedConnector(connector));
						last.setRhs(uo);
					}
				}
			}
		}
		return root;
	}

	public boolean hasQueryParts() {
		return qpiList != null && !qpiList.isEmpty();
	}

	protected static int getSelectedConnector(String connector) {
		return (connector.equals("AND")) ? Operator.AND : Operator.OR;
	}

	public void updateQueryFromJSON(JSONObject js) throws JSONException {
		Assertion.assertTrue(asiList != null && !asiList.isEmpty());
		JSONArray jsArr = js.getJSONArray("asQPIList");
		Map<String, AssessmentSelectionInfo> asiMap = new HashMap<String, AssessmentSelectionInfo>(
				17);
		for (AssessmentSelectionInfo asi : asiList) {
			asiMap.put(asi.getAssessmentID().toString(), asi);
		}
		this.qpiList = new ArrayList<QueryPartInfo>(jsArr.length());
		for (int i = 0; i < jsArr.length(); i++) {
			JSONObject json = jsArr.getJSONObject(i);
			QueryPartInfo qpi = QueryPartInfo.initializeFromJSON(json);
			qpiList.add(qpi);
		}
	}

	public static AssessmentQueryInfo initializeFromJSON(JSONObject js)
			throws JSONException {
		AssessmentQueryInfo aqi = new AssessmentQueryInfo();
		JSONArray jsArr = js.getJSONArray("asQPIList");
		aqi.qpiList = new ArrayList<QueryPartInfo>(jsArr.length());
		Map<BigDecimal, AssessmentSelectionInfo> asiMap = new HashMap<BigDecimal, AssessmentSelectionInfo>(
				7);
		for (int i = 0; i < jsArr.length(); i++) {
			JSONObject json = jsArr.getJSONObject(i);
			QueryPartInfo qpi = QueryPartInfo.initializeFromJSON(json);
			aqi.qpiList.add(qpi);
			AssessmentSelectionInfo asi = asiMap.get(qpi.getAsi()
					.getAssessmentID());
			if (asi == null) {
				asi = new AssessmentSelectionInfo(qpi.getAsi().getName(), qpi
						.getAsi().getAssessmentID());
				asiMap.put(qpi.getAsi().getAssessmentID(), asi);
				aqi.asiList.add(asi);
			}
			asi.addScore(qpi.getScoreInfo());
		}
		return aqi;
	}

	public List<QueryPartInfo> getQpiList() {
		return qpiList;
	}

	public List<AssessmentSelectionInfo> getAsiList() {
		return asiList;
	}

	public JSONArray queryToJSON() throws JSONException {
		Assertion.assertNotNull(qpiList);
		JSONArray jsArr = new JSONArray();
		for (QueryPartInfo qpi : qpiList) {
			jsArr.put(qpi.toJSON(new JSONObject()));
		}
		return jsArr;
	}

	public JSONArray toJSON() throws JSONException {
		JSONArray jsArr = new JSONArray();
		for (AssessmentSelectionInfo asi : asiList) {
			jsArr.put(asi.toJSON(new JSONObject()));
		}
		return jsArr;
	}

	public String getJSONStr() {
		try {
			JSONArray jsArr = toJSON();
			return jsArr.toString(2);
		} catch (JSONException e) {
			e.printStackTrace();
		}
		return "[]";
	}

	public void setQpiList(List<QueryPartInfo> qpiList) {
		this.qpiList = qpiList;
	}
}
