package clinical.web.workflow.cbf;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GridLayout;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import javax.imageio.ImageIO;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JApplet;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import javax.swing.border.EtchedBorder;

public class CBFMultipleROIApplet extends JApplet implements
		PropertyChangeListener {
	private URL imageSrc;
	private ImageLoader loader;
	private MosaicSelectionPanel msp;
	private JPanel messagePanel;
	private JLabel messageLabel;
	// for testing
	static String imageFilename = "brain.png";
	private static final long serialVersionUID = 1L;

	public CBFMultipleROIApplet() {
	}

	public CBFMultipleROIApplet(ImageLoader loader) {
		super();
		this.loader = loader;
	}

	public void init() {
		msp = null;
		messagePanel = null;
		messageLabel = null;
		buildMessagePanel();

		System.out.println("codeBase:" + getCodeBase());
		String base = getCodeBase().toString();
		base = base.replaceFirst("pages\\/", "");

		String url = base + "/cbfroi.do?action=getImages";
		System.out.println("url=" + url);

		List<URL> images = new ArrayList<URL>();
		BufferedReader in = null;
		try {
			URLConnection con = getConnection(url, false);

			in = new BufferedReader(new InputStreamReader(con.getInputStream()));
			String line = null;
			line = in.readLine();
			int numImages = toInt(line, 0);

			URL baseURL = new URL(base);
			for (int i = 0; i < numImages; i++) {
				line = in.readLine();
				System.out.println(line);
				imageSrc = new URL(baseURL, line);
				System.out.println("imageSrc:" + imageSrc.toString());
				images.add(imageSrc);
			}
			System.out.println("numImages:" + numImages);
			if (numImages > 0) {
				int numCols = 6;

				int numRows = numImages / numCols;
				if ((numImages % numCols) > 0) {
					numRows++;
				}
				System.out.println("preparing image loader...");
				this.loader = new ImageLoader(images, numRows, numCols);
			}

		} catch (Throwable t) {
			t.printStackTrace();
			setMessage(t.getMessage(), true);
			System.exit(1);
		} finally {
			if (in != null) {
				try {
					in.close();
				} catch (Exception e) {
				}
			}
		}

		// for testing
		if (images.isEmpty()) {
			try {
				imageSrc = new URL(getCodeBase(), imageFilename);

				for (int i = 0; i < 8; i++) {
					images.add(imageSrc);
				}
				this.loader = new ImageLoader(images, 2, 4);
			} catch (Exception e) {
				e.printStackTrace();
				setMessage(e.getMessage(), true);
				System.exit(1);
			}
		}

		buildUI();
	}

	void buildUI() {
		this.msp = new MosaicSelectionPanel(loader);
		this.msp.addPropertyChangeListener(this);

		add(msp, BorderLayout.CENTER);
	}

	private void buildMessagePanel() {
		this.messagePanel = new JPanel(new FlowLayout(FlowLayout.LEFT, 5, 5));
		messagePanel.setPreferredSize(new Dimension(0, 50));
		messagePanel.setMaximumSize(new Dimension(0, 50));
		messagePanel.setMinimumSize(new Dimension(0, 50));
		// String testText = "<html>line1<br>line2<br>line3<br>line4";
		this.messageLabel = new JLabel("");
		messageLabel.setFont(new Font(Font.DIALOG, Font.PLAIN, 11));
		this.messagePanel.add(this.messageLabel);
		add(messagePanel, BorderLayout.NORTH);
	}

	void setMessage(String msg, boolean errorFlag) {
		if (msg == null) {
			msg = "An unrecognized error has occured!";
		}
		String fm = formatText(msg, 80);
		if (errorFlag) {
			messageLabel.setText("<html><font color='red'><b>Error:" + fm
					+ "</b></font>");
		} else {
			messageLabel.setText("<html>" + fm);
		}
	}

	URLConnection getConnection(String urlStr, boolean hasOutput)
			throws IOException {
		URL serviceURL = new URL(urlStr);

		URLConnection con = serviceURL.openConnection();
		con.setDoInput(true);
		con.setDoOutput(hasOutput);
		con.setUseCaches(false);
		con.setDefaultUseCaches(false);

		con.setRequestProperty("Content-Type", "text/plain");
		return con;
	}

	public static int toInt(String value, int defaultVal) {
		if (value == null) {
			return defaultVal;
		}
		try {
			return Integer.parseInt(value);
		} catch (NumberFormatException nfe) {
			return defaultVal;
		}
	}

	@SuppressWarnings("unchecked")
	public void propertyChange(PropertyChangeEvent evt) {
		System.out.println("propertyChange:" + evt.getPropertyName());
		if (evt.getPropertyName().equals("selectedCell")) {
			int[] selectedCellLoc = (int[]) evt.getNewValue();
			int row = selectedCellLoc[0];
			int col = selectedCellLoc[1];
			BufferedImage cellBI = loader.getCell(row, col);
			int sliceNo = loader.getSelectedSliceNo(row, col);
			ROIComponent roiComp = new ROIComponent(cellBI, sliceNo);
			ROIPanel panel = new ROIPanel(roiComp);
			remove(this.msp);
			add(panel, BorderLayout.CENTER);
			validate();
			repaint();
		} else if (evt.getPropertyName().equals("selectedCellList")) {
			List<ImageLoc> selectedImgLocs = (List<ImageLoc>) evt.getNewValue();
			List<ROIComponent> roiComps = new ArrayList<ROIComponent>(
					selectedImgLocs.size());
			List<Integer> sliceNos = new ArrayList<Integer>(selectedImgLocs
					.size());
			for (ImageLoc il : selectedImgLocs) {
				int row = il.row;
				int col = il.col;
				BufferedImage cellBI = loader.getCell(row, col);
				int sliceNo = loader.getSelectedSliceNo(row, col);
				ROIComponent roiComp = new ROIComponent(cellBI, sliceNo);
				roiComps.add(roiComp);
				sliceNos.add(sliceNo);
			}
			ROIPanel panel = new ROIPanel(roiComps, sliceNos);
			remove(this.msp);
			add(panel, BorderLayout.CENTER);
			validate();
			repaint();
		}
	}

	public static String formatText(String str, int maxLineLen) {
		if (str.length() <= maxLineLen) {
			return str;
		}
		StringBuffer buf = new StringBuffer(str.length() + 10);
		StringTokenizer stok = new StringTokenizer(str);
		int count = 0;
		boolean first = true;
		while (stok.hasMoreTokens()) {
			String tok = stok.nextToken();
			int tokLen = tok.length();
			if (count + tokLen + 1 > maxLineLen) {
				buf.append("<br>");
				buf.append(tok);
				count = tokLen;
			} else {
				if (first) {
					buf.append(tok);
					first = false;
					count += tokLen;
				} else {
					buf.append(' ').append(tok);
					count += tokLen + 1;
				}
			}
		}
		return buf.toString();
	}

	class ROIPanel extends JPanel implements ActionListener {
		private static final long serialVersionUID = 1L;
		ROIComponent roiComp;
		List<ROIComponent> roiCompList;
		List<Integer> sliceNos;
		List<byte[][]> maskList = new ArrayList<byte[][]>(10);
		int curSliceIdx = 0;
		JButton clearButton;
		JButton acceptButton;
		JPanel mainPanel;
		JLabel msgLabel;

		public ROIPanel(ROIComponent roiComp) {
			super(new BorderLayout(5, 5));
			this.roiComp = roiComp;

			JPanel controlPanel = new JPanel(null);
			BoxLayout bl = new BoxLayout(controlPanel, BoxLayout.X_AXIS);
			controlPanel.setLayout(bl);

			clearButton = new JButton("Clear");
			acceptButton = new JButton("Accept");
			clearButton.addActionListener(this);
			acceptButton.addActionListener(this);
			msgLabel = new JLabel("Dbl-Click to finish, Right-Click to undo");
			controlPanel.add(Box.createHorizontalStrut(5));
			controlPanel.add(msgLabel);
			controlPanel.add(Box.createHorizontalStrut(5));
			controlPanel.add(clearButton);
			controlPanel.add(Box.createHorizontalStrut(5));
			controlPanel.add(acceptButton);
			controlPanel.add(Box.createHorizontalGlue());

			mainPanel = new JPanel(new BorderLayout(5, 5));
			mainPanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
			mainPanel.add(roiComp, BorderLayout.CENTER);

			add(mainPanel, BorderLayout.CENTER);
			add(controlPanel, BorderLayout.SOUTH);
		}

		public ROIPanel(List<ROIComponent> roiComps, List<Integer> sliceNos) {
			super(new BorderLayout(5, 5));
			this.roiCompList = roiComps;
			this.sliceNos = sliceNos;
			this.curSliceIdx = 0;
			initPanel(roiCompList.get(curSliceIdx));
			prepControlPanel(curSliceIdx);
		}

		void prepControlPanel(int idx) {
			JPanel controlPanel = new JPanel(null);
			BoxLayout bl = new BoxLayout(controlPanel, BoxLayout.X_AXIS);
			controlPanel.setLayout(bl);

			clearButton = new JButton("Clear");
			acceptButton = new JButton("Accept");
			clearButton.addActionListener(this);
			acceptButton.addActionListener(this);
			String msg = prepareMessage(idx);
			msgLabel = new JLabel(msg);
			controlPanel.add(Box.createHorizontalStrut(5));
			controlPanel.add(msgLabel);
			controlPanel.add(Box.createHorizontalStrut(5));
			controlPanel.add(clearButton);
			controlPanel.add(Box.createHorizontalStrut(5));
			controlPanel.add(acceptButton);
			controlPanel.add(Box.createHorizontalGlue());
			add(controlPanel, BorderLayout.SOUTH);
		}

		String prepareMessage(int idx) {
			StringBuilder sb = new StringBuilder(100);
			sb.append((idx + 1)).append(" of ").append(sliceNos.size());
			sb.append(" (").append(sliceNos.get(idx)).append(") ");
			sb.append("Dbl-Click to finish, Right-Click to undo");
			return sb.toString();
		}

		void initPanel(ROIComponent anROIComp) {
			if (mainPanel != null) {
				remove(mainPanel);
			}

			mainPanel = new JPanel(new BorderLayout(5, 5));
			mainPanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
			mainPanel.add(anROIComp, BorderLayout.CENTER);

			add(mainPanel, BorderLayout.CENTER);
		}

		public void actionPerformed(ActionEvent e) {
			if (e.getSource() == clearButton) {
				if (roiComp != null) {
					roiComp.clear();
				} else {
					ROIComponent roic = roiCompList.get(curSliceIdx);
					roic.clear();
				}
			} else if (e.getSource() == acceptButton) {

				if (roiComp != null) {
					byte[][] mask = roiComp.getMask();

					try {
						// saveMask("/Users/bozyurt/mask.txt", mask);
						sendMask(mask, roiComp.selectedSliceNo);

						// ensure only one press to the Accept button
						acceptButton.setEnabled(false);
						clearButton.setEnabled(false);
					} catch (Throwable ex) {
						setMessage("Cannot send mask! " + ex.getMessage(), true);
						ex.printStackTrace();
					}
				} else {
					ROIComponent roic = roiCompList.get(curSliceIdx);
					byte[][] mask = roic.getMask();
					this.maskList.add(mask);
					if ((curSliceIdx + 1) < sliceNos.size()) {
						curSliceIdx++;
						SwingUtilities.invokeLater(new Runnable() {
							public void run() {
								initPanel(roiCompList.get(curSliceIdx));
								String msg = prepareMessage(curSliceIdx);
								msgLabel.setText(msg);

							}
						});

					} else {
						try {
							sendMasks(maskList, sliceNos);
							// ensure only one press to the Accept button
							acceptButton.setEnabled(false);
							clearButton.setEnabled(false);
						} catch (Throwable ex) {
							setMessage("Cannot send masks! " + ex.getMessage(),
									true);
							ex.printStackTrace();
						}
					}
				}
			}
		}

		void sendMasks(List<byte[][]> masks, List<Integer> selectedSliceNoList)
				throws Throwable {
			String base = getCodeBase().toString();
			base = base.replaceFirst("pages\\/", "");

			String url = base + "cbfroi.do";
			System.out.println("url=" + url);
			Writer out = null;
			BufferedReader in = null;
			try {
				URLConnection con = getConnection(url, true);
				con.setRequestProperty("content-type",
						"application/x-www-form-urlencoded");

				StringBuilder sb = new StringBuilder(51000);
				sb.append("action=sendMasks");
				sb.append("&rows=").append(masks.get(0).length);
				sb.append("&cols=").append(masks.get(0)[0].length);
				StringBuilder sliceSB = new StringBuilder();
				for (Iterator<Integer> it = selectedSliceNoList.iterator(); it
						.hasNext();) {
					Integer sliceNo = it.next();
					sliceSB.append(sliceNo);
					if (it.hasNext())
						sliceSB.append(',');
				}

				sb.append("&sliceNoList=").append(
						URLEncoder.encode(sliceSB.toString(), "UTF-8"));

				for (int i = 0; i < masks.size(); i++) {
					byte[][] mask = masks.get(i);
					sb.append("&mask").append(i).append('=').append(
							URLEncoder.encode(toCSVString(mask), "UTF-8"));
				}
				out = new OutputStreamWriter(con.getOutputStream());
				String data = sb.toString();
				out.write(data);
				out.flush();

				// get and check the response
				in = new BufferedReader(new InputStreamReader(con
						.getInputStream()));
				String line = null;
				line = in.readLine();
				if (line != null) {
					if (line.startsWith("ERROR:")) {
						line = line.substring(6);
						throw new Exception(line);
					} else {
						setMessage(
								"Mask is sent and CBF processing is resumed successfully.",
								false);
					}
				}
				System.out.println("sent mask.");

			} finally {
				if (out != null) {
					try {
						out.close();
					} catch (Exception e) {
					}
				}
				if (in != null) {
					try {
						in.close();
					} catch (Exception e) {
					}
				}
			}
		}

		void sendMask(byte[][] mask, int selectedSliceNo) throws Throwable {
			String base = getCodeBase().toString();
			base = base.replaceFirst("pages\\/", "");

			String url = base + "cbfroi.do";
			System.out.println("url=" + url);
			Writer out = null;
			BufferedReader in = null;
			try {
				URLConnection con = getConnection(url, true);
				con.setRequestProperty("content-type",
						"application/x-www-form-urlencoded");

				StringBuilder sb = new StringBuilder(11000);
				sb.append("action=sendMask");
				sb.append("&rows=").append(mask.length);
				sb.append("&cols=").append(mask[0].length);
				sb.append("&sliceNo=").append(selectedSliceNo);
				sb.append("&mask=").append(
						URLEncoder.encode(toCSVString(mask), "UTF-8"));

				out = new OutputStreamWriter(con.getOutputStream());
				String data = sb.toString();
				out.write(data);
				out.flush();

				// get and check the response
				in = new BufferedReader(new InputStreamReader(con
						.getInputStream()));
				String line = null;
				line = in.readLine();
				if (line != null) {
					if (line.startsWith("ERROR:")) {
						line = line.substring(6);
						throw new Exception(line);
					} else {
						setMessage(
								"Mask is sent and CBF processing is resumed successfully.",
								false);
					}
				}
				System.out.println("sent mask.");

			} finally {
				if (out != null) {
					try {
						out.close();
					} catch (Exception e) {
					}
				}
				if (in != null) {
					try {
						in.close();
					} catch (Exception e) {
					}
				}
			}
		}

		String toCSVString(byte[][] mask) {
			StringBuilder sb = new StringBuilder(11000);
			for (int i = 0; i < mask.length; i++) {
				for (int j = 0; j < mask[i].length; j++) {
					sb.append(mask[i][j]).append(',');
				}
			}
			sb.deleteCharAt(sb.length() - 1);
			String s = sb.toString();
			return s;
		}

		void saveMask(String path, byte[][] mask) throws IOException {
			BufferedWriter out = null;
			try {
				out = new BufferedWriter(new FileWriter(path));

				for (int i = 0; i < mask.length; i++) {
					StringBuilder sb = new StringBuilder(mask[0].length * 2);
					for (int j = 0; j < mask[i].length; j++) {
						sb.append(mask[i][j]);
						if ((j + 1) < mask[i].length) {
							sb.append(',');
						}
					}
					out.write(sb.toString());
					out.newLine();
				}
			} finally {
				if (out != null) {
					try {
						out.close();
					} catch (Exception e) {
					}
				}
			}
		}
	}

	class ROIComponent extends Component implements MouseListener,
			MouseMotionListener {
		private BufferedImage bi;
		private int selectedSliceNo;
		int w, h;
		int scaleFactor = 6;
		Dimension prefSize;
		int startX = -1, startY = -1;
		int x2 = -1, y2 = -1;
		List<Point> points = new ArrayList<Point>();
		private static final long serialVersionUID = 1L;
		private boolean finished = false;
		private boolean didUndo = false;

		public ROIComponent(URL imageSrc, int selectedSliceNo) {
			try {
				this.selectedSliceNo = selectedSliceNo;
				bi = ImageIO.read(imageSrc);

				w = bi.getWidth(null);
				h = bi.getHeight(null);
				prefSize = new Dimension(scaleFactor * w, scaleFactor * h);
			} catch (IOException iox) {
				System.out.println("Image cannot be read:" + iox.getMessage());
				System.exit(1);
			}
			addMouseListener(this);
			addMouseMotionListener(this);
		}

		public ROIComponent(BufferedImage bi, int selectedSliceNo) {
			this.bi = bi;
			this.selectedSliceNo = selectedSliceNo;
			w = bi.getWidth(null);
			h = bi.getHeight(null);
			prefSize = new Dimension(scaleFactor * w, scaleFactor * h);
			// System.out.println("prefSize:" + prefSize);
			addMouseListener(this);
			addMouseMotionListener(this);
		}

		public Dimension getPreferredSize() {
			return prefSize;
		}

		public Dimension getMaximumSize() {
			return prefSize;
		}

		public Dimension getMinimumSize() {
			return prefSize;
		}

		public void paint(Graphics g) {
			Graphics2D g2 = (Graphics2D) g;
			AffineTransform at = AffineTransform.getScaleInstance(scaleFactor,
					scaleFactor);
			AffineTransformOp aop = new AffineTransformOp(at,
					AffineTransformOp.TYPE_BICUBIC);

			g2.drawImage(bi, aop, 0, 0);
			if (!points.isEmpty()) {
				Point p = points.get(0);
				plotDot((int) p.getX(), (int) p.getY(), g2);
				int len = points.size();
				int x = (int) p.getX();
				int y = (int) p.getY();
				for (int i = 1; i < len; i++) {
					p = points.get(i);
					plotDot((int) p.getX(), (int) p.getY(), g2);
					g2.drawLine(x, y, (int) p.getX(), (int) p.getY());
					x = (int) p.getX();
					y = (int) p.getY();
				}
			}
		}

		void addPoint(Point p) {
			if (points.isEmpty()) {
				points.add(p);
			} else {
				Point lastPoint = points.get(points.size() - 1);
				if (!lastPoint.equals(p)) {
					points.add(p);
				}
			}
		}

		void plotDot(int x, int y, Graphics2D g2) {
			Color curColor = g2.getColor();
			g2.setColor(Color.green);
			g2.fillRect(x - 2, y - 2, 5, 5);
			g2.setColor(curColor);
		}

		void replotDot(int x, int y, Graphics2D g2) {
			g2.fillRect(x - 2, y - 2, 5, 5);
		}

		public void clear() {
			points.clear();
			startX = startY = x2 = y2 = -1;
			finished = false;
			repaint();
		}

		public boolean isClosedPath() {
			if (points.isEmpty() || points.size() < 4) {
				return false;
			}
			Point sp = points.get(0);
			Point ep = points.get(points.size() - 1);
			if (Math.abs(sp.x - ep.x) <= scaleFactor
					&& Math.abs(sp.y - ep.y) <= scaleFactor) {
				return true;
			}
			return false;
		}

		public byte[][] getMask() {
			byte[][] mask = new byte[this.h][this.w];
			int n = points.size();
			int[] xs = new int[n];
			int[] ys = new int[n];
			for (int i = 0; i < n; i++) {
				Point p = points.get(i);
				xs[i] = p.x;
				ys[i] = p.y;
			}
			Polygon poly = new Polygon(xs, ys, n);
			for (int i = 0; i < h; i++) {
				for (int j = 0; j < w; j++) {
					int x = j * scaleFactor;
					int y = i * scaleFactor;
					if (poly.contains(x, y)) {
						mask[i][j] = 1;
					}
				}
			}
			return mask;
		}

		public void mouseClicked(MouseEvent e) {
			if (finished) {
				System.out.println("finished:" + finished);
				System.out.println(this);
				return;
			}

			if (e.getX() >= prefSize.width || e.getY() >= prefSize.height) {
				return;
			}

			// if right mouse is clicked
			if (!didUndo
					&& (e.getModifiers() & InputEvent.BUTTON3_MASK) == InputEvent.BUTTON3_MASK) {
				if (points.size() > 1) {
					Point lastPoint = points.remove(points.size() - 1);
					repaint();

					if (!points.isEmpty()) {
						Graphics2D g2 = (Graphics2D) getGraphics();

						lastPoint = points.get(points.size() - 1);
						startX = (int) lastPoint.getX();
						startY = (int) lastPoint.getY();

						x2 = y2 = -1;
						g2.setPaintMode();
						setForeground(Color.green);
						g2.drawLine(startX, startY, e.getX(), e.getY());
						g2.setXORMode(getBackground());
						setForeground(Color.green);

					}
					didUndo = false; // false
					return;
				}
			}

			if (e.getClickCount() > 1) {
				startX = -1;
				startY = -1;
				if (!points.isEmpty()) {
					finished = true;
					System.out.println(this);
				}
				return;
			}

			startX = e.getX();
			startY = e.getY();
			Point p = new Point(startX, startY);
			addPoint(p);

			x2 = y2 = -1;
			System.out.println(e.getX() + "," + e.getY());
			plotDot(e.getX(), e.getY(), (Graphics2D) getGraphics());

			didUndo = false;
		}

		public void mouseEntered(MouseEvent e) {
		}

		public void mouseExited(MouseEvent e) {
		}

		public void mousePressed(MouseEvent e) {
		}

		public void mouseReleased(MouseEvent e) {
		}

		public void mouseDragged(MouseEvent e) {
			drag(e);
		}

		private void drag(MouseEvent e) {
			if (startX == -1 || startY == -1) {
				return;
			}
			Graphics2D g2 = (Graphics2D) getGraphics();
			g2.setXORMode(getBackground());
			setForeground(Color.green);
			if (x2 != -1) {
				// erase
				g2.drawLine(startX, startY, x2, y2);
			}
			x2 = e.getX();
			y2 = e.getY();
			g2.drawLine(startX, startY, x2, y2);
		}

		public void mouseMoved(MouseEvent e) {
			drag(e);
		}
	}// ;

	class MosaicSelectComponent extends Component implements MouseListener,
			MouseMotionListener {
		private static final long serialVersionUID = 1L;
		ImageLoader loader;
		Dimension compSize;
		int ncols;
		int nrows;
		int ch, cw;
		int mch, mcw;
		int selRow = -1, selCol = -1;
		PropertyChangeSupport pcs;
		Set<String> selectedSet = new HashSet<String>();

		public MosaicSelectComponent(ImageLoader loader) {
			this.loader = loader;
			this.ncols = loader.getCols();
			this.nrows = loader.getRows();
			BufferedImage cellBI = loader.getCell(0, 0);
			this.ch = cellBI.getHeight(null);
			this.cw = cellBI.getWidth(null);
			mch = ch + 4;
			mcw = cw + 4;

			int h = mch * nrows;
			int w = mcw * ncols;
			compSize = new Dimension(w, h);

			addMouseListener(this);
			addMouseMotionListener(this);
			setBackground(Color.white);
			pcs = new PropertyChangeSupport(this);
		}

		public Dimension getPreferredSize() {
			return compSize;
		}

		public Dimension getMaximumSize() {
			return compSize;
		}

		public Dimension getMinimumSize() {
			return compSize;
		}

		public void paint(Graphics g) {
			Graphics2D g2 = (Graphics2D) g;
			BufferedImage cellBI = null;
			for (int i = 0; i < nrows; i++) {
				int dy = i * mch;
				for (int j = 0; j < ncols; j++) {
					int dx = j * mcw;
					cellBI = loader.getCell(i, j);
					if (cellBI != null) {
						g2.drawImage(cellBI, dx + 1, dy + 1, dx + cw + 1, dy
								+ ch + 1, 0, 0, cw, ch, null);
					}
				}
			}
		}

		public int[] getSelectedCellLoc(int x, int y) {
			int row = -1;
			int col = -1;
			row = y / mch;
			col = x / mcw;
			return new int[] { row, col };
		}

		void highlite(int[] selectedCellLoc) {
			int row = selectedCellLoc[0];
			int col = selectedCellLoc[1];
			if (!loader.isValidCell(row, col)) {
				return;
			}
			if (selRow == row && selCol == col) {
				return;
			}

			String key = row + ":" + col;
			if (selectedSet.contains(key)) {
				return;
			}
			Graphics2D g2 = (Graphics2D) getGraphics();
			// g2.setXORMode(getBackground());
			int x = -1, y = -1;
			if (selRow != -1) {
				g2.setColor(getBackground());
				x = selCol * mcw;
				y = selRow * mch;
				// System.out.println("erase:: x=" + x + ",y=" + y);
				g2.drawRect(x, y, cw + 1, ch + 1);
			}
			g2.setColor(Color.red);

			x = col * mcw;
			y = row * mch;
			selRow = row;
			selCol = col;
			// System.out.println("x=" + x + ",y=" + y);
			g2.drawRect(x, y, cw + 1, ch + 1);
		}

		void select(int row, int col) {
			if (!loader.isValidCell(row, col)) {
				return;
			}
			String key = row + ":" + col;
			selectedSet.add(key);
			Graphics2D g2 = (Graphics2D) getGraphics();

			g2.setColor(Color.blue);
			int x = -1, y = -1;
			x = col * mcw;
			y = row * mch;
			g2.drawRect(x - 1, y - 1, cw + 3, ch + 3);
		}

		void deselect(int row, int col) {
			if (!loader.isValidCell(row, col)) {
				return;
			}
			String key = row + ":" + col;
			selectedSet.remove(key);
			Graphics2D g2 = (Graphics2D) getGraphics();
			g2.setColor(getBackground());
			int x = -1, y = -1;
			x = col * mcw;
			y = row * mch;
			g2.drawRect(x - 1, y - 1, cw + 3, ch + 3);
		}

		public void mouseClicked(MouseEvent e) {
			int[] selectedCellLoc = getSelectedCellLoc(e.getX(), e.getY());
			System.out.println("Selected Cell:" + selectedCellLoc[0] + ","
					+ selectedCellLoc[1]);
			int row = selectedCellLoc[0];
			int col = selectedCellLoc[1];
			if (!loader.isValidCell(row, col)) {
				return;
			}

			pcs.firePropertyChange(new PropertyChangeEvent(this,
					"selectedCell", null, new int[] { row, col }));
		}

		public void mouseEntered(MouseEvent e) {
			setCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
		}

		public void mouseExited(MouseEvent e) {
			setCursor(Cursor.getDefaultCursor());
		}

		public void mousePressed(MouseEvent e) {
		}

		public void mouseReleased(MouseEvent e) {
		}

		public void mouseDragged(MouseEvent e) {
			int[] selectedCellLoc = getSelectedCellLoc(e.getX(), e.getY());
			highlite(selectedCellLoc);
		}

		public void mouseMoved(MouseEvent e) {
			int[] selectedCellLoc = getSelectedCellLoc(e.getX(), e.getY());
			highlite(selectedCellLoc);
		}

		public void addPropertyChangeListener(PropertyChangeListener listener) {
			pcs.addPropertyChangeListener(listener);
		}
	}// ;

	public static class ImageLoader {
		Map<URL, BufferedImage> biMap = new LinkedHashMap<URL, BufferedImage>();
		BufferedImage[][] biMat;
		int cols;
		int rows;

		public ImageLoader(List<URL> imgSrcList, int rows, int cols)
				throws IOException {
			this.rows = rows;
			this.cols = cols;
			System.out.println("ImageLoader:: rows:" + rows + ",cols:" + cols);
			biMat = new BufferedImage[rows][cols];
			BufferedImage bi = null;
			try {
				int i = 0;
				for (URL imageSrc : imgSrcList) {
					bi = ImageIO.read(imageSrc);
					if (bi == null) {
						System.out.println("bi is null for imageSrc:"
								+ imageSrc);
					} else {
						System.out.println(imageSrc + " h:" + bi.getHeight()
								+ " c:" + bi.getWidth());
					}
					int ir = i / cols;
					int ic = i % cols;
					// System.out.println("ir=" + ir + ", ic=" + ic);
					biMat[ir][ic] = bi;
					biMap.put(imageSrc, bi);
					i++;
				}
			} catch (IOException iox) {
				String errMsg = "Image cannot be read:" + iox.getMessage();
				System.out.println(errMsg);
				throw iox;
			}
		}

		public int getNumImages() {
			return biMap.size();
		}

		public BufferedImage getImage(URL imgSrc) {
			return biMap.get(imgSrc);
		}

		public BufferedImage getCell(int row, int col) {
			return biMat[row][col];
		}

		public int getSelectedSliceNo(int row, int col) {
			return row * cols + col + 1;
		}

		public int getCols() {
			return cols;
		}

		public int getRows() {
			return rows;
		}

		public boolean isValidCell(int row, int col) {
			if (row >= biMat.length || row < 0)
				return false;
			if (col >= biMat[0].length || col < 0)
				return false;
			return biMat[row][col] != null;
		}
	}// ;

	class MosaicSelectionPanel extends JPanel implements ActionListener,
			PropertyChangeListener {
		private static final long serialVersionUID = 1L;
		MosaicSelectComponent msComp;
		JRadioButton selectModeRB;
		JRadioButton deselectModeRB;
		JButton nextButton;
		PropertyChangeSupport pcs;
		List<ImageLoc> selectionList = new ArrayList<ImageLoc>(10);

		public MosaicSelectionPanel(ImageLoader loader) {
			setLayout(new BorderLayout(5, 5));
			setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
			JPanel mainPanel = new JPanel(new BorderLayout(5, 5));
			mainPanel.setBorder(BorderFactory.createTitledBorder("Slices"));

			selectModeRB = new JRadioButton("Slice Selection Mode");
			deselectModeRB = new JRadioButton("Slice Deselection Mode");
			ButtonGroup group = new ButtonGroup();
			group.add(selectModeRB);
			group.add(deselectModeRB);
			selectModeRB.addActionListener(this);
			deselectModeRB.addActionListener(this);
			selectModeRB.setSelected(true);

			nextButton = new JButton("Continue");
			nextButton.addActionListener(this);
			nextButton.setEnabled(false);

			JPanel mcPanel = new JPanel(new BorderLayout(5, 5));
			JLabel msgLabel = new JLabel(
					"Please select a slice for ROI by clicking on it.");
			JPanel controlPanel = new JPanel(null);
			BoxLayout bl = new BoxLayout(controlPanel, BoxLayout.X_AXIS);
			controlPanel.setLayout(bl);
			controlPanel.add(Box.createHorizontalStrut(5));
			controlPanel.add(msgLabel);
			controlPanel.add(Box.createHorizontalGlue());
			mcPanel.add(controlPanel, BorderLayout.CENTER);

			JPanel rbgPanel = new JPanel(new GridLayout(1, 0));
			rbgPanel.setBorder(BorderFactory
					.createEtchedBorder(EtchedBorder.LOWERED));
			rbgPanel.add(selectModeRB);
			rbgPanel.add(deselectModeRB);

			mcPanel.add(rbgPanel, BorderLayout.NORTH);

			JPanel buttonPanel = new JPanel(null);
			bl = new BoxLayout(buttonPanel, BoxLayout.X_AXIS);
			buttonPanel.setLayout(bl);
			buttonPanel.add(Box.createHorizontalGlue());
			buttonPanel.add(nextButton);
			buttonPanel.add(Box.createHorizontalStrut(5));
			mcPanel.add(buttonPanel, BorderLayout.SOUTH);

			msComp = new MosaicSelectComponent(loader);
			mainPanel.add(msComp, BorderLayout.CENTER);
			add(mainPanel, BorderLayout.CENTER);

			add(mcPanel, BorderLayout.SOUTH);

			pcs = new PropertyChangeSupport(this);
			msComp.addPropertyChangeListener(this);
		}

		public void addPropertyChangeListener(PropertyChangeListener listener) {
			// for single ROI selection
			// msComp.addPropertyChangeListener(listener);
			pcs.addPropertyChangeListener(listener);
		}

		public void actionPerformed(ActionEvent e) {
			if (e.getSource() == nextButton) {
				// time for ROI selection
				pcs.firePropertyChange(new PropertyChangeEvent(this,
						"selectedCellList", null, selectionList));
			}

		}

		public void propertyChange(PropertyChangeEvent evt) {
			System.out.println("propertyChange:" + evt.getPropertyName());
			if (evt.getPropertyName().equals("selectedCell")) {
				int[] selectedCellLoc = (int[]) evt.getNewValue();
				int row = selectedCellLoc[0];
				int col = selectedCellLoc[1];
				if (deselectModeRB.isSelected()) {
					for (Iterator<ImageLoc> it = selectionList.iterator(); it
							.hasNext();) {
						ImageLoc il = it.next();
						if (il.row == row && il.col == col) {
							it.remove();
							// remove highlite
							msComp.deselect(row, col);
						}
					}
				} else {
					boolean inAlready = false;
					for (ImageLoc il : selectionList) {
						if (il.row == row && il.col == col) {
							inAlready = true;
							break;
						}
					}
					if (!inAlready) {
						selectionList.add(new ImageLoc(row, col));
						// add highlite
						msComp.select(row, col);
					}
				}
				if (selectionList.isEmpty()) {
					nextButton.setEnabled(false);
				} else {
					nextButton.setEnabled(true);
				}
			}
		}
	}// ;

	class ImageLoc {
		int row;
		int col;

		public ImageLoc(int row, int col) {
			this.row = row;
			this.col = col;
		}
	}// ;

	public static void main(String[] args) throws Exception {
		JFrame frame = new JFrame("CBF ROI");

		frame.setSize(500, 520);
		Dimension dim = new Dimension(500, 520);
		frame.setPreferredSize(dim);
		frame.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				System.exit(0);
			}
		});
		URL imageSrc = null;
		File f = new File(System.getProperty("user.home")
				+ "/Downloads/brain.png");
		// f = new File(System.getProperty("user.home") +
		// "/download_cache/000838972841_1_1270674019975/matlab_out/csf5.TIF");
		f = new File(
				System.getProperty("user.home")
						+ "/tomcat-6.0.26/webapps/cbfbirn_test/1283215788976_bozyurt/csf10.png");
		imageSrc = f.toURI().toURL();
		int numImages = 23;
		int numRows = 4;
		int numCols = 6;
		List<URL> images = new ArrayList<URL>(numImages);
		for (int i = 0; i < numImages; i++) {
			images.add(imageSrc);
		}
		ImageLoader il = new ImageLoader(images, numRows, numCols);

		CBFMultipleROIApplet appl = new CBFMultipleROIApplet(il);
		appl.buildMessagePanel();
		appl.buildUI();
		frame.add(appl, BorderLayout.CENTER);
		frame.pack();
		frame.setVisible(true);
	}

}
