package clinical.web.workflow.cbf;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import clinical.server.vo.Deriveddata;
import clinical.utils.Assertion;
import clinical.utils.FileUtils;
import clinical.web.IAppConfigService;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.download.FileBundleConfig;
import clinical.web.download.Packager;
import clinical.web.download.PathWrapper;
import clinical.web.exception.BaseException;
import clinical.web.scheduler.JobException;
import clinical.web.vo.upload.VisitInfo;
import clinical.web.workflow.common.WFGenUtils;

public class UCLACBFProcessingJob extends AbstractCBFProcessingJob {
	protected UCLACBFProcHelper helper;

	public UCLACBFProcessingJob(String id, UserInfo ui, String description,
			CBFWFContext context, String templateDir, String matlabDir,
			String emailTo, VisitProcessInfo vpi) throws BaseException {
		super();
		this.id = id;
		this.ui = ui;
		this.description = description;
		this.context = context;
		this.templateDir = templateDir;
		this.matlabDir = matlabDir;
		this.emailTo = emailTo;

		this.helper = new UCLACBFProcHelper(ui, context, templateDir);

		VisitInfo vi = context.getViList().get(0);
		String subjectID = vi.getSubjectID();

		String bundleName = subjectID + "_Visit_"
				+ sdf.format(vi.getVisitDate()) + "_" + vi.getVisitId();
		File f = new File(context.getCacheDir(), bundleName + ".tar.gz");
		this.resultTarFile = f.getAbsolutePath();

		this.vpi = vpi;
	}

	@Override
	public void execute(int stageId) throws JobException {
		try {
			handle();
		} catch (Exception e) {
			e.printStackTrace();
			throw new JobException(e);
		}

	}

	protected void handle() throws Exception {
		IAppConfigService configService = ServiceFactory.getAppConfigService();

		prepareEnvironment(configService);

		File matlabDirFile = new File(matlabDir);
		sendMessage("preparing for CBF processing");
		this.vpi = helper.prepare4Job(matlabDirFile);
		if (isCanceled()) {
			return;
		}
		Assertion.assertTrue(!vpi.getCpiList().isEmpty());

		for (int i = 0; i < vpi.getCpiList().size(); i++) {
			CBFProcessInfo cpi = vpi.getCpiList().get(i);
			if (isCanceled()) {
				return;
			}
			sendMessage("processing " + cpi.getCbfSI().getProtocolId());
			helper.handleUCLACBF(cpi, matlabDirFile,
					context.isDoSkullStripping(), context.getGmThreshold(),
					context.getGaussianFilterParam());
		}

		// save derived data and create a bundle of results for download
		postprocess();
	}
	
	protected void postprocess() throws Exception {
		// derived data persistence
		//TODO check
		List<Deriveddata> ddList = helper.saveDerivedData(this.vpi);

		// bundle the results for download
		List<Deriveddata> downloadDDList = new ArrayList<Deriveddata>(10);
		for (Deriveddata dd : ddList) {
			if (dd.getDatauri().endsWith(".mat")
					|| dd.getDatauri().endsWith(".png")
					|| dd.getDatauri().indexOf("intermediate") != -1
					|| dd.getDatauri().indexOf("CBF+orig") != -1
					|| dd.getDatauri().indexOf("MPPCASL_Mask+orig") != -1
					|| dd.getDatauri().indexOf("process_summary.txt") != -1
					|| dd.getDatauri().indexOf("cbfbirn_path1_summary.txt") != -1) {
				downloadDDList.add(dd);
			}
		}

		VisitInfo vi = context.getViList().get(0);
		String subjectID = vi.getSubjectID();

		String bundleName = subjectID + "_Visit_"
				+ sdf.format(vi.getVisitDate()) + "_" + vi.getVisitId();

		List<PathWrapper> allFiles = new ArrayList<PathWrapper>();
		String srcRootDir = null;
		for (Deriveddata dd : downloadDDList) {
			String datauri = dd.getDatauri();
			if (srcRootDir == null) {
				srcRootDir = WFGenUtils.extractRootDir(datauri, subjectID);
			}
			String relativePath = FileUtils
					.getRelativePath(srcRootDir, datauri);
			PathWrapper pw = new PathWrapper(relativePath, srcRootDir);
			allFiles.add(pw);
		}
		// final packaging

		Packager p = new Packager(context.getCacheDir(), bundleName,
				FileBundleConfig.GZIPPED);
		p.includeFiles(allFiles);
		p.pack();

		if (isCanceled()) {
			return;
		}
		handleEmail(JOB_FINISHED_MAIL);
	}

	protected void prepareEnvironment(IAppConfigService configService) {
		String envPath = configService.getParamValue("cbfbirn.env.path");
		helper.setEnvPath(envPath);
		helper.addEnvironmentVar("FSLDIR",
				configService.getParamValue("FSLDIR"));
		helper.addEnvironmentVar("FSLOUTPUTTYPE",
				configService.getParamValue("FSLOUTPUTTYPE"));
		helper.addEnvironmentVar("AFNI_PLUGINPATH",
				configService.getParamValue("AFNI_PLUGINPATH"));
		helper.addEnvironmentVar("FSLMULTIFILEQUIT",
				configService.getParamValue("FSLMULTIFILEQUIT"));
		helper.addEnvironmentVar("FSLCONFDIR",
				configService.getParamValue("FSLCONFDIR"));
		helper.addEnvironmentVar("FSLMACHTYPE",
				configService.getParamValue("FSLMACHTYPE"));
	}

	@Override
	public int getNumberOfStages() {
		return 1;
	}

}
