package clinical.web.workflow.cbf;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import clinical.utils.FileUtils;
import clinical.web.vo.JobProvenanceInfo;

/**
 * Represents the output generated by a step in a CBF workflow. This class is
 * mainly used to skip time consuming downstream steps in a CBF workflow using
 * output from corresponding steps from an previous run on the same data.
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class WFStep {
	protected String name;
	/** AFNI BRIKs that are generated by this workflow step */
	protected List<AFNIBrik> briks = new ArrayList<AFNIBrik>(5);
	protected List<File> otherFiles = new ArrayList<File>(5);
	/**
	 * the job provenance info for the job that reused data comes from
	 */
	protected JobProvenanceInfo jpi;

	public WFStep(String name, JobProvenanceInfo jpi) {
		super();
		this.name = name;
		this.jpi = jpi;
	}

	public String getName() {
		return name;
	}

	public void addBrik(AFNIBrik brik) {
		briks.add(brik);
	}
	
	public List<AFNIBrik> getBriks() {
		return briks;
	}

	public List<File> getOtherFiles() {
		return otherFiles;
	}

	/**
	 * copies the previously generated output represented by this
	 * <code>WFStep</code> object to the provided destination.
	 * 
	 * @param destDir
	 * @param overwrite
	 *            if true overwrites the previous file with the same name in the
	 *            destination.
	 * @throws Exception
	 */
	public void copyToDest(File destDir, boolean overwrite) throws Exception {
		for (AFNIBrik brik : briks) {
			File brikFile = brik.getBrikFile();
			File headerFile = brik.getHeaderFile();
			File brikDestFile = new File(destDir, brikFile.getName());
			File headDestFile = new File(destDir, headerFile.getName());
			if ((brikDestFile.exists() || headDestFile.exists()) && !overwrite) {
				continue;
			}
			FileUtils.copyFile(brikFile.getCanonicalPath(), brikDestFile
					.getCanonicalPath());
			FileUtils.copyFile(headerFile.getCanonicalPath(), headDestFile
					.getCanonicalPath());
		}
		for (File f : otherFiles) {
			File destFile = new File(destDir, f.getName());
			if (destFile.exists() && !overwrite) {
				continue;
			}
			FileUtils.copyFile(f.getCanonicalPath(), destFile
					.getCanonicalPath());
		}
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder(256);
		sb.append("WFStep::[");
		sb.append("name:").append(name);
		for (AFNIBrik brik : briks) {
			sb.append("\n\t").append(brik.toString());
		}
		for (File f : otherFiles) {
			sb.append("\n\tFile:").append(f);
		}
		sb.append(']');
		return sb.toString();
	}

	public JobProvenanceInfo getJpi() {
		return jpi;
	}
}