package clinical.web.workflow.cbf.group;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;

import clinical.server.vo.GroupAnalysisData;
import clinical.server.vo.Jobs;
import clinical.utils.Assertion;
import clinical.utils.DateTimeUtils;
import clinical.utils.Executor;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.CBFBIRNConstants;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.common.vo.Factor;
import clinical.web.exception.BaseException;
import clinical.web.scheduler.JobScheduler;
import clinical.web.services.ICBFROIGroupAnalysisService;
import clinical.web.services.ICBFStandardSpaceGroupAnalysisService;
import clinical.web.services.IJobProvenanceService;
import clinical.web.vo.CBFROIJobAssociation.JobVisitInfo;
import clinical.web.vo.GroupAnalysisCBFJobInfo;
import clinical.web.vo.GroupDerivedDataInfo;
import clinical.web.vo.JobProvenanceInfo;
import clinical.web.vo.JobProvenanceInfo.JobProvenanceParamInfo;
import clinical.web.workflow.cbf.AFNIBrik;
import clinical.web.workflow.common.WFGenUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class GroupAnalysisHelper {
	protected UserInfo ui;
	protected GroupAnalysisContext context;
	protected String templateDir;
	protected String envPath;
	protected Map<String, String> envMap = new HashMap<String, String>(17);
	private boolean canceled = false;
	protected IDBCache dbCache;
	protected ICBFStandardSpaceGroupAnalysisService ssgaService;

	protected static Log _log = LogFactory.getLog(GroupAnalysisHelper.class);

	public GroupAnalysisHelper(UserInfo ui, GroupAnalysisContext context)
			throws BaseException {
		super();
		this.ui = ui;
		this.context = context;
		this.templateDir = context.getTemplateDir();

		dbCache = ServiceFactory.getDBCache(context.getDbID());
		ssgaService = ServiceFactory
				.getCBFStandardSpaceGroupAnalysisService(context.getDbID());
	}

    public String getEnvPath() {
		return envPath;
	}

	public void setEnvPath(String envPath) {
		this.envPath = envPath;
		_log.info("envPath set:" + envPath);
	}

	public void addEnvironmentVar(String name, String value) {
		envMap.put(name, value);
	}

	public void setCanceled(boolean canceled) {
		this.canceled = canceled;
	}

	public boolean isCanceled() {
		return canceled;
	}

	public File prepareInputData() throws Exception {
		File inputRootDir = new File(context.getCacheDir(), "input");
		if (!inputRootDir.mkdir()) {
			_log.error("cannot create standard space group analysis input directory:"
					+ inputRootDir);
			throw new Exception(
					"cannot create standard space group analysis input directory!");
		}
		ssgaService.prepareProcessInputData(ui, inputRootDir,
				context.getJiList());
		return inputRootDir;
	}

	public void handleStandardSpaceAnalysis(SSGAProcessInfo ssgapi,
			File matlabDir, File derivedDir, boolean doQWarp) throws Exception {
		Properties p = new Properties();
		p.setProperty("file.resource.loader.path", this.templateDir);
		p.setProperty("runtime.log.logsystem.class",
				"org.apache.velocity.runtime.log.NullLogSystem");
		Velocity.init(p);
		VelocityContext ctx = new VelocityContext();

		File outputDir = new File(this.context.getCacheDir(), "output");
		if (!outputDir.isDirectory()) {
			outputDir.mkdir();
		}

		File intermediateMatFile = new File(ssgapi.getInputCSVFile()
				.getParentFile(), "cbf.mat");

		List<Factor> selectedFactors = context.getSelectedFactors();

		int qwarp = context.isDoQWarp() ? 1 : 0;
		
		ctx.put("inputCsv", ssgapi.getInputCSVFile().getAbsolutePath());
		ctx.put("assessmentCsv", ssgapi.getAssessmentCSVFile()
				.getAbsolutePath());
		ctx.put("outdir", outputDir.getAbsolutePath());
		ctx.put("matlabdir", matlabDir);
		ctx.put("cbfMatFile", intermediateMatFile.getAbsolutePath());
		ctx.put("doQWarp", qwarp);

		StringWriter sw = new StringWriter();
		Template template = Velocity.getTemplate("do_ssga.vm");
		template.merge(ctx, sw);

		System.out.println(sw.toString());
		File matlabDriverScriptFile = new File(outputDir, "do_ssga.m");
		matlabDriverScriptFile.delete();
		FileUtils.save2File(sw.toString(),
				matlabDriverScriptFile.getAbsolutePath());

		Executor executor = new Executor(CBFBIRNConstants.MATLAB_EXEC, true);
		String cmdLine = "-nodesktop -nosplash < "
				+ matlabDriverScriptFile.getAbsolutePath();

		runProcess(executor, cmdLine);
		File outCSVFile = null;
		File[] outFiles = outputDir.listFiles();
		for (File outFile : outFiles) {
			if (outFile.getName().equals("cbf.csv")) {
				outCSVFile = outFile;
				break;
			}
		}
		Assertion.assertNotNull(outCSVFile);

		augmentSSGAStatsFile(outCSVFile, ssgapi.getAssessmentCSVFile(),
				ssgapi.getOutCSVFile(), true);

		// run genstats
		ctx = new VelocityContext();
		ctx.put("pathType", 3); // for standard space analysis
		ctx.put("repeated", context.isRepeatedMeasures() ? new Integer(1)
				: new Integer(0));
		ctx.put("factors", Factor.toFactorsArgString(selectedFactors));
		ctx.put("levels", Factor.toLevelsArgString(selectedFactors));

		ctx.put("inputCsv", ssgapi.getOutCSVFile().getAbsolutePath());
		ctx.put("outdir", outputDir.getAbsolutePath());
		ctx.put("matlabdir", matlabDir);

		sw = new StringWriter();
		template = Velocity.getTemplate("do_genstats.vm");
		template.merge(ctx, sw);

		String genstatsInCSVFile = ssgapi.getOutCSVFile().getAbsolutePath();
		FileUtils.copyFile(genstatsInCSVFile, genstatsInCSVFile + ".bak");

		System.out.println(sw.toString());
		matlabDriverScriptFile = new File(outputDir, "do_genstats.m");
		matlabDriverScriptFile.delete();
		FileUtils.save2File(sw.toString(),
				matlabDriverScriptFile.getAbsolutePath());

		cmdLine = "-nodesktop -nosplash < "
				+ matlabDriverScriptFile.getAbsolutePath();

		runProcess(executor, cmdLine);

		// now build the final CSV file
		augmentSSGAStatsFile(outCSVFile, ssgapi.getAssessmentCSVFile(),
				ssgapi.getOutCSVFile(), false);

		outFiles = outputDir.listFiles();

		Map<String, AFNIBrik> brikMap = new LinkedHashMap<String, AFNIBrik>();
		for (File outFile : outFiles) {
			String name = outFile.getName();
			if (name.indexOf("+tlrc") != -1 && AFNIBrik.isAFNIBrikFile(name)) {
				String key = AFNIBrik.getSeriesName(name);
				AFNIBrik afniBrik = brikMap.get(key);
				afniBrik = AFNIBrik.create(outFile, afniBrik);
				brikMap.put(key, afniBrik);
			} else if (name.endsWith(".txt")) {
				File destFile = new File(derivedDir, name);
				FileUtils.copyFile(outFile.getAbsolutePath(),
						destFile.getAbsolutePath());
				ssgapi.addResultFile(destFile);
			}
		}
		for (AFNIBrik brik : brikMap.values()) {
			List<File> brikFiles = brik.copyTo(derivedDir, true);
			AFNIBrik destBrik = AFNIBrik.create(brikFiles);
			ssgapi.addResultBrik(destBrik);
		}

		// intermediate TLRC briks 06/03/2013
		brikMap.clear();
		File inputDir = new File(ssgapi.getInputCSVFile().getParentFile()
				.getAbsolutePath());
		List<File> tlrcBrikFiles = FileUtils.getAllFilesMatching(inputDir,
				new FilenameFilter() {
					@Override
					public boolean accept(File dir, String name) {
						return name.indexOf("+tlrc") != -1
								&& AFNIBrik.isAFNIBrikFile(name);
					}
				});

		for (File brikFile : tlrcBrikFiles) {
			String name = brikFile.getName();
			String key = brikFile.getParentFile().getCanonicalPath() + "/"
					+ AFNIBrik.getSeriesName(name);
			AFNIBrik afniBrik = brikMap.get(key);
			afniBrik = AFNIBrik.create(brikFile, afniBrik);
			brikMap.put(key, afniBrik);
		}
		File intermediateDir = new File(derivedDir, "intermediate");
		intermediateDir.mkdirs();
		for (AFNIBrik brik : brikMap.values()) {
			File parentDir = brik.getBrikFile().getParentFile();
			String relativePath = FileUtils.getRelativePath(
					inputDir.getAbsolutePath(), parentDir.getAbsolutePath());
			File destDir = new File(intermediateDir, relativePath);
			if (!destDir.isDirectory() && !destDir.mkdirs()) {
				_log.error("cannot create intermediate dir:" + destDir);
				throw new Exception(
						"Cannot create an intermediate data directory!");
			}
			List<File> brikFiles = brik.copyTo(destDir, true);
			AFNIBrik destBrik = AFNIBrik.create(brikFiles);
			ssgapi.addIntermediateTLRCBrik(destBrik);
		}

		// generate activation maps
		List<String> brikLevelNames = new ArrayList<String>(10);
		for (File outFile : outFiles) {
			String name = outFile.getName();
			if (name.startsWith("3dmerged_anat_")
					&& AFNIBrik.isAFNIBrikFile(name)) {
				String brikLevelName = name.replaceFirst("^3dmerged_anat_", "");
				brikLevelName = brikLevelName.replaceFirst("\\+tlrc\\.\\w+$",
						"");
				brikLevelNames.add(brikLevelName);
			}
		}
		outFiles = outputDir.listFiles();
		JobScheduler scheduler = JobScheduler.getInstance();
		String afniActivationMapScript = context.getAfniActivationMapScript();
		for (String brikLevelName : brikLevelNames) {
			executor = new Executor(afniActivationMapScript);
			// "/usr/local/bin/no_head_afni_activation_map.sh");
			executor.setErrorStreamReliable(false);
			cmdLine = outputDir.getAbsolutePath() + " " + brikLevelName;
			_log.info("running  " + afniActivationMapScript + " " + cmdLine);
			// only one user is allowed at this point all other concurrent users
			// doing activation map image generation must wait
			synchronized (scheduler) {
				runProcess(executor, cmdLine);
			}
		}
		outFiles = outputDir.listFiles();
		for (File outFile : outFiles) {
			String name = outFile.getName();
			if (name.endsWith(".png")) {
				File destFile = new File(derivedDir, name);
				FileUtils.copyFile(outFile.getAbsolutePath(),
						destFile.getAbsolutePath());
				ssgapi.addResultFile(destFile);
			}
		}

	}

	public void augmentSSGAStatsFile(File outCSVFile, File assessmentCSVFile,
			File finalOutCSVFile, boolean addTLRCPath) throws Exception {

		List<List<String>> asRows = WFGenUtils.loadCSV(assessmentCSVFile);
		List<List<String>> outRows = WFGenUtils.loadCSV(outCSVFile);

		List<CSVColumnMapping> headerMappings = prepHeaderMappings(
				outRows.get(0), asRows.get(0), addTLRCPath);

		Map<String, JobVisitInfo> jviMap = new HashMap<String, JobVisitInfo>();
		List<GroupAnalysisCBFJobInfo> jiList = this.context.getJiList();
		for (GroupAnalysisCBFJobInfo ji : jiList) {
			JobVisitInfo jvi = ji.getJvi();
			String key = jvi.getSubjectID() + ":" + jvi.getJobID();
			jviMap.put(key, jvi);
		}

		Map<String, List<String>> asRowMap = new HashMap<String, List<String>>();
		for (List<String> row : asRows) {
			// subjectID ':' expName ':' visitID
			String key = row.get(0) + ":" + row.get(1) + ":" + row.get(2);
			asRowMap.put(key, row);
		}

		List<String> augHeadRow = new ArrayList<String>(headerMappings.size());
		for (CSVColumnMapping mapping : headerMappings) {
			augHeadRow.add(mapping.destKey);
		}

		List<List<String>> augRows = new ArrayList<List<String>>(outRows.size());
		augRows.add(augHeadRow);

		for (int i = 1; i < outRows.size(); i++) {
			List<String> row = outRows.get(i);
			String subjectID = row.get(0);
			String jobID = row.get(8); // 9th column
			String key = subjectID + ":" + jobID;
			JobVisitInfo jvi = jviMap.get(key);
			Assertion.assertNotNull(jvi);
			List<String> augRow = new ArrayList<String>(headerMappings.size());
			augRows.add(augRow);

			String expName = jvi.getExpName();
			int visitID = jvi.getVisitID();
			String visitDate = jvi.getVisitDate();
			String asKey = subjectID + ":" + expName + ":" + visitID;
			List<String> asRow = asRowMap.get(asKey);
			// Assertion.assertNotNull(asRow);
			for (CSVColumnMapping mapping : headerMappings) {
				if (mapping.srcType == CSVColumnMapping.OTHER) {
					if (mapping.srcKey.equals("SubjectID")) {
						augRow.add(subjectID);
					} else if (mapping.srcKey.equals("JobID")) {
						augRow.add(jobID);
					} else if (mapping.srcKey.equals("Project")) {
						augRow.add(expName);
					} else if (mapping.srcKey.equals("VisitID")) {
						augRow.add(String.valueOf(visitID));
					} else if (mapping.srcKey.equals("Visit Date")) {
						augRow.add(visitDate);
					}
				} else if (mapping.srcType == CSVColumnMapping.SSGA) {
					String srcValue = row.get(mapping.srcIdx);
					srcValue = srcValue.equals("NaN") ? "" : srcValue;
					augRow.add(srcValue);
				} else {
					// CSVColumnMapping.ASSESSMENT
					if (asRow != null) {
						String value = null;
						if (asRow.size() <= mapping.srcIdx) {
							_log.warn("asRow size:" + asRow.size()
									+ " mapping.srcIdx:" + mapping.srcIdx);
							GenUtils.showRow("AssessmentRow", asRow);
						} else {
							value = asRow.get(mapping.srcIdx);
							value = WFGenUtils.replaceCommas(value, ";");
						}
						augRow.add(value != null ? value : ".");
					} else {
						// missing value
						augRow.add(".");
					}
				}
			}
		} // i

		// sort by the subject id
		List<String> headerRow = augRows.remove(0);
		Collections.sort(augRows, new Comparator<List<String>>() {
			@Override
			public int compare(List<String> o1, List<String> o2) {
				return o1.get(0).compareTo(o2.get(0));
			}
		});
		augRows.add(0, headerRow);

		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(finalOutCSVFile));
			for (List<String> augRow : augRows) {
				String csvRow = GenUtils.join(augRow, ",");
				out.write(csvRow);
				out.newLine();
			}
		} finally {
			FileUtils.close(out);
		}
	}

	protected List<CSVColumnMapping> prepHeaderMappings(
			List<String> outHeaderRow, List<String> asHeaderRow,
			boolean addTLRCPath) {
		// SubjectID, JobID, Exp Name, Visit ID, <subject information> GM CBF,
		// <rest of assessments>
		List<CSVColumnMapping> mappings = new ArrayList<CSVColumnMapping>(
				asHeaderRow.size() + 5);
		mappings.add(new CSVColumnMapping(outHeaderRow.get(0), outHeaderRow
				.get(0), 0, 0, CSVColumnMapping.OTHER));
		mappings.add(new CSVColumnMapping("JobID", "JobID", 1, 1,
				CSVColumnMapping.OTHER));
		mappings.add(new CSVColumnMapping("Project", "Project", 2, 2,
				CSVColumnMapping.OTHER));
		mappings.add(new CSVColumnMapping("VisitID", "VisitID", 3, 3,
				CSVColumnMapping.OTHER));
		mappings.add(new CSVColumnMapping("Visit Date", "Visit Date", 4, 4,
				CSVColumnMapping.OTHER));
		int startIdx = -1;
		int i = 0;
		for (String scoreHeader : asHeaderRow) {
			if (scoreHeader.startsWith("Subject_Information__")) {
				startIdx = i;
				break;
			}
			i++;
		}
		int destOffset = 5;
		if (startIdx != -1) {
			int j = 0;
			for (i = startIdx; i < asHeaderRow.size(); i++) {
				String scoreHeader = asHeaderRow.get(i);
				if (!scoreHeader.startsWith("Subject_Information__")) {
					destOffset += j;
					break;
				}
				mappings.add(new CSVColumnMapping(scoreHeader, scoreHeader, i,
						destOffset + j, CSVColumnMapping.ASSESSMENT));
				j++;
			}
		}
		// map the GM CBF
		int j = 0;
		for (i = 2; i < outHeaderRow.size(); i++) {
			String outHeader = outHeaderRow.get(i);
			if (outHeader.equals("GM_CBF")) {
				mappings.add(new CSVColumnMapping(outHeader, outHeader, i,
						destOffset + j, CSVColumnMapping.SSGA));
				j++;
				break;
			}
		}
		if (addTLRCPath) {
			for (i = 2; i < outHeaderRow.size(); i++) {
				String outHeader = outHeaderRow.get(i);
				if (outHeader.equals("Path2tlrcCBF")) {
					mappings.add(new CSVColumnMapping(outHeader, outHeader, i,
							destOffset + j, CSVColumnMapping.SSGA));
					j++;
					break;
				}
			}
		}
		destOffset += j;
		// now the rest of the assessments
		j = 0;
		for (i = 3; i < asHeaderRow.size(); i++) {
			String scoreHeader = asHeaderRow.get(i);
			if (!scoreHeader.startsWith("Subject_Information__")) {
				mappings.add(new CSVColumnMapping(scoreHeader, scoreHeader, i,
						destOffset + j, CSVColumnMapping.ASSESSMENT));
				j++;
			}
		}
		return mappings;
	}

	protected void runProcess(Executor executor, String cmdLine)
			throws Exception {
		if (isCanceled()) {
			return;
		}
		executor.addEnvParam("PATH", this.envPath);
		for (String ev : this.envMap.keySet()) {
			String value = envMap.get(ev);
			executor.addEnvParam(ev, value);
		}
		setMatlabPrefDirEnvVar(executor);
		executor.showOutput(true);
		// executor.setErrorStreamReliable(false);
		executor.execute(cmdLine);

		String output = executor.getOutput();
		System.out.println(output);
	}

	private void setMatlabPrefDirEnvVar(Executor executor) {
		File prefDir = new File(this.context.getCacheDir(), ".matlab");
		if (!prefDir.exists()) {
			prefDir.mkdirs();
		}
		Assertion.assertTrue(prefDir.exists() && prefDir.isDirectory());
		executor.addEnvParam("MATLAB_PREFDIR", prefDir.getAbsolutePath());
	}

	public List<GroupAnalysisData> saveDerivedDataWithJobProvenance(
			List<GroupDerivedDataInfo> gddiList, File derivedDir)
			throws Exception {
		IJobProvenanceService jpmService = ServiceFactory
				.getJobProvenanceService(context.getDbID());
		String gaName = context.getGaName() != null ? context.getGaName() : "";
		JobProvenanceInfo jpi = new JobProvenanceInfo(gaName, "");
		jpi.setDataURI(derivedDir.getAbsolutePath());
		Jobs theJob = jpmService.getTheJob(ui, context.getJobID(),
				ui.getPerceivedName());
		jpi.setJob(theJob);
		// add Finish Date also as provenance data
		String value = DateTimeUtils.formatDate(theJob.getJobenddate());
		JobProvenanceParamInfo jppi = new JobProvenanceParamInfo("Finish Date",
				value);
		jpi.addParam(jppi);

		ICBFROIGroupAnalysisService crgaService = ServiceFactory
				.getCBFROIGroupAnalysisService(context.getDbID());
        return crgaService
                .addDerivedDataWithJobProvenance(ui, gddiList, jpi);
	}

	public static class CSVColumnMapping {
		String srcKey;
		String destKey;
		/** column index in the source */
		int srcIdx;
		int destIdx;
		int srcType = OTHER;
		public static final int SSGA = 1;
		public static final int ASSESSMENT = 2;
		public static final int OTHER = 3;

		public CSVColumnMapping(String srcKey, String destKey, int srcIdx,
				int destIdx, int srcType) {
			super();
			this.srcKey = srcKey;
			this.destKey = destKey;
			this.srcIdx = srcIdx;
			this.destIdx = destIdx;
			this.srcType = srcType;
		}
	}
}
