package clinical.web.workflow.cbf.group;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONException;
import org.json.JSONObject;

import clinical.server.vo.GroupAnalysisData;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.CBFBIRNConstants;
import clinical.web.Constants;
import clinical.web.IAppConfigService;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.download.FileBundleConfig;
import clinical.web.download.Packager;
import clinical.web.download.PathWrapper;
import clinical.web.exception.BaseException;
import clinical.web.helpers.EmailHelper;
import clinical.web.scheduler.IJob;
import clinical.web.scheduler.IJobEventListener;
import clinical.web.scheduler.IJobFactory;
import clinical.web.scheduler.JobException;
import clinical.web.scheduler.JobRecord;
import clinical.web.scheduler.JobVisitContext;
import clinical.web.scheduler.remote.WFESLocation;
import clinical.web.vo.GroupDerivedDataInfo;
import clinical.web.workflow.cbf.CleanupUtils;

/**
 * 
 * The final destination od the results have the following path format
 * 
 * <pre>
 *    &lt;cbfbirn.data.root&gt;/analysis/roi_&lt;job ID&gt;
 * </pre>
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class GroupROIAnalysisJob implements IJob {
	private String id;
	private UserInfo ui;
	private String description;
	private boolean canceled = false;
	private String resultFile;
	private String emailTo;
	private String status;
	private GroupROIAnalysisContext context;
	private GroupROIAnalysisJobHelper helper;
	private ROIProcessInfo rpi;
	private File derivedDir;
	private String dataRoot;
	protected List<IJobEventListener> listeners = new ArrayList<IJobEventListener>(
			1);
	protected Log log = LogFactory.getLog(GroupROIAnalysisJob.class);

	public GroupROIAnalysisJob(String id, UserInfo ui, String description,
			GroupROIAnalysisContext context, String emailTo)
			throws BaseException {
		super();
		this.id = id;
		this.ui = ui;
		this.description = description;
		this.emailTo = emailTo;
		this.context = context;
		this.helper = new GroupROIAnalysisJobHelper(ui, context,
				context.getTemplateDir());

		String bundleName = "ROI_Analysis_" + id;
		File f = new File(context.getCacheDir(), bundleName + ".tar.gz");
		this.resultFile = f.getAbsolutePath();

		IAppConfigService configService = ServiceFactory.getAppConfigService();

		this.dataRoot = configService
				.getParamValue(Constants.CBFBIRN_DATA_ROOT_KEY);

		this.derivedDir = new File(dataRoot, "analysis");
		this.derivedDir = new File(this.derivedDir, "roi_" + id);
		if (!derivedDir.mkdirs()) {
			log.error("cannot create directory " + this.derivedDir);
			throw new BaseException(
					"Cannot create the directory to save the results in!");
		}
	}

	@Override
	public String getID() {
		return this.id;
	}

	@Override
	public String getUser() {
		return this.ui.getPerceivedName();
	}

	@Override
	public String getType() {
		return CBFBIRNConstants.CBF_ROI_ANALYSIS;
	}

	@Override
	public String getDescription() {
		return this.description;
	}

	@Override
	public int getNumberOfStages() {
		return 1;
	}

	@Override
	public void execute(int stageId) throws JobException {
		try {
			IAppConfigService configService = ServiceFactory
					.getAppConfigService();

			prepareEnvironment(configService);

			File inputRootDir = helper.prepareInputData();

			File inputCSVFile = new File(inputRootDir, "input.csv");
			// File cbfdapDir = new File(this.dataRoot, "cbfdap");
			// File roiMatFile = new File(cbfdapDir, "roi_labels.mat");
			File roiMatFile = new File(inputRootDir, "roi_labels.csv");

			File outCSVFile = new File(this.derivedDir, "roi_summary_"
					+ this.id + ".csv");
			this.rpi = new ROIProcessInfo(inputCSVFile, outCSVFile, roiMatFile);
			// helper.handleROIStats(rpi, new File(context.getMatlabDir()));
			helper.handleROIGroupAnalysis(rpi, new File(context.getMatlabDir()));

			postprocess();

		} catch (Exception e) {
			cleanupFailedJobDerivedDir();
			throw new JobException(e);
		}
	}

	protected void cleanupFailedJobDerivedDir() {
		FileUtils.deleteRecursively(this.derivedDir);
	}

	@Override
	public void saveResults(File file) throws JobException {
		// no op
	}

	@Override
	public String[] getResultsFiles() {
		String[] arr = new String[1];
		arr[0] = this.resultFile;
		return arr;
	}

	@Override
	public synchronized void cancel() {
		this.canceled = true;
	}

	@Override
	public synchronized boolean isCanceled() {
		return canceled;
	}

	@Override
	public String getDbID() {
		return this.context.getDbID();
	}

	@Override
	public UserInfo getUserInfo() {
		return this.ui;
	}

	@Override
	public int getDurationType() {
		return LONG;
	}

	@Override
	public void addJobEventListener(IJobEventListener listener) {
		this.listeners.add(listener);
	}

	@Override
	public void shutdown() {
		this.listeners = null;
	}

	@Override
	public List<JobVisitContext> getJobVisitContextList() {
		// no job visit context list for this type of job
		return new ArrayList<JobVisitContext>(0);
	}

	@Override
	public String getContextAsJSON() {
		try {
			JSONObject js = context.toJSON();
			return js.toString();
		} catch (JSONException e) {
			log.error("getContextAsJSON", e);
			return null;
		}
	}

	@Override
	public IJobFactory getJobFactory() {
		return new GroupROIAnalysisJobFactory();
	}

	@Override
	public void setStatus(String status) {
		this.status = status;
	}

	@Override
	public String getStatus() {
		return this.status;
	}

	@Override
	public void cleanup() {
		log.info("job cleanup");
		CleanupUtils.cleanupWorkingDir(getStatus(), context.getCacheDir(), log);
	}

	@Override
	public boolean setWFESLocation(WFESLocation location) {
		return false;
	}

	protected void postprocess() throws Exception {
		if (isCanceled()) {
			return;
		}
		List<GroupDerivedDataInfo> gddiList = new ArrayList<GroupDerivedDataInfo>(
				10);
		GroupDerivedDataInfo gddi = new GroupDerivedDataInfo(this.id, rpi
				.getOutCSVFile().getAbsolutePath());
		gddiList.add(gddi);

		if (rpi.getTableCSVFile() != null) {
			gddi = new GroupDerivedDataInfo(this.id, rpi.getTableCSVFile()
					.getAbsolutePath());
			gddiList.add(gddi);
		}
		if (rpi.getReportFile() != null) {
			gddi = new GroupDerivedDataInfo(this.id, rpi.getReportFile()
					.getAbsolutePath());
			gddiList.add(gddi);
		}

		for (File resultImgFile : rpi.getResultImages()) {
			gddi = new GroupDerivedDataInfo(this.id,
					resultImgFile.getAbsolutePath());
			gddiList.add(gddi);
		}

		gddiList.add(gddi);
		// save group analysis data
		List<GroupAnalysisData> gadList = helper
				.saveDerivedDataWithJobProvenance(gddiList, derivedDir);
		// bundle results for download
		List<GroupAnalysisData> downloadGADList = new ArrayList<GroupAnalysisData>(
				2);
		for (GroupAnalysisData gad : gadList) {
			String dataURI = gad.getDatauri();
			if (dataURI.endsWith(".csv") || dataURI.endsWith(".png")
					|| dataURI.endsWith(".txt")) {
				downloadGADList.add(gad);
			}
		}
		String bundleName = "ROI_Analysis_" + id;
		List<PathWrapper> allFiles = new ArrayList<PathWrapper>(1);
		String srcRootDir = this.derivedDir.getAbsolutePath();
		for (GroupAnalysisData gad : downloadGADList) {
			String datauURI = gad.getDatauri();
			String relativePath = FileUtils.getRelativePath(srcRootDir,
					datauURI);
			PathWrapper pw = new PathWrapper(relativePath, srcRootDir);
			allFiles.add(pw);
		}

		// final packaging
		Packager p = new Packager(context.getCacheDir(), bundleName,
				FileBundleConfig.GZIPPED);
		p.includeFiles(allFiles);
		p.pack();
		if (isCanceled()) {
			return;
		}
		handleEmail();
	}

	protected void prepareEnvironment(IAppConfigService configService) {
		String envPath = configService.getParamValue("cbfbirn.env.path");
		helper.setEnvPath(envPath);
		helper.addEnvironmentVar("FSLDIR",
				configService.getParamValue("FSLDIR"));
		helper.addEnvironmentVar("FSLOUTPUTTYPE",
				configService.getParamValue("FSLOUTPUTTYPE"));
		helper.addEnvironmentVar("AFNI_PLUGINPATH",
				configService.getParamValue("AFNI_PLUGINPATH"));
		helper.addEnvironmentVar("FSLMULTIFILEQUIT",
				configService.getParamValue("FSLMULTIFILEQUIT"));
		helper.addEnvironmentVar("FSLCONFDIR",
				configService.getParamValue("FSLCONFDIR"));
		helper.addEnvironmentVar("FSLMACHTYPE",
				configService.getParamValue("FSLMACHTYPE"));
	}

	protected void handleEmail() throws Exception {
		System.out.println("handling email...");
		IAppConfigService configService = ServiceFactory.getAppConfigService();

		String emailHost = configService.getParamValue("email.host");
		String emailUser = configService.getParamValue("email.user");
		String emailPwd = configService.getParamValue("email.pwd");
		String emailFrom = configService.getParamValue("email.from");
		boolean canSendEmail = GenUtils.hasEnoughInfoForEmail(emailHost, emailFrom);
		
		if (!canSendEmail) {
			return;
		}
		EmailHelper helper = new EmailHelper(emailHost, emailUser, emailPwd,
				emailFrom);
		String toEmail = this.emailTo;
		System.out.println("handleEmail:: toEmail:" + toEmail);
		if (toEmail == null) {
			log.error("Cannot send email! Email was null");
			return;
		}
		String subject = "Your ROI Group Analyis with Job ID '" + this.id
				+ "' has finished";
		Map<String, String> paramMap = new HashMap<String, String>(7);
		paramMap.put("email", toEmail);
		paramMap.put("username", ui.getName());
		paramMap.put("jobId", this.id);
		try {
			helper.sendEmail(toEmail, subject, paramMap,
					this.context.getTemplateDir(), "roi_wf_finished.vm", false);
		} catch (Throwable t) {
			log.error("sendEmail", t);
		}
	}

	public static class GroupROIAnalysisJobFactory implements IJobFactory {

		@Override
		public IJob create(JobRecord jr) throws JobException {
			if (jr == null) {
				return null;
			}
			String ctxStr = jr.getContext();
			if (ctxStr == null || ctxStr.trim().length() == 0) {
				return null;
			}
			try {
				JSONObject js = new JSONObject(jr.getContext());
				if (js.has("assocList")) {
					GroupROIAnalysisContext ctx = GroupROIAnalysisContext
							.fromJSON(js);
					UserInfo jobUI = new UserInfo(jr.getUser(), null, null);
                    return new GroupROIAnalysisJob(
                            jr.getJobID(), jobUI, jr.getDescription(), ctx,
                            ctx.getEmailTo());
				}

			} catch (Throwable t) {
				throw new JobException(t);
			}
			return null;
		}
	}

}
