--
-- Autoupdate Schema Changes
--
-- This script prepares the database for easy updating of the schema.  It
-- expands the fields in the version table to contain information to list
-- attempts to start an update as well as end it, and provides functions
-- that check and update the database version.
--

DEFINE tablespace_data = 'HIDPRD_DATA'
DEFINE tablespace_index = 'HIDPRD_INDX'

-- Store the information that we need from the old nc_schemaVersion
CREATE TABLE temp_nc_schemaVersion(
    owner           NUMBER(38, 0)    NOT NULL,
    modUser         NUMBER(38, 0)    NOT NULL
);

INSERT INTO temp_nc_schemaVersion (owner, modUser) VALUES ((SELECT max(owner) from nc_schemaVersion), (SELECT max(modUser) from nc_schemaVersion));

ALTER TABLE nc_tableID DROP CONSTRAINT Refnc_schemaVersion1581;

-- We don't want to retain its data; we want a specific set of starting data.
DROP TABLE nc_schemaversion;

CREATE TABLE nc_schemaVersion(
    majorVersion    NUMBER(38, 0)    NOT NULL,
    minorVersion    NUMBER(38, 0)    NOT NULL,
    updateVersion   NUMBER(38, 0)    NOT NULL,
    updateFinished  NUMBER(1, 0)     NOT NULL,
    uniqueID        NUMBER(38, 0)    NOT NULL,
    tableID         NUMBER(38, 0)    NOT NULL,
    owner           NUMBER(38, 0)    NOT NULL,
    modTime         DATE             NOT NULL,
    modUser         NUMBER(38, 0)    NOT NULL,
    releaseDate     DATE              DEFAULT sysdate NOT NULL,
    description     CLOB,
    CONSTRAINT nc_schemaVersion_pk PRIMARY KEY (majorVersion, minorVersion, updateFinished)
                USING INDEX
                (CREATE INDEX nc_schemaVersion_pidx on nc_schemaVersion(majorVersion, minorVersion, updateFinished)
                TABLESPACE &&tablespace_index),
    CONSTRAINT nc_schemaVersion_sk  UNIQUE (uniqueID, tableID)
                USING INDEX
                (CREATE INDEX nc_schemaVersion_sidx on nc_schemaVersion(uniqueID, tableID)
                TABLESPACE &&tablespace_index)
)
TABLESPACE &&tablespace_data;

COMMENT ON COLUMN nc_schemaVersion.majorVersion IS 'The ID for the major release of the database.  Is rarely upped.';

COMMENT ON COLUMN nc_schemaVersion.minorVersion IS 'The ID for the minor release of the database.  Is upped with every patch.';

COMMENT ON COLUMN nc_schemaVersion.updateVersion IS 'The version of the patch that updated us to this majorversion/minorversion combination.';

COMMENT ON COLUMN nc_schemaVersion.updateFinished IS 'Is this the start of an update or the conclusion?';

COMMENT ON COLUMN nc_schemaVersion.uniqueID IS 'The uniqueID column is one of the 5 core columns (tableID, uniqueID, owner, modTime, modUser) in the database. Taken together with the tableID column these two columns specify a global unique object/tuple identifier within the database.  This unique identifying pair is critical for the extended tuple interface, database access controls and time travel. The uniqueID also servers as the principal unique identifier with in each core table.';

COMMENT ON COLUMN nc_schemaVersion.tableID IS 'The tableID column is one of the 5 core columns (tableID, uniqueID, owner, modTime, modUser) in the database. Taken together with the uniqueID column these two columns specify a global unique object/tuple identifier within the database.   This unique identifying pair is critical for the extended tuple interface, database access controls and time travel. The tableID also servers as a supplementary identifier that allows each tuple''s containing table to be easily denoted.';

COMMENT ON COLUMN nc_schemaVersion.owner IS 'The owner column is one of the 5 core columns (tableID, uniqueID, owner, modTime, modUser) in the database. Taken together with the modUser column these two columns specify who owns the data (owner) in a tuple (e.g. the prinicipal investigator of a project who leads the experiment) and who actually entered the data (modUser) in the tuple (e.g. the PI''s graduate student researcher).';

COMMENT ON COLUMN nc_schemaVersion.modTime IS 'The modTimecolumn is one of the 5 core columns (tableID, uniqueID, owner, modTime, modUser) in the database. Taken together with the modUser column these two columns specify who entered the data (modUser) and when (modTime).  Taken together with a core table (nc_TABLENAME) and its backing table (bk_TABLENAME which is used to store the history of changes to a tuple) modTime allows one to fully recreate the history of every tuple stored in a core table.';

COMMENT ON COLUMN nc_schemaVersion.modUser IS 'The modUser column is one of the 5 core columns (tableID, uniqueID, owner, modTime, modUser) in the database. Taken together with the owner column these two columns specify who owns the data (owner) in a tuple (e.g. the prinicipal investigator of a project who leads the experiment) and who actually entered the data (modUser) in the tuple (e.g. the PI''s graduate student researcher).';

COMMENT ON COLUMN nc_schemaVersion.releaseDate IS 'The versionID domain is used to denote version identifiers. These identifiers can have major and minor revisions.';

COMMENT ON COLUMN nc_schemaVersion.description IS 'The description column is an open field for any comments or descriptions for a specific entry (i.e. tuple) in the database.';

INSERT INTO nc_schemaVersion (majorVersion, minorVersion, updateVersion, updateFinished, uniqueID, tableID, owner, modTime, modUser, releaseDate, description)
VALUES (1, 0, 0, 1, uid_seq.nextval, get_table_id('nc_schemaVersion'), (SELECT owner FROM temp_nc_schemaVersion), SYSDATE, (SELECT modUser FROM temp_nc_schemaVersion), SYSDATE, EMPTY_CLOB());

--ALTER TABLE nc_tableID ADD CONSTRAINT Refnc_schemaVersion1581
--    FOREIGN KEY (majorVersion, minorVersion)
--    REFERENCES nc_schemaVersion(majorVersion, minorVersion, 1);

DROP TABLE temp_nc_schemaVersion;

-- Now, create the version query procedure.

CREATE OR REPLACE FUNCTION check_for_schema_version(
old_majorversion NUMBER,
old_minorversion NUMBER
) RETURN NUMBER
IS
  ret NUMBER;
BEGIN
  SELECT count(uniqueID) INTO ret FROM nc_schemaVersion WHERE majorVersion=old_majorversion AND minorVersion=old_minorVersion AND updateFinished=1;
  RETURN ret;
END;
/

-- Lastly, create the version setting function.

CREATE OR REPLACE FUNCTION set_schema_version(
majorversion NUMBER,
minorversion NUMBER,
updateversion NUMBER,
releasedate DATE,
completed NUMBER
) RETURN NUMBER
IS
  PRAGMA AUTONOMOUS_TRANSACTION;  -- Workaround so that this can be called in a select.
  owner NUMBER;
  modUser NUMBER;
  tableID NUMBER;
  description CLOB;
BEGIN
  SELECT max(owner) INTO owner FROM nc_schemaversion;
  SELECT max(modUser) INTO modUser FROM nc_schemaversion;
  tableID := get_table_id('nc_schemaVersion');
  description := EMPTY_CLOB();

  INSERT INTO nc_schemaVersion (majorVersion, minorVersion, updateVersion, updateFinished, uniqueID, tableID, owner, modTime, modUser, releaseDate, description)
  VALUES (majorversion, minorversion, updateversion, completed, uid_seq.nextval, tableID, owner, SYSDATE, modUser, releasedate, description);

  commit;
  return 0;
END;
/
