#!/bin/bash

#Check to make sure that they've provided the right commandline arguments.
if [ "$#" -ne 2 ]
then
  echo "Usage: apply_updates.sh DatabaseConnectionString VersionNumberToUpdateTo"
  exit 1
fi

connection_string=$1
last_patchmajor=`echo $2 | cut -d"." -f1`
last_patchminor=`echo $2 | cut -d"." -f2`

GetCurrentHIDVersion()
{
  #Fetch their current version from the database
  current_version_info=`echo -e "SELECT majorversion, minorversion FROM nc_schemaversion WHERE modtime=(SELECT max(modtime) FROM nc_schemaversion WHERE updateFinished=1) and updateFinished=1;\nexit;" | sqlplus $connection_string`
  OIFS=$IFS
  IFS='
'	#Set the field separator to not split fields on spaces or tabs - only newlines.
  counter=0
  current_minorversion="NONE"
  current_majorversion="NONE"
  for line in $current_version_info
  do
    if [ x"$line" = x"MAJORVERSION MINORVERSION" ]	#Two lines after we encounter this, we want to grab the line to get the current version out.
    then
      counter=2
    else
      if [ $counter -gt 0 ]
      then
        counter=`expr $counter - 1`
        if [ $counter -eq 0 ]
        then
	  IFS=$OIFS
	  for i in $line
	  do
	    if [ x"$current_majorversion" = x"NONE" ]
	    then
              current_majorversion=$i
	    else
	      current_minorversion=$i
	    fi
	  done
	  break
        fi
      fi
    fi
  done
  IFS=$OIFS

  #If the current version was never set, then we're missing a row in the database.
  if [ x"$current_minorversion" = x"NONE" ]
  then
    echo "ERROR: Your database does not have versioning information in it.  Please correct this."
    exit 2
  fi

  ret1=$current_minorversion
  ret2=$current_majorversion
}

#Run through all of the available patches that are within the needed range.
GetCurrentHIDVersion; current_minorversion=$ret1; current_majorversion=$ret2
minorversion=$current_minorversion
majorversion=$current_majorversion

if [ $last_patchmajor -lt $majorversion ]
then
  echo "ERROR: You cannot rollback patches with this script."
  exit 3
fi

if [ $last_patchmajor -eq $majorversion -a $last_patchminor -lt $minorversion ]
then
  echo "ERROR: You cannot rollback patches with this script."
  exit 3
fi

if [ $last_patchmajor -eq $majorversion -a $last_patchminor -eq $minorversion ]
then
  echo "You are already at this patch version."
  exit 0
fi

while `true`
do
  minorversion=`expr $minorversion + 1`

  #Try to find the patch
  patchname=`ls "$majorversion.$minorversion"_* 2>/dev/null`
  if [ x"$patchname" = x ]
  then
    #Perhaps it is a major version number increase?
    old_majorversion=$majorversion
    majorversion=`expr $majorversion + 1`
    old_minorversion=$minorversion
    minorversion=0
    patchname=`ls "$majorversion.$minorversion"_* 2>/dev/null`
    if [ x"$patchname" = x ]
    then
      #We're out of patches.  Exit.
      echo "ERROR: No update with version $old_majorversion.$old_minorversion or $majorversion.$minorversion was found."
      exit 4
    fi
  fi

  if [ $last_patchmajor -lt $majorversion ]
  then
    echo "ERROR: $last_patchmajor.$last_patchminor is obsoleted by $majorversion.$minorversion."
    exit 6
  fi

  #Attempt to run the patch; catch any exceptions.
  cp "$patchname" temp.sql
  echo "exit" >> temp.sql
  ret=`sqlplus $connection_string @temp.sql 2>&1  | tee logfile | grep "ERROR at line"`
  cat logfile
  rm temp.sql
  rm logfile
  if [ x"$ret" != x ]
  then
    echo "ERROR: An error has been reported by sqlplus when applying update \"$patchname.\""
    exit 5
  else
    echo "Update \"$patchname\" has been successfully applied."
  fi

  #Check for a properly incremented version; if not found, exit out with an error.
  GetCurrentHIDVersion; current_minorversion=$ret1; current_majorversion=$ret2
  if [ $current_minorversion -ne $minorversion -o $current_majorversion -ne $majorversion ]
  then
    echo "ERROR: Patch \"$patchname\" failed to complete successfully.  Please talk to the patch author about how to roll back its effects before you continue!"
    exit 6
  fi

  if [ $last_patchmajor -eq $majorversion -a $last_patchminor -eq $minorversion ]
  then
    echo "Finished."
    exit 0
  fi

done


drop table dummy_table;
delete from nc_schemaversion where minorversion > 0;
commit;

