#!/usr/bin/perl -w

# $Id: ora2postgres.pl,v 1.1 2005/06/16 21:52:06 bozyurt Exp $

use strict;
use Carp;
use File::Basename;
use File::Path;
use Getopt::Std;
require Iterator;

my $oracle_dir = 
  "/home/bozyurt/tmp/test_build/BIRN/HID/Schema/Oracle/db_objects";
my $postgres_dir = "/home/bozyurt/dev/postgres/db_objects";

sub usage() {
die<<EOB
  Tries to convert some of the HID database creation/preparation scripts 
  from Oracle to Postgres
  $0 -i <oracle-scripts-directory> -o <postgres-scripts-directory> [-t|-f|-r|-h]
  where 
  -t to converts table creation scripts ( nc_rangevalidator will have syntax errors you need to correct it by hand)
  -f converts foreign key creation dropping scripts
  -r converts triggers to Postgres
  -h to display this help screen
  
EOB
}

sub parse_create_table_st {
    my ($fh,@lines) = @_;
    my $idx = 0;
    my $line;
    while ($idx < $#lines) {
	$line = $lines[$idx];
       
        if ( $line =~ /^\s*constraint/ ) {
	    my $comma_at_end = 1;
           # combine lines till a , at line end or a ;
	    while( $idx < $#lines) {
		$line =~ s/[\r]?\n^$/ /;
		$line .= " " . $lines[++$idx];
                 
                $comma_at_end = 0 if ( $line =~ /;\s*$/ );
		last if ( $line =~ /;\s*$/ || $line =~ /,\s*$/);
	    } 
            # cut off the portion starting at using index
	    $line =~ s/(.+)\s*using index.+/$1/;
            $line =~ s/\s*$/ /;
            if ($comma_at_end) {
               print $fh "$line,\n"; 
	    } else {
		print $fh "${line});\n";
	    }
	} else {
            if ( $line =~ /number\s*\(\s*1\s*,\s*0\s*\)/) {
		$line =~ s/number\s*\(\s*1\s*,\s*0\s*\)/boolean/;
                $line =~ s/default\s+0/default false/;
                $line =~ s/default\s+1/ default true/;
	    }
            if ( $line =~ /varchar2/) {
              $line =~ s/varchar2/varchar/; 
	    }
            $line =~ s/\s+date\s+/ timestamp /;
            $line =~ s/number\(.+\)/bigint/;
	    $line =~ s/clob/text/;
            $line =~ s/sysdate/now\(\)/;
	    print $fh "$line\n";
	}
	$idx++;
    }
}


sub convert_tables_file {
    my ($fname, $out_file) = @_;
    local *OUT;
    open(IN,"$fname") or croak("cannot open file $fname:$!");
    open(OUT,">$out_file") or croak("cannot create file $out_file:$!");
    my $line;
    while(<IN>) {
	chomp;
	my $line = lc $_;
	$line =~ s/\r//;
        # print "**>> $line\n";
	next if ($line =~/\s*--/);
	next if ($line =~/^\s*$/);
        unless ($line =~ /;\s*$/) {
           my @lines = ();
           push @lines, $line;
	   while(<IN>) {
	       chomp;
               $line = lc $_;
               $line =~ s/\r//;
	       # print ">> $line\n";
	       push @lines, $line;
	       last if ($line =~ /;\s*$/)
           }
           if ( $lines[0] =~ /\s*create\s+table/ ) {
	       parse_create_table_st(*OUT, @lines);
               print OUT "\n"; 
	   } elsif ( $lines[0] =~ /\s*comment/ ) {
               print OUT "-- " . join(" ",@lines) . "\n\n";
	   } elsif ( $lines[0] =~ /\s*drop\s+table/) {
               my $st = join(" ",@lines);
              $st =~ s/cascade.+$/;/;
	       print OUT "$st\n\n";  
	   }
	}
    }
    close(IN);
    close(OUT);
}

# all lines combined including the last line with the end_pattern
sub group_lines {
    my ($fh, $cur_line, $end_pattern) = @_;
    my $line = $cur_line;
    my @lines = ();
    push @lines, $line;
    while(<$fh>) {
       chomp;	
       $line = lc $_;
       $line =~ s/\r//;
       push @lines, $line;
       last if ($line =~ /$end_pattern/);
    }
    return join(" ",@lines) 
}
    
sub parse_trigger_start {
  my ($st) = @_;
  my @toks = split (/\s/,$st);
  my $postgres_st = "CREATE FUNCTION ";
  my $iter = new Iterator(\@toks);
  my $trigger_name = undef;
  my $trigger_def = "";
  while ($iter->has_next() ) {
      my $tok = $iter->next();
      if ($tok =~ /trigger/) {
	  $trigger_name = $iter->next();
          do {
	      $tok = $iter->next();
              last if ($tok =~ /begin/);
	      $trigger_def .= $tok . " ";
          } while( $iter->has_next());
      } 
  }
  return ($trigger_name, $trigger_def);
}



sub convert_trigger_file {
    my ($fname, $out_file) = @_;
    local *OUT;
    local *IN;
    open(IN,"$fname") or croak("cannot open file $fname:$!");
    open(OUT,">$out_file") or croak("cannot create file $out_file:$!");
    my $line;
    my $trigger_name;
    my $trigger_def;
    my $in_body = 0;
    while(<IN>) {
	chomp;
	my $line = lc $_;
	$line =~ s/\r//;
	next if ($line =~/\s*--/);
	next if ($line =~/^\s*$/);
        if ( $line =~ /create.+trigger/) {
	    my $line = group_lines(*IN, $line,"begin");
	   ($trigger_name, $trigger_def) = parse_trigger_start($line);
	    $in_body = 1;
            print OUT "CREATE FUNCTION $trigger_name() RETURNS trigger AS '\nBEGIN\n";
	} elsif ($in_body) {
           if ( $line =~ /end;/) {
	       $in_body = 0;
               print OUT "\treturn new;\n";
	       print OUT "END;\n";
	       print OUT "' LANGUAGE plpgsql;\n\n";
               print OUT "CREATE TRIGGER $trigger_name $trigger_def EXECUTE PROCEDURE $trigger_name();\n\n";
	   } else {
	       $line =~ s/:new/new/g;
	       $line =~ s/:old/old/g;
	       print OUT "$line\n";
	   }
	}         
    }
    close(IN);
    close(OUT);
}


sub to_lowercase {
   my ($fname, $out_file) = @_;
   open(IN,"$fname") or croak("cannot open file $fname:$!");
   open(OUT,">$out_file") or croak("cannot create file $out_file:$!");
   my $line;
   while(<IN>) {
       chomp;
       my $line = lc $_;
       $line =~ s/\r//;
       print OUT "$line\n"; 
   }
   close(IN);
   close(OUT);
}




sub convert_tables() {
    my @table_files = <$oracle_dir/Tables/*.sql>;
    my $pg_root = "${postgres_dir}/Tables";
    unless(-d $pg_root) {
	File::Path::mkpath($pg_root,1);
    } 
    
    foreach my $table_fname (@table_files) {
	my $postgres_file = "${pg_root}/" .  basename($table_fname);
	# no equivalent table as oracle_currentUser
	next if ($postgres_file =~ /oracle_currentUser/);
	print "$postgres_file\n"; 
	convert_tables_file($table_fname, $postgres_file);
    }
}


sub convert_triggers() {
    my @trigger_files = <$oracle_dir/Triggers/*.sql>;
    my $pg_root = "${postgres_dir}/Triggers";
    unless(-d $pg_root) {
	File::Path::mkpath($pg_root,1);
    } 
    foreach my $tf (@trigger_files) {
	my $postgres_tf = "${pg_root}/" .  basename($tf);
	next if ($postgres_tf =~ /drop_triggers.sql/);
        print "$postgres_tf\n";
	convert_trigger_file($tf, $postgres_tf);
    }
}

sub convert_fk_scripts() {
    my @fk_files = <$oracle_dir/FK/*sql>;
    my $pg_root = "${postgres_dir}/FK";
    unless(-d $pg_root) {
	File::Path::mkpath($pg_root,1);
    }
    foreach my $fk_file (@fk_files) {
	my $pg_file = "$pg_root/" . basename($fk_file);
	print "$pg_file\n";
	to_lowercase($fk_file, $pg_file);
    }
}

sub create_driver_script {
    my ($dir, $script_fname, $root_dir) = @_;
    my @files = <$dir/*.sql>;
    open(OUT,">$script_fname") or croak("cannot create file $script_fname:$!");
    foreach my $file (@files) {
        $file =~ s/$root_dir\///;
	print OUT "\\i $file;\n";
    }
    close(OUT);
}




# convert_tables();

# convert_tables_file($table_fname,"xx");
# convert_trigger_file("$oracle_dir/Triggers/nc_expCondition_ins_upd.sql","xx");
# convert_triggers();

# create_driver_script("$postgres_dir/Tables","create_tables.sql", $postgres_dir);

# convert_fk_scripts();

sub main() 
{
    my %opts = ();
    getopts("i:o:tfrh",\%opts);
    usage() unless ( scalar(@ARGV) == 0);
    usage() if ($opts{h});
    unless( $opts{i} && $opts{o} ) {
	usage();
    }
    $oracle_dir = $opts{i};
    $postgres_dir = $opts{o};
    if ($opts{t} ) {
        print "conveting tables...\n";
	convert_tables();
        create_driver_script("$postgres_dir/Tables","create_tables.sql", 
                $postgres_dir);
    }
    if ($opts{r}) {
        print "converting triggers...\n";
	convert_triggers();
    }
    if ($opts{f}) {
       print "conveting foreign key scripts...\n";
       convert_fk_scripts(); 
    }
}

main();
