#!/usr/bin/perl -w

# $Id: cleanup.pl,v 1.5 2005/10/31 19:03:49 bozyurt Exp $

use strict;
use Carp;
use Fcntl ':flock'; # import LOCK_* constants

sub usage() {
die<<EOF
  Usage: $0 <cache-root> <max-age-in-seconds> [max-size-in-bytes]
EOF
}

sub lock {
    my ($lock_file) = @_;
    open(LOCK_HANDLE,"+>> $lock_file") or croak("cannot open $lock_file:$!");
    flock(LOCK_HANDLE, LOCK_EX);       
}

sub unlock() {
    flock(LOCK_HANDLE, LOCK_UN);
    close(LOCK_HANDLE);
}


sub get_dir_size {
  my ($dir) = @_;
  my @files = <$dir/*>;
  my $tot_size = 0;
  foreach my $file (@files) {
    next unless (-f $file);
    $tot_size += (stat($file))[7];
  }
  return $tot_size;
}

sub get_lrus {
  my ($cache_root) = @_;
  my @contents = <$cache_root/*>;
  my @dir_stats = ();
  foreach my $dir (@contents) {
      next unless (-d $dir);
      my $dir_size = get_dir_size($dir);
      my @files = <$dir/*.tgz>;
      # a heisenbug the access time changes whenever we probe for the acess time
      # my $atime = (scalar(@files) == 1) ? ( stat($files[0]))[8] : (stat($dir))[8];
      my $mtime =  (stat($dir))[9];
      my $rec = {
               SIZE => $dir_size,
               MTIME => $mtime,
               DIR => $dir
            };
      push @dir_stats, $rec;
  }
  # order by access time in ascending order
  # order by modification time in ascending order
  # since everytime a request comes for a cached brik, the brik is touched,
  # this becomes a LRU cache
  my @lrus = sort { $a->{MTIME} <=> $b->{MTIME} } @dir_stats;
  return @lrus;
}

sub clean {
  my ($cache_root, $max_age_secs, $max_size) = @_;
  my @lrus = get_lrus($cache_root);
  my $tot_size = 0;
  foreach my $rec (@lrus) {
    # print " -- " . $rec->{DIR} . "\n";
     $tot_size +=  $rec->{SIZE};
  }
  # print("total cache size=$tot_size\n");
  my $rem_size = $tot_size;
  foreach my $rec (@lrus) {
    my $dir = $rec->{DIR};
    if ($dir =~ /\/(\w+)$/ ) {
        my $lock_file = "$cache_root/.$1";
        my $read_lock_file = "${lock_file}_r";
        unless (-e $read_lock_file) {
          lock($lock_file);
          if (-d $dir) {
            my $do_clean = 0;
	    my $age_secs = time() - $rec->{MTIME}; 
	    # print "age in secs = " . $age_secs . "\n";
            $do_clean = $age_secs >  $max_age_secs;
	    if ($max_size > 0 && $rem_size > $max_size) {
                $do_clean |= 1;
	    }
            if ($do_clean) {
               # remove the cached image series
                print "cleaning up $dir...\n";
                my @files = <$dir/*>;
                unlink @files;
                rmdir $dir;
		$rem_size -= $rec->{SIZE};
            }
          }  
          unlock();
        }
    }
  }
}


sub main() {
   my $arglen =  @ARGV;
   usage() unless ( $arglen == 2 || $arglen == 3);
   my $cache_root = shift @ARGV;
   my $max_age_secs = shift @ARGV;
   my $max_size = ($arglen == 3) ? shift @ARGV : -1;
   exit(1) unless (-e $cache_root);
   # print("found $cache_root.\n");
   clean($cache_root, $max_age_secs, $max_size);
}

main();



