#!/usr/bin/perl -w

# $Id: mri_converter.pl,v 1.8 2004/07/10 00:58:35 bozyurt Exp $
use strict;
use Carp;
use Getopt::Std;
use File::Basename;
use dcmutils;
use uploadutils;
use srbutil;
use File::Path;
use Fcntl ':flock'; # import LOCK_* constants

# set to 0 if your Scommands version does not support Sget -b
my $USE_SGET = 1;


sub usage() {
die<<EOF
  Usage: $0 <cache-root> <srb-series-collection>
EOF
}

sub lock {
    my ($lock_file) = @_;
    open(LOCK_HANDLE,"+>> $lock_file") or croak("cannot open $lock_file:$!");
    flock(LOCK_HANDLE, LOCK_EX);       
}

sub unlock() {
    flock(LOCK_HANDLE, LOCK_UN);
    close(LOCK_HANDLE);
}

#
# for SRB 2.1.3 and higher, uses Sget  
#
sub download_series2 {
    my ($series_col, $series_dir) = @_;
    print("*** downloading series $series_col...\n");
    srbutil:sinit();
    # my $cmd = "Sget -b $series_col $series_dir";
    my $cmd = "Sget  $series_col/*.* $series_dir";
    print("cmd=$cmd\n"); 
    my $rc = system($cmd);
    croak "problem in Sget :$!" if ( $rc != 0);
    srbutil:sexit(); 
    my @files = <$series_dir/*>;
    my $sdir = undef;
    foreach my $file (@files) {
	if (-d $file && $file =~ /ser$/ ) {
	    $sdir = $file;
	    last;  
	}
    }
    if (defined($sdir) &&  -d $sdir) {
        print("moving contents of $sdir to  $series_dir.\n");
	system("mv $sdir/* $series_dir");
	system("rmdir $sdir");
    }
}

sub download_series {
    my ($container_name, $series_dir) = @_;
    print("*** downloading series $container_name...\n");
    # specific to UCSD
    my $container_path = "/container/bozyurt.ucsd-fmri/" . $container_name;
    srbutil:sinit();
    my $cmd = "Sbunload $container_path $series_dir";
    print("cmd=$cmd\n"); 
    my $rc = system($cmd);
    croak "problem in Sbunload:$!" if ( $rc != 0);
    srbutil:sexit();   
    my @files = <$series_dir/*>;
    my $sdir = undef;
    foreach my $file (@files) {
	if (-d $file && $file =~ /ser$/ ) {
           $sdir = $file;
           last;  
	}
    }
    if (-d $sdir) { 
      system("mv $sdir/* $series_dir");
      system("rmdir $sdir");
    }
}


sub dcm_to_raw  {
    my ($dcm_dir) = @_;
    my @dcm_files = <$dcm_dir/*.dcm>;
    foreach my $dcm_file (@dcm_files) { 
	my $base = basename($dcm_file,".dcm");
	my $raw_file = "$dcm_dir/${base}.raw";
	if ( system("dc3toraw $dcm_file $raw_file") != 0) {
	    croak("Cannot  dc3toraw and create  $raw_file!");
	}
    }
}


# Caution: No obliques yet!
sub get_orientation {
    my (@vec) = @_;
    return "R-L" if ($vec[0] == 1);
    return "L-R" if ($vec[0] == -1);
    return "A-P" if ($vec[1] == 1);
    return "P-A" if ($vec[1] == -1);
    return "I-S" if ($vec[2] == 1);
    return "S-I" if ($vec[2] == -1);
    return "";
}

sub get_slice_dir_orientation {
    my ($slice_dir_idx, $sign) = @_;
    return "R-L" if ($slice_dir_idx == 0 && $sign > 0);
    return "L-R" if ($slice_dir_idx == 0 && $sign < 0);
    return "A-P" if ($slice_dir_idx == 1 && $sign > 0);
    return "P-A" if ($slice_dir_idx == 1 && $sign < 0);
    return "I-S" if ($slice_dir_idx == 2 && $sign > 0);
    return "I-S" if ($slice_dir_idx == 2 && $sign < 0);
    return "";
}




sub dicom_to_afni {
    my ($series_dir,$brik_prefix) = @_;
    
    my @afni_briks = <$series_dir/$brik_prefix*.tgz>;          
    if (scalar(@afni_briks) > 0) {
	print "Found tar file with the name " . basename($afni_briks[0]) 
                . "! Skipping conversion.\n";
        return;
    }
    dcm_to_raw($series_dir);
 


    # use file names to determine slice order assuming they reflect the slice order
    my $first_slice = "$series_dir/001.dcm";
    my $last_slice = "$series_dir/124.dcm";
    my @fields = qw(ImagePositionPatient ImageOrientationPatient Rows Columns PixelSpacing ReconstructionDiameter ImagesInAcquisition SliceThickness SpacingBetweenSlices);
    my %value_map = dcmutils::get_dicom_header_values($first_slice, @fields);
     
    my $last_img_pos_str = dcmutils::get_dicom_header_value($last_slice,"ImagePositionPatient");

    my $fov = $value_map{ReconstructionDiameter};
    my @ps = split(/\\/, $value_map{PixelSpacing});
    my @dc =  split(/\\/, $value_map{ImageOrientationPatient});   
    my @vcols = @dc[0,1,2];
    my @vrows = @dc[3,4,5];   
    
    my @first_img_pos = split(/\\/, $value_map{ImagePositionPatient});
    my @last_img_pos =split(/\\/, $last_img_pos_str);
    my @orient = ();
    # x -> cols , y -> rows , z-> slices (for AFNI)
    $orient[0] =  get_orientation(@vcols);
    $orient[1] =  get_orientation(@vrows);
    my $slice_dir_idx = -1;
    for(my $i = 0; $i < 3; $i++) {
        if ($vcols[$i] == 0 && $vrows[$i] == 0) {
	    $slice_dir_idx = $i;
            last;
        } 
    } 
    my $slice_dir_sign = ($first_img_pos[ $slice_dir_idx ] < $last_img_pos[ $slice_dir_idx ]) ? 1 : -1;
    $orient[2] = get_slice_dir_orientation( $slice_dir_idx, $slice_dir_sign);
    print "orientation= " . join(',',@orient) . "\n";    

    # for AFNI
    my $fovx = $value_map{Rows} * $ps[0];
    my $fovy = $value_map{Columns} * $ps[1];
    my $fovz = $value_map{ImagesInAcquisition} * $value_map{SliceThickness};
    my $geom = " -xFOV " . $fovx/2 . $orient[0] . " -yFOV ". $fovy/2 . $orient[1] . " -zFOV " . $fovz/2 . $orient[2] . " ";
    print "geom: $geom\n";  

    my $curdir = $ENV{"PWD"};
    croak ("cannot cd to $series_dir:$!\n") unless chdir($series_dir);
     
    my $cmd = "to3d  $geom -prefix $brik_prefix [0-9][0-9][0-9].raw";
    print("$cmd\n");
    system($cmd); 
    system("tar czvf $brik_prefix.tgz $brik_prefix*.BRIK $brik_prefix*.HEAD"); 

    if (-e $series_dir) {
	system("rm -f $series_dir/*.raw");
        system("rm -f $series_dir/*.dcm");
        system("rm -f $series_dir/*.BRIK");
        system("rm -f $series_dir/*.HEAD");
    } 
 
    if (length $curdir > 0) {
      croak ("cannot cd to $curdir:$!\n") unless chdir($curdir);
    }
    
}


sub create_brick_prefix {
    my ($series_col) = @_;
    my @parts = split(/\//, $series_col);
    if ( $series_col =~ /__/ ) {
       # new SRB hierarchy 7/7/04
       # just the birn id
	return $parts[4];
    } else {
        return uploadutils::create_container_name($parts[6], $parts[7], $parts[8], $parts[10]);
    }

} 

sub convert {
    my ($cache_root, $series_col) = @_;

    my @parts = split(/\//, $series_col);
    my $container = undef;
    if ( $series_col =~ /__/) {
        $container = $parts[4]; 
    } else {
	$container = uploadutils::create_container_name($parts[6], $parts[7], $parts[8], $parts[10]);
    }


    my $lock_file = "$cache_root/.$container";
    lock($lock_file);
    my $series_dir = "$cache_root/$container";
   
    File::Path::mkpath($series_dir) unless (-e $series_dir);

    my $brik_prefix = create_brick_prefix($series_col); 

    # check if already cached
    my @data = <$series_dir/$brik_prefix*.tgz>;      
    print "+++ size=" . scalar(@data) . "\n";
    if ( scalar(@data) == 0) {
        if ($USE_SGET) {
	    download_series2($series_col, $series_dir);
	} else {  
	    download_series($container, $series_dir);
        }
        dicom_to_afni($series_dir, $brik_prefix);
        print("saved tar file:$series_dir/${brik_prefix}.tgz\n");
    } else {
	print("using cached  tar file:$series_dir/${brik_prefix}.tgz\n");
        # change the modifcation time for LRU cache
        system("touch -c  $series_dir"); 
    }
    unlock();  

}

sub main() {
  usage() if ( scalar(@ARGV) != 2); 
  my $cache_root = shift @ARGV;
  my $series_coll = shift @ARGV;
  croak("Cannot find cache_root:$cache_root:$!") unless (-e $cache_root);
  convert($cache_root, $series_coll);
}


main();






