package cbfbirn

import static groovy.io.FileType.*;

import java.io.File;

import groovy.transform.ToString
import groovy.xml.MarkupBuilder

class TMARC2UploadInputBuilder {
	def local2SubjectIDMap = [:]

	@ToString
	static class SubjectInfo {
		def localId
		def subjectId
		def gender
		def birthDate
		def condition
		def location
		def visitDate
		def scanner
	}

	@ToString
	static class Demographics {
		String magnet
		String group
		int age
		String gender
	}

	Demographics extractDemo(File sDir) {
		File readmeFile
		sDir.eachFile { if (it.name.startsWith("readme")) readmeFile = it }
		assert readmeFile
		Demographics d = new Demographics()
		readmeFile.text.eachLine { line ->
			String[] toks = line.split(/\s+=\s+/)
			if (toks[0] == "Magnet") {
				d.magnet = toks[1].trim()
			} else if (toks[0] == "Group") {
				d.group = toks[1].trim()
			} else if (toks[0] == "Gender") {
				d.gender = toks[1].trim()
			} else if (toks[0] == "Age") {
				d.age = (toks[1].trim() as int)
			}
		}
		return d
	}

	def prepSubjectList(File inRootDir) {
		def condMap = ['Control': 'Control','HIV':'HIV','METH':'Meth','B':'HIV/Meth','Both':'HIV/Meth', 'Meth':'Meth']
		def genderMap = ['M' :'Male','F':'Female'];
		def scannerMap = ['Old 3TW':'SIGNA HDx fmri3tw','3TW':'DISCOVERY MR750 fmri3tw', '3TE' :'SIGNA HDx fmri3te']
		def siList = []
		inRootDir.eachDir {
			if (it.name.startsWith("T")) {
				String localId = it.name.replaceFirst('\\.CBF$', "")
				Demographics d = extractDemo(it)
				File pFile
				new File(it,"ASL").eachFileMatch(FILES, ~/.*\.7$/, {pFile =it })
				assert pFile
				String visitDate = getScanDate(pFile)
				int visitYear = getYear( visitDate )
				def birthYear = visitYear - d.age;
				def birthDate = "01/01/" + birthYear;
				SubjectInfo si = new SubjectInfo(localId:localId, gender: genderMap[ d.gender ],
				birthDate:birthDate, condition: condMap[d.group ], visitDate: visitDate,
				scanner:"GE MEDICAL SYSTEMS " + scannerMap[ d.magnet ])
				siList << si
				//  println d
			}
		}

		siList.each { println it }
		return siList
	}

	def writeXml(siList, File inputDir, File outFile) {
		assert local2SubjectIDMap.size() > 0
		def writer = new StringWriter()
		def xml = new MarkupBuilder(writer)
		xml.cbfImport() {
			for(si in siList) {

				ds(subjectID: si.subjectId,  diagnosis:si.condition, expName:'TMARC_4S_FAIR',
				visitDate:si.visitDate, birthDate:si.birthDate, gender: si.gender,
				scannerInfo: si.scanner, location: si.location)
			}
		}
		println writer.toString()
		if (outFile != null) {
			writeFile(outFile, writer.toString())
		}
	}

	def static writeFile(File outFile, String content) {
		outFile.withWriter { out -> out.write(content); out.append("\n") }
		println "wrote file:$outFile"
	}

	static String toDate(String adf) {
		def m = adf =~ /(\d\d)\/(\d\d)\/(\d\d)/
		return m[0][1] + "/" + m[0][2] + "/20" + m[0][3]
	}

	static int getYear(String adf) {
		def m = adf =~ /(\d\d)\/(\d\d)\/(\d+)/
		return (m[0][3] as int)
	}

	def loadLocalId2SubjectIDMap(File cvsLookupFile) {
		cvsLookupFile.text.eachLine { line ->
			def toks = line.split(/,/)
			assert toks.size() == 2
			local2SubjectIDMap[toks[0]] = toks[1]
		}
	}

	def checkForMatchingSubject(siList) {
		siList.each { si ->
			if ( local2SubjectIDMap[si.localId]) {
				println "found matching for localId:${si.localId}"
			}
		}
	}

	def prepRawDataDirs(File inRoot, File outRoot, siList, skipCopy = false) {
		assert !local2SubjectIDMap.isEmpty()
		def sdList = []
		def map = [:]
		siList.each { si ->
			si.subjectId = local2SubjectIDMap[si.localId]
			map[si.localId] = si
		}
		inRoot.eachDir {
			String localId = it.name.replaceFirst('\\.CBF$', "")
			if (local2SubjectIDMap.containsKey(localId) ) {
				sdList << it
			}
		}
		sdList.each { sd ->
			def sdName = sd.name
			def seriesList = []
			sd.eachDir { d -> seriesList << d }
			def seriesMap = [:]
			seriesList.each { seriesMap[it.name] = it }

			String localId = sdName.replaceFirst('\\.CBF$', "")
			def si = map[localId]
			String vdName = localId + '_' + si.visitDate.replaceAll("\\/","_")
			File destSubjectPath = new File(outRoot, vdName);
			destSubjectPath.mkdirs();
			si.location = destSubjectPath

			def srcPFile = getPfile(seriesMap["ASL"]);
			File destPFile = new File(destSubjectPath, srcPFile.name)
			println "cp $srcPFile $destPFile"
			if (!skipCopy) {
				copyFile(srcPFile, destPFile)
			}

			srcPFile = getPfile(seriesMap["CSF"]);
			destPFile = new File(destSubjectPath, srcPFile.name)
			println "cp $srcPFile $destPFile"
			if (!skipCopy) {
				copyFile(srcPFile, destPFile)
			}

			srcPFile = getPfile(seriesMap["MinCon"]);
			destPFile = new File(destSubjectPath, srcPFile.name)
			println "cp $srcPFile $destPFile"
			if (!skipCopy) {
				copyFile(srcPFile, destPFile)
			}

			File srcDir = seriesMap["T1"]
			File destDir = new File(destSubjectPath, "T1")
			println "cp-dir $srcDir $destDir"
			if (!skipCopy) {
				copyDir(srcDir, destDir)
			}

			/*
			 def brikFiles = getBrikFiles(seriesMap["T1"])
			 brikFiles.each { file ->
			 File destFile = new File(destSubjectPath, file.name)
			 println "cp $file $destFile"
			 if (!skipCopy) {
			 copyFile(file, destFile)
			 }
			 }
			 */
		}
		println "-" *80
		siList.each { println it }
	}

	File getPfile(File dir) {
		File pFile
		dir.eachFileMatch(FILES, ~/.*\.7$/, {pFile =it })
		return pFile;
	}

	def getBrikFiles(File dir) {
		def list = []
		dir.eachFile { it -> if (it.name.endsWith(".HEAD") || it.name.endsWith(".BRIK")) list << it }
		return list
	}

	String getScanDate(File pFile) {
		def proc = "rdgehdr $pFile".execute();
		proc.waitFor();
		if (proc.exitValue() == 0) {
			String text = proc.in.text;
			def m = text =~ /Scan date: (\d+\/\d+\/\d+)/;
			String sd = m[0][1];
			String[] toks = sd.split(/\//)
			int year = (toks[2] as int) + 1900
			StringBuilder sb = new StringBuilder()
			sb << toks[0] << "/" + toks[1] << "/" << year
			return sb.toString()
		}
		return null
	}

	static void copyFile(File src, File dest) {
		new AntBuilder().copy(file:"$src.canonicalPath",tofile:"$dest.canonicalPath",overwrite:'true')
	}

	static void copyDir(File src, File dest) {
		dest.mkdir();
		def srcFiles = []
		src.eachFile { if (it.isFile()) srcFiles << it }
		srcFiles.each { srcFile ->
			copyFile(srcFile, new File(dest, srcFile.name ));
		}
	}

	void addAnatDicoms(inRoot) {
		def map = [:]

		inRoot.eachDir {
			if (it.name =~ /.CBF$/) {
				String localId = it.name.replaceFirst('\\.CBF$', "")
				if (local2SubjectIDMap.containsKey(localId) ) {
					map[localId] = it
				}
				new File(it,"T1").eachFile { f ->
					if (f.name.endsWith(".BRIK") || f.name.endsWith(".HEAD")) {
						f.delete();
					}
				}
			}
		}
		new File('/data/burak/tmarc2_in/data/TMARC/Human').eachDir { sd ->

			File destDir = map[sd.name]
			if (destDir) {
				destDir = new File(destDir,"T1")
				File srcDir = new File(sd,"raw/Project2/FSPGR1")
				println "destDir:$destDir"
				println "srcDir:$srcDir"
				copyDir(srcDir,destDir)
			}
		}
	}

	static void main(args) {
		TMARC2UploadInputBuilder builder = new TMARC2UploadInputBuilder()

		def siList = builder.prepSubjectList(new File('/data/burak/tmarc2_in'))

		File inputDir = new File('/data/burak/tmarc2_in')
		File outputDir = new File('/data/burak/tmarc2_out')
		builder.loadLocalId2SubjectIDMap(new File('/home/bozyurt/dev/java/clinical/scripts/TMARC2_project/tmarc_birnid_map.csv'))
		// builder.checkForMatchingSubject(siList)

		//builder.addAnatDicoms(inputDir)
		builder.prepRawDataDirs(inputDir, outputDir, siList, false)

		File importXml = new File('/home/bozyurt/dev/java/clinical/scripts/TMARC2_project','TMARC2_data_import.xml')
		builder.writeXml(siList, inputDir, importXml)
	}
}
