package cbfbirn

import groovy.transform.ToString;
import groovy.xml.MarkupBuilder;

import java.io.File;
import java.text.SimpleDateFormat

class TMARCUploadInputBuilder {
	def local2SubjectIDMap = [:]
	
	@ToString
	static class SubjectInfo {
		def localId
		def subjectId
		def gender
		def birthDate
		def condition
		def location
		def visitDate
		def scanner
	}
	
	def loadCSV(File tmarcCSVFile) {
		def condMap = ['Control': 'Control','HIV':'HIV','METH':'Meth','H+/M+':'HIV/Meth']
		def genderMap = ['1' :'Male','2':'Female'];
		def scannerMap = ['Old 3TW':'SIGNA HDx fmri3tw','3TW':'DISCOVERY MR750 fmri3tw', '3TE' :'SIGNA HDx fmri3te']
		def lines = []
		tmarcCSVFile.text.eachLine { line, lineNo -> 
		   if (lineNo > 0 && lineNo < 83) lines << line	
		}
		def siList = []
		lines.each {
			def toks = it.split(/,/)
			def localId = toks[0]
			def visitYear = getYear(toks[1])
			def age = toks[3].toInteger()
			def birthYear = visitYear - age;
			def birthDate = "01/01/" + birthYear;
			SubjectInfo si = new SubjectInfo(localId:localId, gender: genderMap[ toks[7]],
				birthDate:birthDate, condition: condMap[ toks[4]], visitDate: toDate(toks[1]),
			    scanner:"GE MEDICAL SYSTEMS " + scannerMap[ toks[6] ])
			siList << si
		}
		// siList.each { println it }
		return siList
	}
	
	def writeXml(siList, File inputDir, File outFile) {
		assert local2SubjectIDMap.size() > 0
		def writer = new StringWriter()
		def xml = new MarkupBuilder(writer)
		xml.cbfImport() {
			for(si in siList) {
				def subjectID = local2SubjectIDMap[si.localId]
				def location = 'NA';
				if (inputDir != null && new File(inputDir, si.localId).isDirectory()) {
					location = new File(inputDir, si.localId).canonicalPath
				}
				ds(subjectID:subjectID,  diagnosis:si.condition, expName:'TMARC_4S_FAIR',
					visitDate:si.visitDate, birthDate:si.birthDate, gender: si.gender,
					scannerInfo: si.scanner, location: location)
			}
		}
		println writer.toString()
		if (outFile != null) {
			writeFile(outFile, writer.toString())
		}
	}
	
	def static writeFile(File outFile, String content) {
		outFile.withWriter { out -> out.write(content); out.append("\n") }
		println "wrote file:$outFile"
	}
	
	static String toDate(String adf) {
	   def m = adf =~ /(\d\d)\/(\d\d)\/(\d\d)/
	   return m[0][1] + "/" + m[0][2] + "/20" + m[0][3] 	
	}
	
	static int getYear(String adf) {
		def m = adf =~ /(\d\d)\/(\d\d)\/(\d\d)/
		return ("20" + m[0][3]).toInteger()
	}
	
	def loadLocalId2SubjectIDMap(File cvsLookupFile) {
		cvsLookupFile.text.eachLine { line ->
			def toks = line.split(/,/)
			assert toks.size() == 2
			local2SubjectIDMap[toks[0]] = toks[1]
		}
	}
	
	
	static void main(args) {
		TMARCUploadInputBuilder builder = new TMARCUploadInputBuilder()
		
		def siList = builder.loadCSV(new File('/home/bozyurt/dev/java/clinical/scripts/TMARC_project/TMARC_T03.csv'))
		
		File inputDir = new File('/data/burak/tmarc_upload')
		builder.loadLocalId2SubjectIDMap(new File('/home/bozyurt/dev/java/clinical/scripts/TMARC_project/tmarc_birnid_map.csv'))
	
		File importXml = new File('/home/bozyurt/dev/java/clinical/scripts/TMARC_project','TMARC_data_import.xml')
		builder.writeXml(siList, inputDir, importXml)
	}
	
}
