package fbirn

import java.util.HashSet;

import java.util.Set;
import java.util.Map;

import com.pixelmed.dicom.DicomFileUtilities;


import groovy.util.AntBuilder;
import groovy.util.XmlSlurper;

class ASLImgDataOrganizer {
	def String srcRoot = '/data/bozyurt/fbirn'
	def String destRoot = '/data/bozyurt/cbfbirn/upload'
	
	static class SeriesInfo {
		def String subjectId;
		def String visitId;
		def String siteId;
		def String seriesName;
		def String srcPath;
		def String magnetName;
		
		@Override
		public String toString() {
			return "[visitId:$visitId, siteId:$siteId, seriesName:$seriesName, magnetName:$magnetName,srcPath:$srcPath]"
		}
	}
	
	static class VisitInfo {
		def String subjectId;
		def visitId;
		def String type;
		def String timeStamp
		def siList = []
		File visitDestDir
		def studyContentMap = [:]
		def studyMap = [:]
		def Map<String, File> studyDestDirMap = [:]
		
		public String getKey() {
			return "${subjectId}:${visitId}"
		}
		
		public def prepStudyMap() {
			siList.each { si -> 
				String key = si.magnetName
				if (key.length() == 0) key='default'
				def studySiList = studyMap[key]
				if (studySiList == null) {
					studySiList = []
					studyMap[key] = studySiList
				}
				studySiList << si
			}
			studyMap.keySet().each { k -> studyContentMap[k] = [:] }
			return studyMap
		}
		
		public String getDestVisitDirName() {
			def matcher = timeStamp =~ /(\d+)\-(\d+)\-(\d+)/
			return 'Visit_' + matcher[0][2] + '_' + matcher[0][3] + '_' + matcher[0][1]
		}
		
		@Override
		public String toString() {
			return "[subjectId:$subjectId, visitId:$visitId, type:$type, timeStamp:$timeStamp]"
		}
		
		def writeContents(String studyKey) {
			def visitDestDir = 	studyDestDirMap[studyKey]
			assert visitDestDir, "no visitDestDir for key:$studyKey"
			def contentMap = studyContentMap[studyKey]
			if (contentMap.empty) { return
			}
			
			def outFile = new File(visitDestDir,'CONTENTS')
			outFile.withWriter { writer ->
				contentMap.each {k,v ->
					writer.writeLine("${k}\t${v}")
				}
			}
		}
	}
	
	
	def prepSiteVisitMap(expInfoXmlFiles) {   
		// 'ucsf':'0002'
		def sidMap = ['ucsd':'0008','uci':'0009','duke':'0003','ucsf':'0020','mrn':'0010','uiowa':'0012','umn':'0013']
		def siteMap = [:]
		for(xmlFile in expInfoXmlFiles) {
			def name = new File(xmlFile).name
			def matcher = name =~ /\d_([^\.]+)\.xml$/
			name = matcher[0][1]
			def siteId = sidMap[name]
			assert siteId
			def visitMap = siteMap[siteId]
			if (visitMap == null) {
				visitMap = [:]
				siteMap[siteId] = visitMap
			}
			def xml = new XmlSlurper().parse(xmlFile)
			xml.subjects[0].subjExp.each { se ->
				println se.'@subjectid'
				se.visit.each { v ->
					VisitInfo vi = new VisitInfo(subjectId: v.'@subjectid',visitId:v.'@componentid',
							timeStamp:v.'@timeStamp',type:v.'@visittype')
					visitMap.put vi.key, vi
				}
			}
		}
		siteMap.each {k,vMap ->  
			println "${k}:"
			vMap.each {key,v ->
				println "\t" + v.toString()
			}
		}
		
		return siteMap
	}
	
	def copyB02Dest(String expName, siMap, siteMap) {
		def expDir = new File(destRoot, expName)
		assert expDir.directory
		def scanVisitMap = [:]
		siMap.each {subjectid, siList ->
			def subjDir = new File(expDir,subjectid)
			assert subjDir.directory
			println '-' * 40
			for(SeriesInfo si in siList) {
				def visitMap = siteMap[si.siteId]
				if (visitMap == null) {
					println "** no visitMap for si.siteId:${si.siteId}"
				}
				assert visitMap
				String key = si.subjectId + ':' + si.visitId
				VisitInfo vi = visitMap[key]
				assert vi
				File srcDir = new File(si.srcPath)
				String name = srcDir.name.toLowerCase()
				if (name.startsWith('b0')) {
					vi.siList << si
					//println "name: ${name} vi.siList.size:" + vi.siList.size()
					//println srcDir.canonicalPath
				}
			}
		}
		siteMap.values().each { visitMap ->
			visitMap.each {k, vi ->
				vi.prepStudyMap()
			}
		}
		
		siMap.each {subjectid, siList ->
			def subjDir = new File(expDir,subjectid)
			assert subjDir.directory
			println '-' * 40
			for(SeriesInfo si in siList) {
				def visitMap = siteMap[si.siteId]
				assert visitMap
				String key = si.subjectId + ':' + si.visitId
				VisitInfo vi = visitMap[key]
				assert vi
				
				File srcDir = new File(si.srcPath)
				String name = srcDir.name.toLowerCase()
				if (name.startsWith('b0')) {
					println "==> $subjectid - " + vi.destVisitDirName + ' ' + name + ' ' + srcDir
					File visitDir = new File(subjDir, vi.destVisitDirName +'_' + si.siteId);
					// assert visitDir.directory, "visitDir directory: $visitDir"
					if (!visitDir.directory) {
						print "**** No dest visit directory:$visitDir";
						continue;
					}
					if (si.magnetName.length() > 0) {
						visitDir = new File(visitDir, si.magnetName);
						assert visitDir.directory
						vi.studyDestDirMap[si.magnetName] = visitDir
					} else {
						vi.studyDestDirMap['default'] = visitDir
					}
				}
			}
		}
		
		siteMap.values().each { visitMap ->
			for(VisitInfo vi in visitMap.values()) {
				vi.studyMap.each { key,siList ->
					println "key:$key"
					if (vi.studyDestDirMap.containsKey(key) ) {
						prepDestB0(vi, siList, vi.studyDestDirMap[key], key)
					}
				}
			}
		}
	}
	
	def copy2Dest(String expName, siMap, siteMap) {
		def expDir = new File(destRoot, expName)
		expDir.mkdirs()
		def scanVisitMap = [:]
		siMap.each {subjectid, siList ->
			def subjDir = new File(expDir,subjectid)
			subjDir.mkdir()
			println '-' * 40
			for(SeriesInfo si in siList) {
				def visitMap = siteMap[si.siteId]
				if (visitMap == null) {
					println "** no visitMap for si.siteId:${si.siteId}"
				}
				assert visitMap
				String key = si.subjectId + ':' + si.visitId
				VisitInfo vi = visitMap[key]
				assert vi
				File srcDir = new File(si.srcPath)
				String name = srcDir.name
				if (name.startsWith('asl') || name == 't1') {
					vi.siList << si
				}
			}
		}
		siteMap.values().each { visitMap -> 
			visitMap.each {k, vi ->
				vi.prepStudyMap()
			}
		}
		
		siMap.each {subjectid, siList ->
			def subjDir = new File(expDir,subjectid)
			subjDir.mkdir()
			println '-' * 40
			for(SeriesInfo si in siList) {
				def visitMap = siteMap[si.siteId]
				assert visitMap
				String key = si.subjectId + ':' + si.visitId
				VisitInfo vi = visitMap[key]
				assert vi
				
				if (!hasASL(vi)) {
					continue
				}
				File srcDir = new File(si.srcPath)
				String name = srcDir.name
				if (name.startsWith('asl') || name == 't1') {		
					println "$subjectid - " + vi.destVisitDirName + ' ' + name + ' ' + srcDir
					File visitDir = new File(subjDir, vi.destVisitDirName +'_' + si.siteId);
					visitDir.mkdir();
					if (si.magnetName.length() > 0) {
						visitDir = new File(visitDir, si.magnetName);
						visitDir.mkdir();
						vi.studyDestDirMap[si.magnetName] = visitDir
					} else {
						vi.studyDestDirMap['default'] = visitDir
					}
				}
			}
		}
		
		siteMap.values().each { visitMap ->
			for(VisitInfo vi in visitMap.values()) {
				vi.studyMap.each { key,siList -> 
					println "key:$key"
					if (vi.studyDestDirMap.containsKey(key) ) {
						prepDest(vi, siList, vi.studyDestDirMap[key], key)
					}
				}
			}
		}
	}
	
	
	def prepDest(VisitInfo vi, siList, destRoot, studyKey) {
		println "preparing visit data for CBF upload (${vi.subjectId} ${vi.visitId} - ${siList[0].siteId})"
		for(SeriesInfo si in siList) {
			File srcRoot = new File(si.srcPath)
			def files = []
			srcRoot.eachFileRecurse { files << it }
			dispatchSeriesPrep(vi, si, files, destRoot, studyKey)
		}
		vi.writeContents(studyKey)
	}
	
	def boolean hasSiemensB0(siList) {
		def b0Series = []
		for(SeriesInfo si in siList) {
			if (si.seriesName.startsWith('B0')) {
				b0Series << si.seriesName;
			}
		}
		return (b0Series.size() == 2 && b0Series.contains('B0_phase1') && b0Series.contains('B0_mag1') )
	}
	
	def copyDicomFiles2Dest(VisitInfo vi, SeriesInfo si, files, destRoot, studyKey) {
		def dicomFiles = []
		def dicomTarBalls = []
		files.each { f ->
			if (isDicomTarBall(f)) dicomTarBalls << f
			if (DicomFileUtilities.isDicomOrAcrNemaFile(f.canonicalPath)) dicomFiles << f
		}
		if (!dicomFiles.empty) {
			File destDir = new File(destRoot,si.seriesName);
			destDir.mkdir();
			for(dcmFile in dicomFiles) {
				File dest = new File(destDir, dcmFile.name)
				copyFile(dcmFile,dest);
			}
		} else if (!dicomTarBalls.empty) {
			def dtb = dicomTarBalls[0]
			File tmpDir = new File(destRoot,"tmp")
			tmpDir.mkdir()
			extractTarBall(dtb.canonicalPath, tmpDir.canonicalPath)
			
			def destDir = new File(destRoot,si.seriesName)
			File dicomDir = new File(tmpDir,'DICOM')
			if (dicomDir.directory) {
				println "renaming $dicomDir to $destDir"
				dicomDir.renameTo(new File(destRoot,si.seriesName))
				tmpDir.delete();
			} else {
			      def dcmFiles = []
				  tmpDir.eachFileRecurse { f -> if (f.file) dcmFiles << f }
				  for(dcmFile in dcmFiles) {
					  File dest = new File(destDir, dcmFile.name)
					  copyFile(dcmFile,dest);
				  }
				  // delete tmpDir and all its contents
				  new AntBuilder().delete(dir: tmpDir.canonicalPath)
			}
		}
	}
	
	def prepDestB0(VisitInfo vi, siList, destRoot, studyKey) {
		if (hasSiemensB0(siList)) {
			for(SeriesInfo si in siList) {
				if (si.seriesName.startsWith('B0')) {
					File srcRoot = new File(si.srcPath)
					def files = []
					srcRoot.eachFileRecurse { files << it }
					copyDicomFiles2Dest(vi, si, files, destRoot, studyKey)
				}
			}
			return
		}
		def dicomTarBalls = []
		for(SeriesInfo si in siList) {
			File srcRoot = new File(si.srcPath)
			def files = []
			srcRoot.eachFileRecurse { files << it }
			files.each { f -> if (isDicomTarBall(f)) dicomTarBalls << f}
		}
		assert !dicomTarBalls.empty
		// group fm1 and fm2
		def b01TarBalls = []
		def b02TarBalls = []
		if (dicomTarBalls.size() == 8) {
			dicomTarBalls.each { println it }
			dicomTarBalls.each { dtb -> 
				if (dtb =~ /B0_mag1/ || dtb =~ /B0_phase1/ || dtb =~ /B0_real1/ || dtb =~ /B0_imag1/) { b01TarBalls << dtb
				}
				if (dtb =~ /B0_mag2/ || dtb =~ /B0_phase2/ || dtb =~ /B0_real2/ || dtb =~ /B0_imag2/) { b02TarBalls << dtb
				}
			}
			assert b01TarBalls.size() == 4
			for(dtb in b01TarBalls) {
				add2B0Dest(dtb, destRoot,'fm1');
			}
			for(dtb in b02TarBalls) {
				add2B0Dest(dtb, destRoot,'fm2');
			}
			println "finished GE b0 preparation for  ${vi.subjectId} - " + vi.visitDestDir
		}
	}
	
	def add2B0Dest(dtb, destRoot, destSeriesName) {
		extractTarBall(dtb.canonicalPath, destRoot.canonicalPath)
		File dicomDir = new File(destRoot,'DICOM')
		assert dicomDir.directory
		File destDir = new File(destRoot,destSeriesName)
		destDir.mkdir()
		moveFiles( dicomDir.canonicalPath, destDir.canonicalPath)
		boolean ok = dicomDir.delete()
	}
	
	def dispatchSeriesPrep(VisitInfo vi, SeriesInfo si, files, destRoot, studyKey) {
		def pfiles = []
		def dicomTarBalls = []
		def brikFiles = []
		def extraFiles = []
		files.each { f ->  
			if (isPFile(f)) pfiles << f
			if (isDicomTarBall(f)) dicomTarBalls << f
			if (isBrikFile(f)) brikFiles << f	
			if (f.name.equals('ImageWrapper.xml')) extraFiles << f
		}
		if (!pfiles.empty) {
			def pfile = pfiles[0]
			String name = pfile.name
			name = name.replaceFirst(/\..+$/,'.7')
			File dest = new File(destRoot,name)
			copyFile(pfile,dest)
			File wrapperFile = findInSameDirAs(pfile, extraFiles)
			if (wrapperFile != null) {
				String wfn = wrapperFile.name.replaceFirst(/\.xml$/,"_${si.seriesName}.xml")
				File wDest = new File(destRoot,wfn)
				copyFile(wrapperFile,wDest)
			}
			def contentMap = vi.studyContentMap[studyKey]
			contentMap[si.seriesName] = name
		} else if (!dicomTarBalls.empty) {
			def dtb = dicomTarBalls[0]
			extractTarBall(dtb.canonicalPath, destRoot.canonicalPath)
			File dicomDir = new File(destRoot,'DICOM')
			assert dicomDir.directory
			def destDir = new File(destRoot,si.seriesName)
			println "renaming $dicomDir to $destDir"
			dicomDir.renameTo(new File(destRoot,si.seriesName))
		} else if (!brikFiles.empty) {
			String brikName = ''
			for(bf in brikFiles) {
				String name = si.seriesName + "+orig"
				name += (bf.name =~ /\.HEAD$/) ? '.HEAD' : '.BRIK'
				if (bf.name =~ /\.BRIK$/) { brikName = name
				}
				File dest = new File(destRoot,name)
				copyFile(bf,dest)
			}
			File wrapperFile = findInSameDirAs(brikFiles[0], extraFiles)
			if (wrapperFile != null) {
				String wfn = wrapperFile.name.replaceFirst(/\.xml$/,"_${si.seriesName}.xml")
				File wDest = new File(destRoot,wfn)
				copyFile(wrapperFile,wDest)
			}
			def contentMap = vi.studyContentMap[studyKey]
			contentMap[si.seriesName] = brikName
			
			// throw new RuntimeException("Brik files are not supported yet!")
		}
	}
	
	static void copyFile(File src, File dest) {
		new AntBuilder().copy(file:"$src.canonicalPath",tofile:"$dest.canonicalPath",overwrite:'true')
	}
	
	def findInSameDirAs(File refFile, files) {
		for(f in files) {
			if (refFile.parent.equals(f.parent)) {
				return f
			}
		}
		return null
	}
	def hasASL(VisitInfo vi) {
		def files = []
		for(SeriesInfo si in vi.siList) {
			if (si.seriesName.toLowerCase().startsWith("asl")) {
				File srcRoot = new File(si.srcPath)
				srcRoot.eachFileRecurse {f ->
					if (f.file && isASLFile(f)) files << f
				}
				if (!files.empty) return true
			}
		}
		return !files.empty
	}
	
	
	
	def isASLFile(File f) {
		String n = f.name;
		return (n =~ /\.7$/ || n =~ /\.7_PFILE$/ || n =~ /\.tar\.gz$/ || n =~ /\.HEAD$/ || n =~ /\.BRIK$/)
	}
	
	def isDicomTarBall(File f) {
		f.name.equals('DICOM.tar.gz')
	}
	
	def isPFile(File f) {
		String n = f.name;
		return (n =~ /\.7$/ || n =~ /\.7_PFILE$/)
	}
	
	def isBrikFile(File f) {
		String n = f.name;
		return (n =~ /\.HEAD$/ || n =~ /\.BRIK$/)
	}
	
	def extractTarBall(String tarballPath, String dest) {
		def ant = new AntBuilder()
		ant.untar(src:tarballPath, dest:dest,compression:"gzip")
	}
	
	def moveFiles(String fromDir, String toDir) {
		def ant = new AntBuilder()
		ant.move(todir:toDir) { fileset(dir:fromDir) }
	}
	
	
	def extractSourceSeries(String expName) {
		def siMap = [:]
		def expDir = new File("${srcRoot}/$expName")
		def subjectDirs = []
		expDir.eachFile { f ->
			if (f.directory) {
				def count = 0;
				f.eachFileRecurse {
					if (it.file) count++
				}
				if (count > 0) subjectDirs << f
			}
		}
		def suffixMap = [:]
		for(sDir in subjectDirs) {
			String subjectId = sDir.name
			def siList = siMap[subjectId]
			if (siList == null) {
				siList = []
				siMap[subjectId] = siList
			}
			def visitDirs = [];
			sDir.eachFile { f -> 
				if (f.directory && f.name.startsWith("scanVisit")) visitDirs << f
			}
			for(vDir in visitDirs) {
				String s = vDir.name
				def matcher = s =~ /(\d+)__000(\d$)/
				// assert matcher.matches(), "no match for $s"
				def siteId = matcher[0][1]
				def visitId = matcher[0][2]
				
				def scannerDirs = []
				vDir.eachFile {
					if (it.directory && it.name.startsWith("MRI")) scannerDirs << it
				}
				for(dir in scannerDirs) {
					def sn = dir.name
					sn = sn.replaceAll(/__000[12]$/,'')
					if (sn != 'MRI-E' && sn != 'MRI-W' && sn != 'MRI-verio' && sn != 'MRI-ge') {
						sn = ''
					}
					def seriesDirs = []
					dir.eachFile {
						if (it.directory) seriesDirs << it
					}
					for(sd in seriesDirs) {
						
						sd.eachFileRecurse {f -> 
							if (f.file) {
								def m = f.name =~ /\.(.+)$/
								m.each { all, suf ->
									suffixMap[suf] = suf
								}
							}
						}
						SeriesInfo si = new SeriesInfo(subjectId:subjectId, visitId:visitId, 
								seriesName:sd.name, srcPath: sd.absolutePath, siteId:siteId, magnetName:sn)
						siList << si
					}
				}
			}
		}
		println "Suffix List"
		suffixMap.each { k,v -> println k }
		println '-' * 40
		return siMap
	}
	
	static void handlePhase3() {
		ASLImgDataOrganizer aido = new ASLImgDataOrganizer()
		def xmlRoot = '/home/bozyurt/dev/java/clinical/scripts/results'
		def siteMap = aido.prepSiteVisitMap([
			"${xmlRoot}/exp_fBIRNPhaseIII__0090_duke.xml",
			"${xmlRoot}/exp_fBIRNPhaseIII__0090_ucsd.xml",
			"${xmlRoot}/exp_fBIRNPhaseIII__0090_uci.xml",
			"${xmlRoot}/exp_fBIRNPhaseIII__0090_ucsf.xml",
			"${xmlRoot}/exp_fBIRNPhaseIII__0090_mrn.xml",
			"${xmlRoot}/exp_fBIRNPhaseIII__0090_uiowa.xml",
			"${xmlRoot}/exp_fBIRNPhaseIII__0090_umn.xml"
		])
		def sites = [
			'duke',
			'ucsd',
			'uci',
			'ucsf',
			'mrn',
			'uiowa',
			'umn'
		]
		for(site in ['duke']) {
			
			def siMap = aido.extractSourceSeries("fBIRNPhaseIII__0090/$site")
			println siMap.size()
			int count = 0
			def vMap = [:]
			siMap.each {key, list ->
				println key
				list.each { si ->
					println "\t" + si.toString();
					count++
					def path = new File(si.srcPath).parent
					vMap[path] = path
				}
			}
			println "count:$count # of scanVisits:" + vMap.size()
			println '-' * 40
			
			aido.copy2Dest("fBIRNPhaseIII__0090/$site", siMap, siteMap)
		}
	}
	
	static void handlePhase3B0() {
		ASLImgDataOrganizer aido = new ASLImgDataOrganizer()
		def xmlRoot = '/home/bozyurt/dev/java/clinical/scripts/results'
		def siteMap = aido.prepSiteVisitMap([
			"${xmlRoot}/exp_fBIRNPhaseIII__0090_duke.xml",
			"${xmlRoot}/exp_fBIRNPhaseIII__0090_ucsd.xml",
			"${xmlRoot}/exp_fBIRNPhaseIII__0090_uci.xml",
			"${xmlRoot}/exp_fBIRNPhaseIII__0090_ucsf.xml",
			"${xmlRoot}/exp_fBIRNPhaseIII__0090_mrn.xml",
			"${xmlRoot}/exp_fBIRNPhaseIII__0090_uiowa.xml",
			"${xmlRoot}/exp_fBIRNPhaseIII__0090_umn.xml"
		])
		def sites = [
			'duke',
			'ucsd',
			'uci',
			'ucsf',
			'mrn',
			'uiowa',
			'umn'
		]
		for(site in ['ucsd']) {			
			def siMap = aido.extractSourceSeries("fBIRNPhaseIII__0090/$site")
			println siMap.size()
			int count = 0
			def vMap = [:]
			siMap.each {key, list ->
				println key
				list.each { si ->
					println "\t" + si.toString();
					count++
					def path = new File(si.srcPath).parent
					vMap[path] = path
				}
			}
			println "count:$count # of scanVisits:" + vMap.size()
			println '-' * 40
			
			aido.copyB02Dest("fBIRNPhaseIII__0090/$site", siMap, siteMap)
		}
	}
	
	static void handleWCTS() {
		ASLImgDataOrganizer aido = new ASLImgDataOrganizer()
		def xmlRoot = '/home/bozyurt/dev/java/clinical/scripts/cbfbirn/results'
		def siteMap = aido.prepSiteVisitMap([
			"${xmlRoot}/exp_ucsd.xml",
			"${xmlRoot}/exp_uci.xml"
		])
		
		def siMap = aido.extractSourceSeries('FBIRNSubject2007TW__0051')
		println siMap.size()
		int count = 0
		def vMap = [:]
		siMap.each {key, list ->
			println key
			list.each { si ->
				println "\t" + si.toString();
				count++
				def path = new File(si.srcPath).parent
				vMap[path] = path
			}
		}
		println "count:$count # of scanVisits:" + vMap.size()
		
		aido.copy2Dest('FBIRNSubject2007TW__0051', siMap, siteMap)
	}
	
	static void handleWCTSB0() {
		ASLImgDataOrganizer aido = new ASLImgDataOrganizer()
		def xmlRoot = '/home/bozyurt/dev/java/clinical/scripts/results'
		def siteMap = aido.prepSiteVisitMap([
			"${xmlRoot}/exp_FBIRNSubject2007TW__0051_ucsd.xml",
			"${xmlRoot}/exp_FBIRNSubject2007TW__0051_uci.xml"
		])
		
		def siMap = aido.extractSourceSeries('FBIRNSubject2007TW__0051')
		println siMap.size()
		int count = 0
		def vMap = [:]
		siMap.each {key, list ->
			println key
			list.each { si ->
				println "\t" + si.toString();
				count++
				def path = new File(si.srcPath).parent
				vMap[path] = path
			}
		}
		println "count:$count # of scanVisits:" + vMap.size()
		
		aido.copyB02Dest('FBIRNSubject2007TW__0051', siMap, siteMap)
	}
	
	static void main(args) {
		// handlePhase3()
		handleWCTSB0()
		//	handlePhase3B0()
	}
}
