package gtools.utils;

import groovy.sql.Sql;
import org.dbunit.database.IDatabaseConnection;
import org.dbunit.database.DatabaseConnection;
import org.dbunit.database.QueryDataSet;
import org.dbunit.dataset.xml.FlatDtdDataSet;
import org.dbunit.dataset.xml.FlatXmlDataSet;
import org.dbunit.dataset.ITable;
import org.dbunit.dataset.ITableMetaData;
import org.dbunit.dataset.ITableIterator;
import org.dbunit.dataset.IDataSet;
import org.dbunit.dataset.Column;
import org.dbunit.dataset.*;
import org.dbunit.dataset.datatype.DataType;
import org.dbunit.dataset.datatype.UnknownDataType;

import org.dbunit.DBTestCase;
import org.dbunit.Assertion;
import org.dbunit.operation.DatabaseOperation;
import org.dbunit.dataset.filter.DefaultColumnFilter;
import org.dbunit.dataset.SortedTable;
import junit.framework.Assert;

/**
  * 
  * @author I. Burak Ozyurt
  * @version $Id: DBStaticDataIntegrityTest.groovy,v 1.5 2008/04/11 00:27:09 bozyurt Exp $
  */
class DBStaticDataIntegrityTest extends DBTestCase {
	Properties props;
	def db;
	def dbType;
	
	public DBStaticDataIntegrityTest() {
		super()
	}
	
	
	public DBStaticDataIntegrityTest(String testName) {
		super(testName)
		props = GenUtils.loadProperties("gtools.properties")
		assert props
		if ( props.getProperty('db.type') == 'postgres') {
			db = Sql.newInstance(props.getProperty('db.url'), 
			 		 props.getProperty('db.user'),props.getProperty('db.pwd'),  
			   		 'org.postgresql.Driver');
			dbType = 'postgres';
		 } else {
		     db = Sql.newInstance(props.getProperty('db.url'), 
		    		 props.getProperty('db.user'),props.getProperty('db.pwd'),  
		    		 'oracle.jdbc.driver.OracleDriver');
		     dbType = 'oracle';
		 }  	   		
	}
	
	public void initialize(String usersXMLFile, String staticDataFile, String schemaName = null) {
	  props = new Properties();
	  props.setProperty('static.data.file', staticDataFile)
          if (schemaName) {
             props.setProperty('schema.name', schemaName.toUpperCase());
          }
	  def reader = new FileReader(usersXMLFile);
	  def recs = new XmlParser().parse(reader);
	  def theDBElem = recs.databases[0].database.find { it.'@default' =='true'};
	  assert(theDBElem)
	  def dbID = theDBElem.'@id';
	  def dbURL = theDBElem.'db-url'[0].text();
	  this.dbType = theDBElem.'db-type'[0].text();
      def theDBUserElem = recs.dbusers[0].dbuser.find { it.'@dbid' == dbID }
      assert theDBUserElem;
      def user = theDBUserElem.'@name';
      def pwd = theDBUserElem.'@pwd';
      println "Connecting to the database via JDBC: $dbURL ...";
      if ( dbType == 'postgres') {
             db = Sql.newInstance(dbURL, user, pwd,'org.postgresql.Driver');
      } else {
             db = Sql.newInstance(dbURL, user, pwd, 'oracle.jdbc.driver.OracleDriver');
      }
		
	}

	protected IDataSet getDataSet() {
 	  assert(props)
	  def staticDataFile = props.getProperty('static.data.file');
	  return new FlatXmlDataSet( new FileInputStream( 
				  GenUtils.adjustPath(staticDataFile) ) );	
	}
	
	protected DatabaseOperation getSetUpOperation() throws Exception {
		return DatabaseOperation.NONE;	
	}
	
	protected DatabaseOperation getTearDownOperation() throws Exception {
		return DatabaseOperation.NONE;
	}
	
	protected IDatabaseConnection getConnection() {
		assert db
            def schemaName = props.getProperty('schema.name');
            if ( schemaName) {   
	       return new  DatabaseConnection(db.connection,schemaName)
            } else {
                return new  DatabaseConnection(db.connection);
            }
	}
	
	public void testStaticDataIntegrity() {
	    IDataSet dbDataSet = getConnection().createDataSet();
	    IDataSet expectedDataSet = getDataSet();
	    println 'Tables in schema:'
	    ITableIterator it1 = dbDataSet.iterator();
	    while( it1.next()) {
                println it1.tableMetaData.tableName;
            }
            println '--------------'; 
	    
	    ITable actualTable = dbDataSet.getTable("nc_visittype");
	    ITable expectedTable = expectedDataSet.getTable("nc_visittype");
	    
	    actualTable = DefaultColumnFilter.includedColumnsTable(actualTable, 
	    		["visittype"] as String[] );
	    expectedTable = DefaultColumnFilter.includedColumnsTable(expectedTable, 
	    		["visittype"] as String[] );
	    
	    DBAssertion.assertFullyContains(new SortedTable(expectedTable), 
	    		new SortedTable(actualTable, expectedTable.getTableMetaData()) );
	    // nc_assessmentinformant
	    actualTable = dbDataSet.getTable("nc_assessmentinformant");
	    expectedTable = expectedDataSet.getTable("nc_assessmentinformant");
	    actualTable = DefaultColumnFilter.includedColumnsTable(actualTable, 
	    		["informantrelation"] as String[] );
	    expectedTable = DefaultColumnFilter.includedColumnsTable(expectedTable, 
	    		["informantrelation"] as String[] );
	    DBAssertion.assertFullyContains(new SortedTable(expectedTable), 
	    		new SortedTable(actualTable, expectedTable.getTableMetaData()) );	
	    
     //	  nc_userclass
	    actualTable = dbDataSet.getTable("nc_userclass");
	    expectedTable = expectedDataSet.getTable("nc_userclass");
	    actualTable = DefaultColumnFilter.includedColumnsTable(actualTable, 
	    		["userclass","name", "description"] as String[] );
	    expectedTable = DefaultColumnFilter.includedColumnsTable(expectedTable, 
	    		["userclass","name", "description"] as String[] );
	    DBAssertion.assertFullyContains(new SortedTable(expectedTable), 
	    		new SortedTable(actualTable, expectedTable.getTableMetaData()) );
	    
	    // nc_dataclassification
	    actualTable = dbDataSet.getTable("nc_dataclassification");
	    expectedTable = expectedDataSet.getTable("nc_dataclassification");
	    actualTable = DefaultColumnFilter.includedColumnsTable(actualTable, 
	    		["name", "description"] as String[] );
	    expectedTable = DefaultColumnFilter.includedColumnsTable(expectedTable, 
	    		["name", "description"] as String[] );
	    DBAssertion.assertFullyContains(new SortedTable(expectedTable), 
	    		new SortedTable(actualTable, expectedTable.getTableMetaData()) );
	    
      //	  nc_tableid
	    actualTable = dbDataSet.getTable("nc_tableid");
	    expectedTable = expectedDataSet.getTable("nc_tableid");
	    actualTable = DefaultColumnFilter.includedColumnsTable(actualTable, 
	    		["tablename"] as String[] );
	    expectedTable = DefaultColumnFilter.includedColumnsTable(expectedTable, 
	    		["tablename"] as String[] );
	    DBAssertion.assertFullyContains(new SortedTable(expectedTable), 
	    		new SortedTable(actualTable, expectedTable.getTableMetaData()) );
	    
      //	nc_tupleclass
	    actualTable = dbDataSet.getTable("nc_tupleclass");
	    expectedTable = expectedDataSet.getTable("nc_tupleclass");
	    actualTable = DefaultColumnFilter.includedColumnsTable(actualTable, 
	    		["tupleclass"] as String[] );
	    expectedTable = DefaultColumnFilter.includedColumnsTable(expectedTable, 
	    		["tupleclass"] as String[] );
	    DBAssertion.assertFullyContains(new SortedTable(expectedTable), 
	    		new SortedTable(actualTable, expectedTable.getTableMetaData()) );
	   
      //	nc_animalspecies
	    actualTable = dbDataSet.getTable("nc_animalspecies");
	    expectedTable = expectedDataSet.getTable("nc_animalspecies");
	    actualTable = DefaultColumnFilter.includedColumnsTable(actualTable, 
	    		["name", "strain"] as String[] );
	    expectedTable = DefaultColumnFilter.includedColumnsTable(expectedTable, 
	    		["name", "strain"] as String[] );
	    DBAssertion.assertFullyContains(new SortedTable(expectedTable), 
	    		new SortedTable(actualTable, expectedTable.getTableMetaData()) );
	    
      //	nc_userstatus
	    actualTable = dbDataSet.getTable("nc_userstatus");
	    expectedTable = expectedDataSet.getTable("nc_userstatus");
	    actualTable = DefaultColumnFilter.includedColumnsTable(actualTable, 
	    		["userstatus", "name", "description"] as String[] );
	    expectedTable = DefaultColumnFilter.includedColumnsTable(expectedTable, 
	    		["userstatus", "name", "description"] as String[] );
	    DBAssertion.assertFullyContains(new SortedTable(expectedTable), 
	    		new SortedTable(actualTable, expectedTable.getTableMetaData()) );
	    
      //	nc_assessmentstatus
	    actualTable = dbDataSet.getTable("nc_assessmentstatus");
	    expectedTable = expectedDataSet.getTable("nc_assessmentstatus");
	    actualTable = DefaultColumnFilter.includedColumnsTable(actualTable, 
	    		["status", "name", "description"] as String[] );
	    expectedTable = DefaultColumnFilter.includedColumnsTable(expectedTable, 
	    		["status", "name", "description"] as String[] );
	    DBAssertion.assertFullyContains(new SortedTable(expectedTable), 
	    		new SortedTable(actualTable, expectedTable.getTableMetaData()) );
	    
      //	nc_securityclassification
	    actualTable = dbDataSet.getTable("nc_securityclassification");
	    expectedTable = expectedDataSet.getTable("nc_securityclassification");
	    actualTable = DefaultColumnFilter.includedColumnsTable(actualTable, 
	    		["securityclassification", "description"] as String[] );
	    expectedTable = DefaultColumnFilter.includedColumnsTable(expectedTable, 
	    		["securityclassification", "description"] as String[] );
	    DBAssertion.assertFullyContains(new SortedTable(expectedTable), 
	    		new SortedTable(actualTable, expectedTable.getTableMetaData()) );
	    
      //	nc_ontologysource
	    actualTable = dbDataSet.getTable("nc_ontologysource");
	    expectedTable = expectedDataSet.getTable("nc_ontologysource");
	    actualTable = DefaultColumnFilter.includedColumnsTable(actualTable, 
	    		["ontologysource"] as String[] );
	    expectedTable = DefaultColumnFilter.includedColumnsTable(expectedTable, 
	    		["ontologysource"] as String[] );
	    DBAssertion.assertFullyContains(new SortedTable(expectedTable), 
	    		new SortedTable(actualTable, expectedTable.getTableMetaData()) );

	    //	nc_researchgrouptype
	    actualTable = dbDataSet.getTable("nc_researchgrouptype");
	    expectedTable = expectedDataSet.getTable("nc_researchgrouptype");
	    actualTable = DefaultColumnFilter.includedColumnsTable(actualTable, 
	    		["name"] as String[] );
	    expectedTable = DefaultColumnFilter.includedColumnsTable(expectedTable, 
	    		["name"] as String[] );
	    DBAssertion.assertFullyContains(new SortedTable(expectedTable), 
	    		new SortedTable(actualTable, expectedTable.getTableMetaData()) );

	    //	nc_schemaversion
	    if (!dbType.equals('postgres')) {
	      actualTable = dbDataSet.getTable("nc_schemaversion");
	      expectedTable = expectedDataSet.getTable("nc_schemaversion");
	      actualTable = DefaultColumnFilter.includedColumnsTable(actualTable, 
	        		["majorversion", "minorversion"] as String[] );
	      expectedTable = DefaultColumnFilter.includedColumnsTable(expectedTable, 
	    		["majorversion", "minorversion"] as String[] );
	      DBAssertion.assertFullyContains(new SortedTable(expectedTable), 
	    	  	new SortedTable(actualTable, expectedTable.getTableMetaData()) );

            }
            println "The necessary static data to ensure correct operation seems to exists in your database!";
	}

}

class DBAssertion {
   static ColumnComparator COLUMN_COMPARATOR = new ColumnComparator();

   public static void assertFullyContains(ITable expectedTable, ITable actualTable) {
      if ( expectedTable == actualTable) return;
      ITableMetaData expectedMetaData = expectedTable.getTableMetaData();
      ITableMetaData actualMetaData = actualTable.getTableMetaData();
      String expectedTableName = expectedMetaData.getTableName();

      // Verify columns
      Column[] expectedColumns = getSortedColumns(expectedMetaData);
      Column[] actualColumns = getSortedColumns(actualMetaData);
      Assert.assertEquals('column count (table=$expectedTableName)',
                expectedColumns.length, actualColumns.length);
        def idx = 0; 
        for (ec in expectedColumns) {
            String expectedName = ec.columnName;
            String actualName = actualColumns[idx].columnName;
            if (!expectedName.equalsIgnoreCase(actualName))
            {
                Assert.fail("expected columns " + getColumnNamesAsString(expectedColumns) +
                        " but was " + getColumnNamesAsString(actualColumns) +
                        " (table=$expectedTableName)");
            }
            ++idx;
        }
        // make sure actual table has >= rows than the expected table
        Assert.assertTrue("Actual table row count (table=$expectedTableName) was less than expected", 
             expectedTable.rowCount <= actualTable.rowCount); 

       def actualTableMap = [:];
       for(ec in expectedColumns) {
          def atColValueMap = actualTableMap[ec.columnName];
	  if (atColValueMap == null) {
             atColValueMap = [:];
	     actualTableMap.put(ec.columnName, atColValueMap);
	  }
          for (i in 0..<actualTable.rowCount) {
            def actualValue = actualTable.getValue(i, ec.columnName);
	    List idxs = atColValueMap["$actualValue"];
	    if (idxs == null) {
               idxs = new ArrayList(1);
	       atColValueMap["$actualValue"] = idxs;
	    }
	    idxs << i;
	  }
       }
       // values as strings
       for (i in 0..<expectedTable.rowCount)
        {
	    def first = true;
	    def actualRowIdx = -1;
            for (j in 0..<expectedColumns.length)
            {
                Column expectedColumn = expectedColumns[j];
                Column actualColumn = actualColumns[j];

                String columnName = expectedColumn.getColumnName();
                def expectedValue = expectedTable.getValue(i, columnName);

                def atColValueMap = actualTableMap[columnName];
                def idxList = atColValueMap["$expectedValue"];
		if (idxList == null) {
                    Assert.fail("value (table=$expectedTableName, row=$i, col=$columnName): <" +
                            "$expectedValue> not found!");
		} else {
		   if (first) {
		     actualRowIdx = idxList[0];
		     first = false;
		   }
		}
                def actualValue = actualTable.getValue(actualRowIdx, columnName);

                DataType dataType = getComparisonDataType(
                        expectedTableName, expectedColumn, actualColumn);
                if (dataType.compare(expectedValue, actualValue) != 0)
                {
                    Assert.fail("value (table=$expectedTableName, row=$i, col=$columnName): expected:<" +
                            "$expectedValue> but was:<$actualValue>");
                }
            }
        }
   }

   static DataType getComparisonDataType(String tableName, Column expectedColumn,
            Column actualColumn)
    {
        DataType expectedDataType = expectedColumn.dataType;
        DataType actualDataType = actualColumn.dataType;

        // The two columns have different data type
        if (!expectedDataType.getClass().isInstance(actualDataType))
        {
            // Expected column data type is unknown, use actual column data type
            if (expectedDataType instanceof UnknownDataType)
            {
                return actualDataType;
            }

            // Actual column data type is unknown, use expected column data type
            if (actualDataType instanceof UnknownDataType)
            {
                return expectedDataType;
            }

            // Impossible to determine which data type to use
            Assert.fail("Incompatible data types: $expectedDataType, " +
                    "$actualDataType (table=$tableName, col=" +
                    expectedColumn.getColumnName() + ")");
        }

        // Both columns have same data type, return any one of them
        return expectedDataType;
    }

    static Column[] getSortedColumns(ITableMetaData metaData)
            throws DataSetException
    {
        Column[] columns = metaData.columns;    
    
        def sortColumns = [];
	columns.each { sortColumns << it; }
        
        sortColumns.sort(COLUMN_COMPARATOR);
        return sortColumns;
    }    
}


class ColumnComparator implements Comparator {
        public int compare(Object o1, Object o2)
        {
            Column column1 = (Column)o1;
            Column column2 = (Column)o2;

            String columnName1 = column1.columnName;
            String columnName2 = column2.columnName;
            return columnName1.compareToIgnoreCase(columnName2);
        }
}
