package gtools.utils;

import java.sql.SQLException;
import groovy.sql.Sql;
import clinical.server.utils.OracleUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: SRBImagePathImporter.groovy,v 1.6 2009/02/11 20:39:42 bozyurt Exp $
 */


class ExperimentDAO extends DataAccessObject {
	List getFields() {
		return ['uniqueid','name','baseuri', 'isregressiondata'];
	}
}

class HumansubjectDAO extends DataAccessObject {
	List getFields() {
		return ['subjectid','owner','siteid']; 
	}
}//;

class ExpcomponentDAO extends DataAccessObject {
	List getFields() {
		return ['componentid','nc_experiment_uniqueid','subjectid', 'uniqueid',
		'tableid','owner', 'modtime', 'moduser', 'time_stamp', 
		'description','visittype', 'name', 'timeinterval', 'istimeinterval'];
	}	
	
	def create(Map argMap) {
		def uid = nextUID();
		def map = [componentid : null, nc_experiment_uniqueid : null, 
				subjectid :null, uniqueid : uid , tableid: getTableID(),
				owner: ownerID , modtime : modTimeCmd, moduser : ownerID , time_stamp : modTimeCmd,
				description: null, visittype : "'fMRI scan'", istimeinterval : getBoolean(0), 
				timeinterval: null , name: null];
		def sql = prepInsertQuery(argMap, map, true);
		db.executeUpdate sql, [];
	}
}//;

class ExpsegmentDAO extends DataAccessObject {
	List getFields() {
		return ['segmentid','componentid','nc_experiment_uniqueid','subjectid', 'uniqueid',
		'tableid', 'owner','modtime','moduser', 'time_stamp','description',
		'studyid', 'name', 'protocolversion', 'protocolid','name', 'istimeinterval','timeinterval',
		'isbad'];
	}
	
	def create(Map argMap) {    	
		def uid = nextUID();
		def map = [segmentid : null, componentid : null, nc_experiment_uniqueid : null, 
				subjectid :null, uniqueid : uid , tableid: getTableID(),
				owner: ownerID , modtime : modTimeCmd , moduser :ownerID , time_stamp : modTimeCmd,
				istimeinterval : getBoolean(0), timeinterval: null, isbad : getBoolean(0) , name: null];
		def sql = prepInsertQuery(argMap, map, true);
		db.executeUpdate sql, [];
	}
}//;

class ExpstudyDAO extends DataAccessObject {
	List getFields() {
		return ['studyid','componentid','experimentid','subjectid', 'uniqueid',
		'tableid', 'owner','modtime','moduser', 'time_stamp','description',
		'name','istimeinterval', 'timeinterval'];
	}
	def create(Map argMap) {    	
		def uid = nextUID();
		def map = [studyid : null, componentid : null, nc_experiment_uniqueid : null, 
				subjectid :null, uniqueid : uid , tableid: getTableID(),
				owner: ownerID , modtime : modTimeCmd , moduser :ownerID , time_stamp : modTimeCmd,
				istimeinterval : getBoolean(0), timeinterval: null,  name: "'FMRI'"];
		def sql = prepInsertQuery(argMap, map);
		db.executeUpdate sql, [];
	}
}

class RawdataDAO extends DataAccessObject {
	List getFields() {
		return ['uniqueid','tableid', 'owner', 'modtime', 'moduser', 
		'extensionname', 'segmentid','componentid','nc_experiment_uniqueid','subjectid', 
		'datauri','israw','protocolversion', 'protocolid', 'nc_colequipment_uniqueid',
		'ontologysource', 'conceptid', 'isbad'];
	}
	
	def create(Map argMap) {    	
		def uid = nextUID();
		def map = [uniqueid : uid, tableid: getTableID(),
				owner: ownerID , modtime : modTimeCmd , moduser :ownerID, 
				extensionname : "'nc_rawdata'",
				segmentid : null, componentid : null, nc_experiment_uniqueid : null, 
				subjectid :null ,
				datauri : null, israw : getBoolean(1), nc_colequipment_uniqueid : null,	    	           
				ontologysource : "'UNKNOWN'", conceptid : "'9999'", isbad : getBoolean(0) ];
		def sql = prepInsertQuery(argMap, map, true);
		db.executeUpdate sql, [];
	}
}//;

class DeriveddataDAO extends DataAccessObject {
	List getFields() {
		return ['uniqueid','tableid', 'owner', 'modtime', 'moduser', 
		'extensionname', 'segmentid','componentid','nc_experiment_uniqueid','subjectid', 
		'datauri','israw', 'ontologysource', 'conceptid', 'isbad'];
	}
	def create(Map argMap) {    	
		def uid = nextUID();
		def map = [uniqueid : uid, tableid: getTableID(),
				owner: ownerID , modtime : modTimeCmd , moduser :ownerID, extensionname : "'nc_deriveddata'",
				segmentid : null, componentid : null, nc_experiment_uniqueid : null, 
				subjectid :null ,     datauri : null, israw : getBoolean(0), 	    	           
				ontologysource : "'UNKNOWN'", conceptid : "'9999'", isbad : getBoolean(0) ];
		def sql = prepInsertQuery(argMap, map, true);
		db.executeUpdate sql, [];
	}	
}//;

class CollectionequipmentDAO extends DataAccessObject {
	List getFields() {
		return ['uniqueid','tableid', 'owner', 'modtime', 'moduser', 
		'make','model'];
	}
	
	def create(Map argMap) {    	
		def uid = nextUID();
		def map = [uniqueid : uid, tableid: getTableID(),
				owner: ownerID , modtime : modTimeCmd , moduser :ownerID];
		def sql = prepInsertQuery(argMap, map, true);
		db.executeUpdate sql, [];
	}	
}//;


class DataobjectDAO extends DataAccessObject {
	List getFields() {
		return ['uniqueid','tableid', 'owner', 'modtime', 'moduser',
		'extensionname','dataid','objecttypeid','objectname','objectsize','description'];
	}
	def create(Map argMap) {    	
		def uid = nextUID();
		def map = [uniqueid : uid, tableid: getTableID(),
				owner: ownerID , modtime : modTimeCmd , moduser : ownerID,
				extensionname: "'rawData'",  objectname : null,
				description : null];
		def sql = prepInsertQuery(argMap, map, true);
		db.executeUpdate sql, [];
	}
}//;

class DataobjecttypeDAO extends DataAccessObject {
	List getFields() {
		return ['uniqueid','tableid', 'owner', 'modtime', 'moduser',
		'objecttype','applicableactions','description'];
	}
	
	def create(Map argMap) {    	
		def uid = nextUID();	   	
		def map = [uniqueid : uid, tableid: getTableID(),
				owner: ownerID , modtime : modTimeCmd , moduser :ownerID,
				applicableactions: "'DOWNLOAD'",    description : null];
		println "argMap: $argMap"
		println "map: $map"
		assert argMap instanceof java.util.Map
		assert map instanceof java.util.Map
		def sql = prepInsertQuery(argMap, map, true);
		db.executeUpdate sql, [];
	}
	
}//;

class SRBImagePathImporter {
	static String usersXMLFile;
	static def dbu;
	static String dbType;
	
	static void findDirs(parentDir, pathList) {
		parentDir.eachDir { dir ->
			if (!dir.absolutePath.endsWith('MRI__0001/Analysis')) {
				pathList << dir.absolutePath;
				println "added ${dir.absolutePath}"
				findDirs(dir,pathList);
			} 
		}
	}
	
	def static getAllPaths(String rootDir, boolean dirOnly=true) {
		def list = [];
		if (dirOnly) {
			findDirs( new File(rootDir), list)
		} else {
			new File(rootDir).eachFileRecurse { list << it.absolutePath }
		}
		return list;
	}
	
	
	static def getDb() {
		def props = new Properties();
		
		def env = System.getenv();
		if (env['CLINICAL_HOME'] == null) {
		   println "You need to set CLINICAL_HOME environment variable!"
		   System.exit(1);
		}
		usersXMLFile = new File(env['CLINICAL_HOME'],"conf/users.xml");
		
		println "usersXMLFile: $usersXMLFile";
		def reader = new FileReader(usersXMLFile);
		def recs = new XmlParser().parse(reader);
		def theDBElem = recs.databases[0].database.find { it.'@default' =='true'};
		assert(theDBElem)
		def dbID = theDBElem.'@id';
		def theDBUserElem = recs.dbusers[0].dbuser.find { it.'@dbid' == dbID }
		assert theDBUserElem;
		props.setProperty('db.type',  theDBElem.'db-type'[0].text());
		props.setProperty('db.url',theDBElem.'db-url'[0].text());
		props.setProperty('db.user', theDBUserElem.'@name');
		props.setProperty('db.pwd', theDBUserElem.'@pwd');
		SRBImagePathImporter.dbType = theDBElem.'db-type'[0].text();
		println "SRBImagePathImporter.dbType: " +  SRBImagePathImporter.dbType
		
		def db = null;
		if ( props.getProperty('db.type') == 'postgres') {
			db = Sql.newInstance(props.getProperty('db.url'), 
					props.getProperty('db.user'),props.getProperty('db.pwd'),  
					'org.postgresql.Driver');
		} else {
			db = Sql.newInstance(props.getProperty('db.url'), 
					props.getProperty('db.user'),props.getProperty('db.pwd'),  
					'oracle.jdbc.driver.OracleDriver');
		}  
		return db;
		
	}
	
	static void prepareDataObjectTypes(db = null) {
		if (db == null) { db = getDb(); }
		def dotDAO = new DataobjecttypeDAO(db: db);
		dotDAO.setDBType(SRBImagePathImporter.dbType)
		dotDAO.setOwner(SRBImagePathImporter.dbu)
		
		def data = [ [objecttype : "'1_SRB COL'", description : "'SRB collection specified by nc_researchdata.dataURI'"],
				[objecttype : "'1_SRB CHILD COL'", description : "'SRB child collection under the collection specified by nc_researchdata.dataURI'"],
				[objecttype : "'1_SRB FILE'", description : "'SRB File specified by nc_researchdata.dataURI'"],
				[objecttype : "'1_SRB CHILD FILE'", description : "'SRB child file under the collection specified by nc_researchdata.dataURI'"],	              
				[objecttype : "'2_HTML'", description : "'HTML file specified by both nc_researchdata.dataURI and objectname'"],
				[objecttype : "'2_DICOM'", description : "'DICOM image file specified by both nc_researchdata.dataURI and objectname'"],
				[objecttype : "'2_NIFTI'", description : "'NIFTI image file specified by both nc_researchdata.dataURI and objectname'"],
				[objecttype : "'3_SRB COL A_7.5'", description : "'A directory contains ANALYZE_7.5 image data.'"],
				[objecttype : "'3_SRB COL Eprime'", description : "'Eprime file contains fMRI task data.'"],
				[objecttype : "'2_NIFTI GZ'", description : "'Gzipped NIFTI image file specified by both nc_researchdata.dataURI and objectname'"],
				[objecttype : "'3_LOCAL COL DICOM'"],
				[objecttype : "'3_LOCAL COL NIFTI'"] 
				];
		data.each { argMap ->
			dotDAO.create(argMap);
		}
	}
	
	static void prepareWCTSDataObjectTypes(db = null) {
		if (db == null) {db = getDb(); }
		def dotDAO = new DataobjecttypeDAO(db: db);
		dotDAO.setDBType(SRBImagePathImporter.dbType)
		dotDAO.setOwner(SRBImagePathImporter.dbu)
		
		def dotList = dotDAO.find();
		def dotMap = [:];
		dotList.each { dot -> dotMap[dot.objecttype] = dot }
		def data = [ [objecttype : "'3_SRB COL AFNI'"],
				[objecttype : "'3_SRB COL NIFTI'"], 
				[objecttype : "'3_SRB COL DICOM'"], 
				[objecttype : "'3_LOC COL DICOM'"] 
				];
		data.each { argMap ->
			if (! dotMap.containsKey(argMap.objecttype[1..argMap.objecttype.length()-2])) {
				dotDAO.create(argMap);
			}
		}
	}
	
	static void prepareDataObjects4Subject(String rootDir, String subjectID, db = null) {
		if (db == null) {db = getDb(); }
		
		def dotDAO = new DataobjecttypeDAO(db: db);
		def doDAO = new DataobjectDAO(db :db);
		def rawDataDAO = new RawdataDAO(db: db);
		
		doDAO.setDBType(SRBImagePathImporter.dbType)
		doDAO.setOwner(SRBImagePathImporter.dbu)
		
		def dotList = dotDAO.find();
		def dotMap = [:];
		dotList.each { dot -> dotMap[dot.objecttype] = dot }
		def rdList = rawDataDAO.find("where subjectid='$subjectID'",'segmentid');
		rdList.each { rd ->
			def objectType = null;
			def path = "$rootDir/${subjectID}/${rd.datauri}";
			def size = GenUtils.dirSize(path);
			println "PATH: $path (size: $size bytes)" 
			if (rd.datauri =~ /DICOM$/) {
				objectType = dotMap['3_LOCAL COL DICOM']; 
			} else if (rd.datauri =~ /NIFTI$/) {
				objectType = dotMap['3_LOCAL COL NIFTI'];	  		
			}	  
			doDAO.create([dataid : rd.uniqueid, objecttypeid: objectType.uniqueid, objectsize : size]);
		}
	}
	
	static void prepareDataObjects4SubjectInSRB(String subjectID, db = null, local=false) {
		if (db == null) {db = getDb(); }
		def dotDAO = new DataobjecttypeDAO(db: db);
		def doDAO = new DataobjectDAO(db :db);
		def rawDataDAO = new RawdataDAO(db: db);
		doDAO.setDBType(SRBImagePathImporter.dbType)
		doDAO.setOwner(SRBImagePathImporter.dbu)
		
		def storageType = (local) ? 'LOC' : 'SRB';
		
		Random r = new Random();
		def dotList = dotDAO.find();
		def dotMap = [:];
		dotList.each { dot -> dotMap[dot.objecttype] = dot }
		def rdList = rawDataDAO.find("where subjectid='$subjectID'",'segmentid');
		rdList.each { rd ->
			def objectType = null;
			def path = "${rd.datauri}";
			// just a random size for testing
			def size = 5000000 + r.nextInt(1000000);
			println "PATH: $path (size: $size bytes)"  
			if (rd.datauri =~ /AFNI$/) {
			    objectType = local ? dotMap['3_LOC COL AFNI'] : dotMap['3_SRB COL AFNI']; 
			} else if (rd.datauri =~ /NIFTI4D$/) {
			   objectType = local ? dotMap['3_LOC COL NIFTI'] : dotMap['3_SRB COL NIFTI'];
			} else if (rd.datauri =~ /DICOM$/) {
			   objectType = local ? dotMap['3_LOC COL DICOM'] : dotMap['3_SRB COL DICOM'];  		
			} else if ( rd.datauri =~ /DICOM\.tar\.gz$/) {
			   objectType = local ? dotMap['3_LOC FILE DICOM'] : dotMap['3_SRB FILE DICOM'];
			}
			if ( objectType != null) {  
				doDAO.create( [dataid : rd.uniqueid, objecttypeid: objectType.uniqueid, objectsize : size]);
			}
		}
	}
	
	static def createGetExperiment(String expName, db = null) {
		if (db == null) { db = getDb() }
		def expDAO = new ExperimentDAO(db: db);
		//TODO
		return null;
	}
	
	static void prepSegAndPaths(db, segmentDAO, rawDataDAO, paramMap, segName, paths) {
		def visitID = paramMap.visitID;
		def subjectID = paramMap.subjectID;
		def expID = paramMap.expID;
		def segmentID = paramMap.segmentID;
		def scheme = paramMap.scheme;
		def collectionEquipmentId = paramMap.collectionEquipmentId
		// assert scheme
		assert collectionEquipmentId
		
		def wherePart = "where nc_experiment_uniqueid = $expID and subjectid = '$subjectID' and " + 
				"componentid = $visitID and segmentid = $segmentID";
		def segList = segmentDAO.find(wherePart,null);
		if (segList.isEmpty()) {
			segmentDAO.create([segmentid: segmentID, componentid :visitID, 
					subjectid : "'${subjectID}'", nc_experiment_uniqueid : expID,
					name : "'$segName'", protocolversion : 1, 
					protocolid : "'Subjects screening protocol'",
					studyid: null]);
		}
		paths.each { fullPath -> 
			def wp = "where nc_experiment_uniqueid = $expID and subjectid = '$subjectID' and " + 
					"componentid = $visitID " +
					" and segmentid = $segmentID and datauri = '${scheme}$fullPath'";
			def rdList = rawDataDAO.find(wp, null);
			if ( rdList.isEmpty()) { 
				rawDataDAO.create([segmentid: segmentID, componentid :visitID,
						subjectid : "'${subjectID}'", nc_experiment_uniqueid : expID,
						protocolversion : 1, 
						protocolid : "'Subjects screening protocol'",
						datauri : "'${scheme}$fullPath'",
						nc_colequipment_uniqueid : collectionEquipmentId] );
			}
		}
	}
	
	// assumption: the subject(s) are already registered with the experiment
	// segments are given arbitrary ids and timestamps
	// so use this method for realistic looking test data generation
	
	static void importSubjectImageData(Map paramMap, db = null) {
		if (db == null) { db = getDb() }
		def hsDAO = new HumansubjectDAO(db: db);
		def visitDAO = new ExpcomponentDAO(db : db);
		def studyDAO = new ExpstudyDAO(db: db);
		def segmentDAO = new ExpsegmentDAO(db: db);
		def rawDataDAO = new RawdataDAO(db: db);
		def expDAO = new ExperimentDAO(db: db);
		def ceDAO = new CollectionequipmentDAO(db : db);
		[visitDAO,segmentDAO, rawDataDAO].each { dao ->
			dao.setOwner(dbu)  
			dao.setDBType(dbType)
		}
		
		def model = paramMap.model;
		assert model
		def ceList = ceDAO.find("where model='$model'",null);
		assert !ceList.isEmpty()
		def ce = ceList.get(0);
		
		def sortBy = 'subjectid';
		def subjects = null; 
		if (SRBImagePathImporter.dbType == 'postgres') {
			subjects = hsDAO.find('', sortBy);
		} else {	  		
			subjects = hsDAO.find('where siteid is not null', sortBy);
		}
		subjects.each { s -> hsDAO.dumpVO(s) }
		
		def expName = paramMap.expName; 
		def experiments = expDAO.find("where name ='$expName'", null); 
		def expID = null;
		experiments.each { exp -> expDAO.dumpVO(exp); 
			expID = exp.uniqueid; }
		assert expID != null 
		Map visitSegments = paramMap.visitSegments;                   
		if ( !visitSegments) {
			return;
		}	
		
		String uriScheme = "srb://";
		if ( paramMap.uriScheme) {
			if (paramMap.uriScheme == 'none') { 
				uriScheme = '';
			} else {
				uriScheme = paramMap.uriScheme;
			}
		}
		String subjectID = paramMap.subjectID; 
		assert subjectID != null 
		visitSegments.each { visitid,segs -> 
			def visitList = visitDAO.find("where nc_experiment_uniqueid = $expID and " + 
					"subjectid = '$subjectID' and componentid = $visitid",null);
			if (visitList.isEmpty() ) {
				visitDAO.create([componentid : visitid, nc_experiment_uniqueid : expID,
						subjectid : "'$subjectID'"]); 
			}
			visitList = visitDAO.find("where nc_experiment_uniqueid = $expID and " + 
					"subjectid = '$subjectID' and componentid = $visitid",null);
			assert !visitList.isEmpty() 
			def visit = visitList.get(0);
			
			// println "segs:$segs"
			int segIdx = 1;
			segs.each { segName, paths -> 
				def argMap = [visitID : visit.componentid, subjectID : subjectID , 
						expID : expID, scheme : uriScheme, segmentID : segIdx, 
						collectionEquipmentId : ce.uniqueid];
				
				SRBImagePathImporter.prepSegAndPaths(db, segmentDAO, rawDataDAO, argMap, segName, paths);
				segIdx++;
			}      
		}                   	  
	}
	
	
	
	static void doIt(Map visitSegments = null, String subjectID = null, collectionEquipmentId = null, db = null) {
		def dbType= 'postgres';
		if (db == null) db = getDb();
		def hsDAO = new HumansubjectDAO(db: db);
		def visitDAO = new ExpcomponentDAO(db : db);
		def studyDAO = new ExpstudyDAO(db: db);
		def segmentDAO = new ExpsegmentDAO(db: db);
		def rawDataDAO = new RawdataDAO(db: db);
		def expDAO = new ExperimentDAO(db: db);
		
		def sortBy = 'subjectid';
		def subjects = null; 
		if (dbType == 'postgres') {
			subjects = hsDAO.find('', sortBy);
		} else {	  
			subjects = hsDAO.find('where siteid is not null', sortBy);
		}
		subjects.each { s ->
			// println "${s.subjectid} ${s.owner} ${s.siteid}"
			hsDAO.dumpVO(s)
		}
		def experiments = expDAO.find("where name ='download_test'", null);
		def expID = null;
		experiments.each { exp -> expDAO.dumpVO(exp); 
			expID = exp.uniqueid; }
		
		def subjectIds = subjects.collect { it.subjectid }
		def visitWherePart = "where subjectid in ('" + subjectIds.join(',') + "') and visittype='fMRI scan'";
		println visitWherePart
		def visits = visitDAO.find(visitWherePart, sortBy);
		visits.each { visit ->
			visitDAO.dumpVO(visit)
		}
		def visitIds = visits.collect { it.uniqueid }
		def studyWherePart = "where subjectid in ('" + subjectIds.join(',') + "') and name='FMRI'";
		println studyWherePart
		def studies = studyDAO.find(studyWherePart, sortBy);
		studies.each { study ->
			studyDAO.dumpVO(study)
		}
		def studyIds = studies.collect { it.uniqueid }
		def segmentWherePart = " where subjectid in ('" + subjectIds.join(',') + "')";
		println segmentWherePart;
		def segments = segmentDAO.find(segmentWherePart, sortBy);
		segments = segments.grep { seg -> seg.componentid > 1 }
		segments.each { segment ->
			segmentDAO.dumpVO(segment)
		}	 
		
		if ( visitSegments) {
			
			// segmentDAO.delete([subjectid : "'${subjectID}'", nc_experiment_uniqueid : expID, componentid:2]);
			
			visitSegments.each { visitid,segs -> 
				println "segs:$segs"
				def segmentid = 2;
				
				segs.each { segName, paths ->
					/*
					 segmentDAO.create([segmentid: segmentid, componentid :visitid, 
					 subjectid : "'${subjectID}'", nc_experiment_uniqueid : expID,
					 name : "'$segName'", protocolversion : 1, 
					 protocolid : "'Subjects screening protocol'",
					 studyid: 1]);
					 */
					paths.each { relPath ->
						rawDataDAO.create([segmentid: segmentid, componentid :visitid,
								subjectid : "'${subjectID}'", nc_experiment_uniqueid : expID,
								protocolversion : 1, 
								protocolid : "'Subjects screening protocol'",
								datauri : "'$relPath'",
								nc_colequipment_uniqueid : collectionEquipmentId
								]);
					}
					
					++segmentid;
					
				}		 
			}
			
		}	    	
	}
	
	static void testSequenceQuery(Map visitSegments = null, String subjectID = null) {
		def db = getDb()
		def uid = -1;
		db.eachRow('select uid_seq.nextval from dual') { row -> uid = row[0] }
		println "uid=$uid"
	}
	
	static void prepareCollectionEquipments(db = null) {
		if (db == null) db = getDb()
		def CollectionequipmentDAO ceDAO = new CollectionequipmentDAO(db :db);
		println "adding collection equipment."
		ceDAO.create([make: "'GE'" , model: "'bay4oc'"])
	}
	
	static void prepareWCTSCollEquipments(db = null) {
		if (db == null) 
		   db = getDb()
		def CollectionequipmentDAO ceDAO = new CollectionequipmentDAO(db :db);
		def ceList = ceDAO.find();
		def ceMap = ['Signa Excite East' : false, 'Signa Excite West' : false, 'CIGAL' : false];
		ceList.each { ce ->  
			if ( ceMap.containsKey(ce.model) )
				ceMap[ce.model] = true;
		}
		
		//println "adding collection equipment."
		if ( ceMap['Signa Excite East'] == false)
			ceDAO.create([make: "'GE Medical Systems'" , model: "'Signa Excite East'"]);
		if ( ceMap['Signa Excite West'] == false)
			ceDAO.create([make: "'GE Medical Systems'" , model: "'Signa Excite West'"]);
		if ( ceMap['CIGAL'] == false)          
			ceDAO.create([make: "'James Voyvodic'" , model: "'CIGAL'"]);
	}
	
	
	static Map extractVisitSegments(String commonPath, List subjectPaths) {
		Map visitSegments = [:]; 
		subjectPaths.each { path ->
			if (path.startsWith(commonPath)) {
				def fullPath = path 
				path = path.substring(commonPath.length())
				path = path.replaceFirst('^\\/+','')
				def parts = path.split('/')
				// first part will be the scan Visit 
				def matcher = null
				if ( (matcher = parts[0] =~ /_[0]*([1-9]\d*)$/) ) { 
					def visitId = matcher.group(1); 
					def segments = visitSegments[visitId]; 
					if (!segments) { 
						segments = [:] 
						visitSegments[visitId] = segments
					}
					if ( parts.size() >=3) {
						if (! parts[2].startsWith("QA")) { 
							def segPaths = segments[parts[2]]; 
							if (segPaths == null) { 
								segPaths = []; 
								segments[parts[2]] = segPaths; 
							}
							// segPaths << parts.join('/'); // for relative path
							segPaths << fullPath;
						}
					}
				}
			}	      		  
		}
		return visitSegments;
	}
	
	static void prepPathFile() {
		File out = new File("/home/bozyurt/work/phase2_paths.txt")
		def list = getAllPaths('/data/fBIRN/PhaseII_Data/000602155404')  
		list.each { println it }
		out.write( list.join("\n"))	     
	}
	
	/**
	 * given a list of paths finds the longest paths which are not included in other paths
	 */
	def static getMaximalPaths(List paths, Set suffixSet) {
		def maximalPaths = [];
		def len = paths.size();
		for(i in 0..<len) {
			def included = false;
			def lastPart = paths[i].substring( paths[i].lastIndexOf('/') + 1);
			if ( !suffixSet.contains(lastPart) ) {
				for(j in 0..<len) {
					if (j != i) {
						if (paths[i].length() < paths[j].length()) {               
							if ( paths[j].startsWith(paths[i])) {
								included = true;
								break;
							}
						}
					}
				}//j
			}  
			if (!included) {
				maximalPaths << paths[i];
			}
		}//i
		return maximalPaths;
	}
	
	def static getFromSRB(String srbRootPath, verbose=true) {
		def proc = "Sls -r  $srbRootPath".execute();
		def paths = [];
		proc.in.eachLine { line ->
			line = line.trim();
			if (line.startsWith('C-')) {
				line = line.substring(2);
				paths << line;
				if (verbose) println line;
			}
		}
		return paths;
	}
	
	def static getFromLocal(String subjectDirPath, verbose=true) {
		def paths = []
		new File(subjectDirPath).eachFileRecurse { f -> paths << f.getAbsolutePath() 
			if (verbose) { println f }
		}
		return paths
	}
	
	
	static void importTestSubjectImageData() {
		def commonPath = "/home/Projects/FBIRNSubject2007TW__0051/testData/000880305228/"; 
		commonPath = DBUtils.getAnswerOrDefault(commonPath, "Please enter the SRB subject path");
		def parts = commonPath.split(/\//);
		def subjectID = parts[-1];
		def pathsFile = "/home/bozyurt/work/${subjectID}_paths.txt";
		File out = new File(pathsFile);
		
		def paths = null;
		if ( !out.exists()) {
			def p = commonPath.replaceFirst('\\/$','');
			paths =  getFromSRB(p);
			new File(pathsFile).write( paths.join("\n") );          
		} else {
			//File out = new File("/home/bozyurt/work/000880305228_paths.txt")       
			paths = out.readLines();
		}
		def modelName = DBUtils.getAnswerOrDefault('Signa Excite West',"Please enter the scanner model");
		def expName = DBUtils.getAnswerOrDefault('download_test',"Please enter the experiment name");
		
		Set suffixSet = new HashSet(7);
		suffixSet.add("NIFTI4D");
		suffixSet.add("AFNI");       
		suffixSet.add("DICOM");
		paths = getMaximalPaths(paths, suffixSet);
		
		paths = paths.grep { path -> !(path =~ /SRB/) && !(path =~ /Analysis/) && !(path =~ /ANALYZE_7\.5/) }
		
		
		def visitSegments = extractVisitSegments(commonPath, paths)
		visitSegments.each { println it }
		prepareWCTSCollEquipments();
		def db = getDb();
		db.connection.autoCommit = false;
		try {
			SRBImagePathImporter.dbu = DBUtils.getDatabaseUser(db);
			
			prepareWCTSDataObjectTypes(db);
			def paramMap = [ expName : "$expName" , visitSegments : visitSegments, 
					subjectID: "$subjectID", model : "$modelName", uriScheme : "none"]
			
			importSubjectImageData(paramMap, db);
			prepareDataObjects4SubjectInSRB(paramMap.subjectID, db);
			db.commit()
		} catch(SQLException e) {
			e.printStackTrace();
			db.rollback();
		} finally {
			db.connection.close();
		}
	}
	
	static void addDataObjects() {
		def db = getDb();
		db.connection.autoCommit = false;
		try {
			SRBImagePathImporter.dbu = DBUtils.getDatabaseUser(db);
			
			prepareDataObjects4SubjectInSRB("000880305228",db);
			db.commit()
		} catch(SQLException e) {
			e.printStackTrace();
			db.rollback();
		} finally {
			db.connection.close();
		}
	}
	
	static void test1() {
	    
		def subjectDirPath = '/home/bozyurt3/Projects/FBIRNSubject2007TW__0051/Data/000955483063'
		def parts = subjectDirPath.split(/\//);
		def subjectID = parts[-1];
		def paths = getFromLocal(subjectDirPath,false);
		Set suffixSet = new HashSet(7);
		suffixSet.add("NIFTI4D");
		suffixSet.add("AFNI");       
		suffixSet.add("DICOM");
		paths = getMaximalPaths(paths, suffixSet);
		
		paths = paths.grep { path -> !(path =~ /SRB/) && !(path =~ /Analysis/) && !(path =~ /ANALYZE_7\.5/) }
		// paths.each { println it }
		
		println 'Visit Segments\n-----------------\n' 
		def visitSegments = extractVisitSegments(subjectDirPath, paths)
		visitSegments.each { println it }
		
		prepareWCTSCollEquipments();
		def db = getDb();
		db.connection.autoCommit = false;
		try {
			SRBImagePathImporter.dbu = DBUtils.getDatabaseUser(db);
			
			prepareWCTSDataObjectTypes(db);
			
			println "subjectDirPath: $subjectDirPath"
			
			def modelName = DBUtils.getAnswerOrDefault('Signa Excite West',"Please enter the scanner model");
			def expName = DBUtils.getAnswerOrDefault('download_test',"Please enter the experiment name");
			println "expName: $expName modelName: $modelName"
		
			def paramMap = [ expName : "$expName" , 
			                 visitSegments : visitSegments, 
					subjectID: "$subjectID", 
					model : "$modelName", 
					uriScheme : "none"]
			
			importSubjectImageData(paramMap, db);
			prepareDataObjects4SubjectInSRB(paramMap.subjectID,db, true);		
			db.commit()
		} catch(SQLException e) {
			e.printStackTrace();
			db.rollback();
		} finally {
			db.connection.close();
		}
	}
	
	static void test0() {
		//prepPathFile();
		File out = new File("/home/bozyurt/work/phase2_paths.txt")
		//doIt();
		
		def paths = out.readLines();
		paths = paths.grep { path -> !(path =~ /SRB/) && !(path =~ /Analysis/) && !(path =~ /ANALYZE_7\.5/) }
		// paths.each {	 println it     }
		
		def visitSegments = extractVisitSegments("/data/fBIRN/PhaseII_Data/000602155404/", paths)
		visitSegments.each { 	 println it   }
		
		//prepareCollectionEquipments();
		//doIt(visitSegments, '000602155404', 356)
		
		// prepareDataObjectTypes();
		// prepareDataObjects4Subject("/data/fBIRN/PhaseII_Data", "000602155404");
	}
	
	static void main(args) {
		test1();
	}
	
}
