package clinical.server;

import java.math.BigDecimal;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.Types;
import java.util.List;

import clinical.server.dao.oracle.AssessmentDAO;
import clinical.server.dao.oracle.AssessmentscoreDAO;
import clinical.server.vo.Assessment;
import clinical.server.vo.Assessmentscore;
import clinical.upload.ILog;
import clinical.web.DBUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: AssessmentManager.java 91 2009-08-17 23:38:26Z bozyurt $
 */
public class AssessmentManager {

	public AssessmentManager() {
	}

	public BigDecimal addAssessment(Connection con, String asName,
			String asDescp, String owner, String modUser, BigDecimal secLabel)
			throws Exception {
		CallableStatement cst = null;
		/**
		 * @todo 2nd tier validation or rely on servlet container side
		 *       validation ?
		 */
		try {
			cst = con.prepareCall("{? = call add_assessment(?,?,?,?,?)}");
			cst.registerOutParameter(1, Types.NUMERIC);
			cst.setString(2, asName);
			cst.setString(3, asDescp);
			cst.setString(4, owner);
			cst.setString(5, modUser);
			cst.setBigDecimal(6, secLabel);
			cst.execute();
			BigDecimal asID = cst.getBigDecimal(1);

			return asID;
		} finally {
			if (cst != null)
				try {
					cst.close();
				} catch (Exception x) { /* ignore */
				}
		}
	}

	public void removeAssessment(Connection con, String asName)
			throws Exception {
		Assessment criteria = new Assessment();
		AssessmentDAO dao = new AssessmentDAO();
		criteria.setName(asName);
		dao.delete(con, criteria);
	}

	public void removeAssessmentFull(Connection con, String asName,
			String owner, String modUser) throws Exception {
		Assessment criteria = new Assessment();
		criteria.setName(asName);
		AssessmentDAO dao = new AssessmentDAO();
		List<Assessment> asList = dao.find(con, criteria);
		if (asList.isEmpty()) {
			System.err.println("Assessment " + asName + " not found!");
			return;
		}
		Assessment theAs = asList.get(0);

		Assessmentscore cr = new Assessmentscore();
		cr.setAssessmentid(theAs.getAssessmentid());
		AssessmentscoreDAO asDAO = new AssessmentscoreDAO();
		List<Assessmentscore> scores = asDAO.find(con, cr);
		for (Assessmentscore score : scores) {
			removeAssessmentScore(con, asName, owner, modUser, score
					.getScorename());

		}
		dao.delete(con, criteria);
	}

	public void removeAssessmentScore(Connection con, String asName,
			String owner, String modUser, String scoreName) throws Exception {

		CallableStatement cst = null;
		/**
		 * @todo 2nd tier validation or rely on servlet container side
		 *       validation ?
		 */
		try {
			cst = con.prepareCall("{call delete_assessment_score(?,?,?,?)}");
			cst.setString(1, asName);
			cst.setString(2, scoreName);
			cst.setString(3, owner);
			cst.setString(4, modUser);
			cst.execute();

		} finally {
			DBUtils.close(cst);			
		}
	}

	/* for old BIRN db schema (08/14/03) */
	public void addAssessmentScore(Connection con, String asName, String owner,
			String modUser, String scoreName, String scoreType,
			String secClass, String defaultValue, boolean nullable,
			String ontology, String concept) throws Exception {

		CallableStatement cst = null;
		/**
		 * @todo 2nd tier validation or rely on servlet container side
		 *       validation ?
		 */
		try {
			cst = con
					.prepareCall("{call add_assessment_score(?,?,?,?,?,?,?,?,?,?)}");
			cst.setString(1, asName);
			cst.setString(2, owner);
			cst.setString(3, modUser);
			cst.setString(4, scoreName);
			cst.setString(5, scoreType);
			cst.setString(6, secClass);
			if (defaultValue == null) {
				cst.setNull(7, Types.VARCHAR);
			} else {
				cst.setString(7, defaultValue);
			}
			cst.setBigDecimal(8, (nullable) ? new java.math.BigDecimal("1")
					: new java.math.BigDecimal("0"));
			cst.setString(9, ontology);
			cst.setString(10, concept);

			cst.execute();

		} finally {
			DBUtils.close(cst);				
		}
	}

	/* for new BIRN db schema (08/14/03) */
	public void addAssessmentScore2(Connection con, String asName,
			String owner, String modUser, BigDecimal secLabel,
			String scoreName, String scoreType, BigDecimal scoreLevel,
			BigDecimal parentAsID,
			String parentScore, // score is same as asName
			String secClass, String defaultValue, boolean nullable,
			String ontology, String concept) throws Exception {

		CallableStatement cst = null;
		/**
		 * @todo 2nd tier validation or rely on servlet container side
		 *       validation ?
		 */
		try {
			cst = con
					.prepareCall("{call add_assessment_score(?,?,?,?,?,?,?,?,?,?,?,?,?,?)}");
			cst.setString(1, asName);
			cst.setString(2, owner);
			cst.setString(3, modUser);
			cst.setBigDecimal(4, secLabel);
			cst.setString(5, scoreName);
			cst.setString(6, scoreType);
			cst.setBigDecimal(7, scoreLevel);
			cst.setBigDecimal(8, parentAsID);
			cst.setString(9, parentScore);
			cst.setString(10, secClass);
			if (defaultValue == null) {
				cst.setNull(11, Types.VARCHAR);
			} else {
				cst.setString(11, defaultValue);
			}
			cst.setBigDecimal(12, (nullable) ? new java.math.BigDecimal("1")
					: new java.math.BigDecimal("0"));
			cst.setString(13, ontology);
			cst.setString(14, concept);

			cst.execute();
		} finally {
			DBUtils.close(cst);	
			
		}
	}

	public void addScoreValues(Connection con, ScoreInfo si, String owner,
			String modUser, BigDecimal secLabel, boolean testMode,
			ILog logCollector) throws Exception {
		con.setAutoCommit(false);
		try {
			for (ScoreInfo.AssessmentValue asValue : si.getValues()) {
				System.out.println("adding score value " + asValue);
				if (logCollector != null)
					logCollector.setMessage("adding score value " + asValue);
				if (!testMode) {
					addScoreValue2(con, si, asValue, owner, modUser, secLabel);
				}
			}
			if (!testMode) {
				con.commit();
			}
		} catch (Exception x) {
			con.rollback();
			throw x;
		}
	}

	/* for old BIRN db schema (08/14/03) */
	public BigDecimal addScoreValue(Connection con, ScoreInfo si,
			ScoreInfo.AssessmentValue asValue, String owner, String modUser)
			throws SQLException {
		BigDecimal asScoreID = null;
		CallableStatement cst = null;
		/**
		 * @todo 2nd tier validation or rely on servlet container side
		 *       validation ?
		 */
		try {
			cst = con
					.prepareCall("{? = call store_assessment_value(?,?,?,?,?,?,?,?,?,?,?,?,?,?)}");
			cst.registerOutParameter(1, Types.NUMERIC);
			cst.setBigDecimal(2, si.getAssessment().getAssessmentid());
			cst.setString(3, owner);
			cst.setString(4, modUser);
			cst.setBigDecimal(5, new BigDecimal(String.valueOf(asValue
					.getVisitID())));
			cst.setBigDecimal(6, new BigDecimal(String.valueOf(asValue
					.getExperimentID())));
			cst.setString(7, asValue.getSubjectID());
			cst.setString(8, si.getAsScore().getScorename());
			cst.setString(9, si.getAsScore().getScoretype());
			cst.setDate(10, new java.sql.Date(asValue.getDate().getTime()));
			switch (si.getType()) {
			case ScoreInfo.INT:
				cst.setBigDecimal(11, new BigDecimal(String
						.valueOf(((Integer) asValue.getValue()).intValue())));
				cst.setNull(12, Types.VARCHAR);
				if (asValue.getNormValue() == null)
					cst.setNull(13, Types.NUMERIC);
				else
					cst.setBigDecimal(13, new BigDecimal(String
							.valueOf(((Integer) asValue.getNormValue())
									.intValue())));
				cst.setNull(14, Types.VARCHAR);
				break;
			case ScoreInfo.FLOAT:
				cst.setBigDecimal(11, new BigDecimal(String
						.valueOf(((Float) asValue.getValue()).floatValue())));
				cst.setNull(12, Types.VARCHAR);
				if (asValue.getNormValue() == null)
					cst.setNull(13, Types.NUMERIC);
				else
					cst.setBigDecimal(13, new BigDecimal(String
							.valueOf(((Float) asValue.getNormValue())
									.floatValue())));
				cst.setNull(14, Types.VARCHAR);
				break;
			case ScoreInfo.BOOL:
				cst.setBigDecimal(11, ((Boolean) asValue.getValue())
						.booleanValue() ? new BigDecimal("1") : new BigDecimal(
						"0"));
				cst.setNull(12, Types.VARCHAR);
				// normValue is mandatory for boolean
				cst.setBigDecimal(13, ((Boolean) asValue.getNormValue())
						.booleanValue() ? new BigDecimal("1") : new BigDecimal(
						"0"));
				cst.setNull(14, Types.VARCHAR);
				break;
			case ScoreInfo.STRING:
				cst.setNull(11, Types.NUMERIC);
				cst.setString(12, (String) asValue.getValue());
				cst.setNull(13, Types.NUMERIC);
				if (asValue.getNormValue() == null)
					cst.setNull(14, Types.VARCHAR);
				else
					cst.setString(14, (String) asValue.getNormValue());
				break;
			default:
				throw new RuntimeException("Not a supported type!");

			}
			if (asValue.getComments() == null)
				cst.setNull(15, Types.VARCHAR);
			else
				cst.setString(15, asValue.getComments());

			cst.execute();
			asScoreID = cst.getBigDecimal(1);
		} finally {
			DBUtils.close(cst);			
		}
		return asScoreID;

	}

	/* for new BIRN db schema (08/14/03) */
	
	public BigDecimal addScoreValue2(Connection con, ScoreInfo si,
			ScoreInfo.AssessmentValue asValue, String owner, String modUser,
			BigDecimal secLabel) throws SQLException {
		BigDecimal asScoreID = null;
		CallableStatement cst = null;
		/**
		 * @todo 2nd tier validation or rely on servlet container side
		 *       validation ?
		 */
		try {

			cst = con
					.prepareCall("{? = call store_assessment_value(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)}");
			cst.registerOutParameter(1, Types.NUMERIC);
			cst.setBigDecimal(2, si.getAssessment().getAssessmentid());
			cst.setString(3, owner);
			cst.setString(4, modUser);
			cst.setBigDecimal(5, secLabel);
			cst.setBigDecimal(6, new BigDecimal(String.valueOf(asValue
					.getVisitID())));
			cst.setBigDecimal(7, new BigDecimal(String.valueOf(asValue
					.getExperimentID())));

			cst.setBigDecimal(8, new BigDecimal(String.valueOf(asValue
					.getSegmentID())));

			cst.setString(9, asValue.getSubjectID());
			cst.setString(10, si.getAsScore().getScorename());
			cst.setString(11, si.getAsScore().getScoretype());

			// score order (for multi-valued scores)
			/** @todo support multi-valued scores */
			cst.setBigDecimal(12, new BigDecimal("0"));
			cst.setDate(13, new java.sql.Date(asValue.getDate().getTime()));
			switch (si.getType()) {
			case ScoreInfo.INT:
				cst.setBigDecimal(14, new BigDecimal(String
						.valueOf(((Integer) asValue.getValue()).intValue())));

				cst.setNull(15, Types.VARCHAR);
				if (asValue.getNormValue() == null) {
					cst.setNull(16, Types.NUMERIC);
				} else {
					cst.setBigDecimal(16, new BigDecimal(String
							.valueOf(((Integer) asValue.getNormValue())
									.intValue())));
				}
				cst.setNull(17, Types.VARCHAR);
				break;
			case ScoreInfo.FLOAT:
				cst.setBigDecimal(14, new BigDecimal(String
						.valueOf(((Float) asValue.getValue()).floatValue())));
				cst.setNull(15, Types.VARCHAR);
				if (asValue.getNormValue() == null) {
					cst.setNull(16, Types.NUMERIC);
				} else {
					cst.setBigDecimal(16, new BigDecimal(String
							.valueOf(((Float) asValue.getNormValue())
									.floatValue())));
				}
				cst.setNull(17, Types.VARCHAR);
				break;
			case ScoreInfo.BOOL:
				cst.setBigDecimal(14, ((Boolean) asValue.getValue())
						.booleanValue() ? new BigDecimal("1") : new BigDecimal(
						"0"));
				cst.setNull(15, Types.VARCHAR);
				// normValue is mandatory for boolean
				cst.setBigDecimal(16, ((Boolean) asValue.getNormValue())
						.booleanValue() ? new BigDecimal("1") : new BigDecimal(
						"0"));
				cst.setNull(17, Types.VARCHAR);
				break;
			case ScoreInfo.STRING:
				cst.setNull(14, Types.NUMERIC);
				cst.setString(15, (String) asValue.getValue());
				cst.setNull(16, Types.NUMERIC);
				if (asValue.getNormValue() == null) {
					cst.setNull(17, Types.VARCHAR);
				} else {
					cst.setString(17, (String) asValue.getNormValue());
				}
				break;
			default:
				throw new RuntimeException("Not a supported type!");

			}
			if (asValue.getComments() == null) {
				cst.setNull(18, Types.VARCHAR);
			} else {
				cst.setString(18, asValue.getComments());
			}

			cst.execute();
			asScoreID = cst.getBigDecimal(1);
		} finally {
			if (cst != null)
				try {
					cst.close();
				} catch (Exception x) { /* ignore */
				}
		}
		return asScoreID;

	}

	public void addOntologySource(Connection con, String ontologySource,
			String sourceURI, String desc, String owner, String modUser)
			throws SQLException {
		CallableStatement cst = null;
		/**
		 * @todo 2nd tier validation or rely on servlet container side
		 *       validation ?
		 */
		try {
			cst = con.prepareCall("{call add_ontology_source(?,?,?,?,?)}");
			cst.setString(1, ontologySource);
			cst.setString(2, owner);
			cst.setString(3, modUser);
			if (sourceURI == null)
				cst.setNull(4, Types.VARCHAR);
			else
				cst.setString(4, sourceURI);
			if (desc != null)
				cst.setNull(5, Types.VARCHAR);
			else
				cst.setString(5, desc);

			cst.execute();

		} finally {
			if (cst != null)
				try {
					cst.close();
				} catch (Exception x) { /* ignore */
				}
		}
	}

	public void removeScoreValues(Connection con, ScoreInfo si, String owner,
			String modUser) throws Exception {
		con.setAutoCommit(false);
		try {
			for (ScoreInfo.AssessmentValue asValue : si.getValues()) {
				System.out.println("removing " + asValue);
				removeScoreValue2(con, si, asValue);
			}
			con.commit();
		} catch (Exception x) {
			con.rollback();
			throw x;
		}
	}

	public void removeScoreValue(Connection con, ScoreInfo si,
			ScoreInfo.AssessmentValue asValue) throws SQLException {
		CallableStatement cst = null;
		/**
		 * @todo 2nd tier validation or rely on servlet container side
		 *       validation ?
		 */
		try {
			cst = con
					.prepareCall("{call delete_assessment_value(?,?,?,?,?,?)}");
			cst.setBigDecimal(1, si.getAssessment().getAssessmentid());
			cst.setBigDecimal(2, new BigDecimal(String.valueOf(asValue
					.getVisitID())));
			cst.setBigDecimal(3, new BigDecimal(String.valueOf(asValue
					.getExperimentID())));
			cst.setString(4, asValue.getSubjectID());
			cst.setString(5, si.getAsScore().getScoretype());
			cst.setString(6, si.getAsScore().getScorename());

			cst.execute();
		} finally {
			DBUtils.close(cst);			
		}
	}

	/* for new BIRN db schema (08/21/03) */
	/*
	 * assessment_id_in NUMBER, component_id_in NUMBER, experiment_id_in NUMBER, --
	 * IBO segmentID is added (7/15) segment_id_in NUMBER, subject_id_in
	 * VARCHAR2, as_score_type_in VARCHAR2, as_score_name_in VARCHAR2, -- IBO
	 * (7/15) score_order_in NUMBER
	 */
	public void removeScoreValue2(Connection con, ScoreInfo si,
			ScoreInfo.AssessmentValue asValue) throws SQLException {
		CallableStatement cst = null;
		/**
		 * @todo 2nd tier validation or rely on servlet container side
		 *       validation ?
		 */
		try {
			cst = con
					.prepareCall("{call delete_assessment_value(?,?,?,?,?,?,?,?)}");
			cst.setBigDecimal(1, si.getAssessment().getAssessmentid());
			cst.setBigDecimal(2, new BigDecimal(String.valueOf(asValue
					.getVisitID())));
			cst.setBigDecimal(3, new BigDecimal(String.valueOf(asValue
					.getExperimentID())));
			cst.setBigDecimal(4, new BigDecimal(String.valueOf(asValue
					.getSegmentID())));
			cst.setString(5, asValue.getSubjectID());
			cst.setString(6, si.getAsScore().getScoretype());
			cst.setString(7, si.getAsScore().getScorename());
			/** @todo no multivalue support yet */
			cst.setBigDecimal(8, new BigDecimal("0"));
			cst.execute();
		} finally {
			DBUtils.close(cst);			
		}
	}

	public void addOntologyConcept(Connection con, String ontologySource,
			BigDecimal conceptID, String concept, String ontologyPath,
			String owner, String modUser) throws SQLException {
		CallableStatement cst = null;
		/**
		 * @todo 2nd tier validation or rely on servlet container side
		 *       validation ?
		 */
		try {
			cst = con.prepareCall("{call add_ontology_concept(?,?,?,?,?,?)}");
			cst.setString(1, ontologySource);
			cst.setBigDecimal(2, conceptID);
			cst.setString(3, concept);
			cst.setString(4, owner);
			cst.setString(5, modUser);
			if (ontologyPath == null)
				cst.setNull(6, Types.VARCHAR);
			else
				cst.setString(6, ontologyPath);
			cst.execute();

		} finally {
			DBUtils.close(cst);			
		}

	}

	public void testScores(Connection con, String owner, String modUser)
			throws Exception {
		/*
		 * addOntologySource(con,"UMLS","no URI","just a placeholder", owner,
		 * modUser); addOntologyConcept(con,"UMLS", new
		 * BigDecimal("9999"),"Assessment", "no path", owner,modUser);
		 */
		System.out.println("Adding assessment scores");
		removeAssessmentScore(con, "WCS", owner, modUser, "Errors");
		addAssessmentScore(con, "WCS", owner, modUser, "Errors", "integer",
				"DI", null, true, "UMLS", "Assessment");
		System.out.println("assessment score test finished successfully");
	}

	public void testScoreValues(Connection con, String owner, String modUser,
			BigDecimal secLabel, boolean testMode) throws Exception {
		Assessment as = new Assessment();
		as.setAssessmentid(new BigDecimal("43"));
		as.setName("WCS");
		Assessmentscore score = new Assessmentscore();
		score.setScorename("Errors");
		score.setScoretype("integer");
		ScoreInfo si = new ScoreInfo(as, score);
		/** @todo check segmentID */
		si.addValue(new Integer(28), null, 42, 1/* segment id */, 16,
				"UCSD0001", Utils.parseUSDate("06/25/1996"), null);
		// first remove the score values
		removeScoreValues(con, si, owner, modUser);
		System.out.println("Now adding test score values");
		addScoreValues(con, si, owner, modUser, secLabel, testMode, null);
		System.out.println("Score value test has finished");
	}

	public static void usage() {
		System.err
				.println("java clinical.server.AssessmentManager <dbURL> <usr> <pwd>");
		System.err
				.println("\twhere dbURL has the following syntax jdbc:oracle:thin:@<host>:<port>:<sid>");
		System.exit(1);
	}

	// test driver
	public static void main(String[] args) {
		Connection con = null;
		AssessmentManager am = new AssessmentManager();
		try {
			if (args.length != 3)
				usage();

			String dbURL = args[0];
			String user = args[1];
			String pwd = args[2];

			Class.forName("oracle.jdbc.driver.OracleDriver");
			con = DriverManager.getConnection(dbURL, user, pwd);

			// am.removeAssessment(con,"MMSE");

			// BigDecimal asID = am.addAssessment(con,"MMSE","Mini Mental State
			// Examination","ucsd_fmri","ucsd_fmri");
			// System.out.println("asID="+ asID);

			// am.testScores(con,"ucsd_fmri","ucsd_fmri");

			// public:loc,brn
			am.testScoreValues(con, "ucsd_fmri", "ucsd_fmri", new BigDecimal(
					"11000"), true);

		} catch (Exception x) {
			x.printStackTrace();
		} finally {
			if (con != null)
				try {
					con.close();
				} catch (Exception x) { /* ignore */
				}
		}
	}
}