package clinical.server.upload;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.DriverManager;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import clinical.server.AssessmentManager;
import clinical.server.ExperimentManager;
import clinical.server.ScoreInfo;
import clinical.server.dao.oracle.AssessmentDAO;
import clinical.server.dao.oracle.AssessmentscoreDAO;
import clinical.server.dao.oracle.ExpcomponentDAO;
import clinical.server.dao.oracle.ProtocolDAO;
import clinical.server.vo.Assessment;
import clinical.server.vo.Assessmentscore;
import clinical.server.vo.Expcomponent;
import clinical.server.vo.Protocol;
import clinical.utils.CSVParser;
import clinical.utils.GenUtils;
import clinical.web.Constants;

/**
 * 
 * @version $Id: UploadManager.java 91 2009-08-17 23:38:26Z bozyurt $
 * @author I. Burak Ozyurt
 */

public class UploadManager {
	protected List<List<String>> dataRows;
	protected AssessmentManager asMan;
	protected ExperimentManager expMan;
	protected boolean testMode = false;

	protected Map<String, ColumnInfo> columnMap = new HashMap<String, ColumnInfo>(
			11);

	public UploadManager() {
		asMan = new AssessmentManager();
		expMan = new ExperimentManager();
	}

	public void loadData(String csvFile) throws Exception {
		CSVParser parser = new CSVParser();
		parser.extractData(csvFile);
		dataRows = parser.getRows();
		if (dataRows.isEmpty())
			return;
		columnMap.put("birnID", new ColumnInfo("birnID", 0));
		columnMap.put("diagnosis", new ColumnInfo("diagnosis", 1));
		columnMap.put("age", new ColumnInfo("age", 2, new ScoreConditioner(
				ScoreConditioner.INTEGER)));
		columnMap.put("gender", new ColumnInfo("gender", 3));
		columnMap.put("mmse", new ColumnInfo("mmse", 4, new ScoreConditioner(
				ScoreConditioner.INTEGER)));
		columnMap.put("mmse-date", new ColumnInfo("mmse-date", 5,
				new DateConditioner("dd-MMM-yyyy")));
		columnMap.put("scan-date", new ColumnInfo("scan-date", 6,
				new DateConditioner("dd-MMM-yyyy")));

	}

	static class AssessmentInfo {
		Assessment assessment;
		List<?> scores;

		public AssessmentInfo(Assessment assessment, List<?> scores) {
			this.assessment = assessment;
			this.scores = scores;
		}

		public Assessmentscore getScore(String scoreName) {
			for (Iterator<?> it = scores.iterator(); it.hasNext();) {
				Assessmentscore score = (Assessmentscore) it.next();
				if (score.getScorename().equals(scoreName))
					return score;
			}
			return null;
		}

	}

	// initial set
	// birnID, age, gender, diagnosis, MMSE, MMSEDate, ScanDate

	public void insertScoreValues(Connection con, String owner, String modUser,
			BigDecimal secLabel) throws Exception {

		// get all assessments
		List <?>asList = getAssessments(con, new Assessment());
		// prepare the ScoreInfo objects
		Map<String, AssessmentInfo> scoreMap = new HashMap<String,AssessmentInfo>(
				7);
		AssessmentscoreDAO scoreDAO = new AssessmentscoreDAO();
		for (Iterator<?> it = asList.iterator(); it.hasNext();) {
			Assessment as = (Assessment) it.next();
			if (as.getName().equals("MMSE")
					|| as.getName().equals("Demographics")
					|| as.getName().equals("Diagnosis")) {
				Assessmentscore criteria = new Assessmentscore();
				criteria.setAssessmentid(as.getAssessmentid());
				List<?> scoreList = scoreDAO.find(con, criteria);
				scoreMap.put(as.getName(), new AssessmentInfo(as, scoreList));
			}
		}

		/** @todo specific generalize */
		// create score infos
		AssessmentInfo ai = scoreMap.get("MMSE");
		ScoreInfo mmseSi = new ScoreInfo(ai.assessment, ai
				.getScore("MMSE Score"));

		ai = scoreMap.get("Diagnosis");
		ScoreInfo diagSi = new ScoreInfo(ai.assessment, ai
				.getScore("Diagnosis"));
		ai = scoreMap.get("Demographics");
		ScoreInfo ageSi = new ScoreInfo(ai.assessment, ai.getScore("Age"));

		ai = scoreMap.get("Demographics");
		ScoreInfo genderSi = new ScoreInfo(ai.assessment, ai.getScore("Gender"));

		// get visit info for each subject
		Expcomponent criteria = new Expcomponent();
		ExpcomponentDAO visitDAO = new ExpcomponentDAO();
		List<?> visits = visitDAO.find(con, criteria);
		Map<String, List<? super Expcomponent>> visitMap = 
			new HashMap<String, List<? super Expcomponent>>();
		for (Iterator<?> it = visits.iterator(); it.hasNext();) {
			Expcomponent visit = (Expcomponent) it.next();
			List<? super Expcomponent> visitsPerSubject = null;
			if ((visitsPerSubject = visitMap.get(visit.getSubjectid())) == null) {
				visitsPerSubject = new LinkedList<Expcomponent>();
				visitMap.put(visit.getSubjectid(), visitsPerSubject);
			}
			visitsPerSubject.add(visit);
		}

		ColumnInfo birnidCol = columnMap.get("birnID");
		ColumnInfo ageCol = columnMap.get("age");
		ColumnInfo scanCol = columnMap.get("scan-date");
		ColumnInfo diagCol = columnMap.get("diagnosis");
		ColumnInfo genderCol = columnMap.get("gender");
		ColumnInfo mmseCol = columnMap.get("mmse");
		ColumnInfo mmseDateCol = columnMap.get("mmse-date");

		// now it is time to prepare score values for upload
		for (Iterator<List<String>> it = dataRows.iterator(); it.hasNext();) {
			List<String> row = it.next();
			String[] colValues = new String[row.size()];
			colValues = row.toArray(colValues);

			String birnID = birnidCol.getColValue(colValues);
			List<? super Expcomponent> visitsPerSubject = visitMap.get(birnID);
			if (visitsPerSubject == null) {
				System.err.println("Skipping subject " + birnID
						+ " since no visits exists!");
				continue;
			}
			Expcomponent scanVisit = getVisit(Constants.VISIT_TYPE_SCAN,
					visitsPerSubject);
			Expcomponent clinVisit = getVisit(Constants.VISIT_TYPE_CLINICAL,
					visitsPerSubject);

			String gender = genderCol.getColValue(colValues).equals("1") ? "M"
					: "F";

			ageSi.addValue((Integer) ageCol.transform(colValues), null,
					scanVisit.getComponentid().intValue(), // visitID
					1, // segmentID
					scanVisit.getNcExperimentUniqueid().intValue(), // experimentID
					birnID, // subjectID
					(Date) scanCol.transform(colValues), // scan date
					"" // no comments
			);

			genderSi.addValue(gender, null, scanVisit.getComponentid()
					.intValue(), // visitID
					1, // segmentID (assuming 1 segment per visit)
					scanVisit.getNcExperimentUniqueid().intValue(), // experimentID
					birnID, (Date) scanCol.transform(colValues), // scan date
					"" // no comments
			);
			Integer mmse = null;
			if (mmseCol.getColValue(colValues).equalsIgnoreCase("n/a"))
				mmse = null;
			else {
				try {
					mmse = (Integer) mmseCol.transform(colValues);
				} catch (ConditionException ce) {
					mmse = null;
				}
			}
			if (mmse != null) {
				mmseSi.addValue(mmse, null,
						clinVisit.getComponentid().intValue(), // visitID
						1, // segmentID
						clinVisit.getNcExperimentUniqueid().intValue(), birnID,
						(Date) mmseDateCol.transform(colValues), // MMSE date
						"" // no comments
				);
			}
			diagSi.addValue(formatDiagnosis(diagCol.getColValue(colValues)),
					null,
					scanVisit.getComponentid().intValue(), // visitID
					1, // segmentID
					scanVisit.getNcExperimentUniqueid().intValue(), birnID,
					(Date) scanCol.transform(colValues), // scan date
					"" // no comments
			);

		}
		/*
		 * System.out.println("MMSE score values"); for (Iterator it =
		 * mmseSi.getValues().iterator(); it.hasNext(); ) {
		 * ScoreInfo.AssessmentValue item = (ScoreInfo.AssessmentValue)
		 * it.next(); System.out.println(item); }
		 */
		// con.setAutoCommit(true);
		try {
			/*
			 * asMan.removeScoreValues(con, genderSi, owner, modUser);
			 * asMan.addScoreValues(con, genderSi, owner, modUser);
			 * System.out.println("*** finished adding gender scores");
			 */

			System.out.println("started adding scores");
			asMan.addScoreValues(con, mmseSi, owner, modUser, secLabel,
					testMode, null);
			System.out.println("*** finished adding MMSE scores");

			asMan.addScoreValues(con, ageSi, owner, modUser, secLabel,
					testMode, null);
			System.out.println("*** finished adding age scores");

			asMan.addScoreValues(con, genderSi, owner, modUser, secLabel,
					testMode, null);
			System.out.println("*** finished adding gender scores");

			asMan.addScoreValues(con, diagSi, owner, modUser, secLabel,
					testMode, null);
			System.out.println("*** finished adding diagnosis scores");

			System.out.println("finished adding scores");

		} catch (Exception x) {
			throw x;
		} finally {
			con.setAutoCommit(true);
		}

	}

	protected String formatDiagnosis(String value) {
		value = value.toLowerCase();
		if (value.startsWith("alzheimer"))
			return "Alzheimer";
		if (value.startsWith("mem"))
			return "Mem Imp";
		if (value.startsWith("control"))
			return "Control";
		return null;
	}

	protected Expcomponent getVisit(String visitType, List<? super Expcomponent> visits) {
		// linear search O(n)
		for (Iterator<?> it = visits.iterator(); it.hasNext();) {
			Expcomponent visit = (Expcomponent) it.next();
			if (visit.getVisittype().equalsIgnoreCase(visitType)) {
				return visit;
			}
		}
		return null;
	}

	public List<?> getProtocols(Connection con) throws Exception {
		ProtocolDAO dao = new ProtocolDAO();
		List<?> protocols = dao.find(con, new Protocol());
		return protocols;
	}

	public Protocol findProtocol(String visitType, List<?> protocols) {
		/**
		 * @todo make this to use ge header info to figure out the appropriate
		 *       protocol
		 */
		for (Iterator<?> iter = protocols.iterator(); iter.hasNext();) {
			Protocol protocol = (Protocol) iter.next();
			if (visitType.equalsIgnoreCase(Constants.VISIT_TYPE_SCAN)) {
				if (protocol.getProtocolid().equalsIgnoreCase("jernigan_spgr"))
					return protocol;
			} else if (visitType
					.equalsIgnoreCase(Constants.VISIT_TYPE_CLINICAL)) {
				if (protocol.getProtocolid().equalsIgnoreCase(
						"ucsd_adrc_battery"))
					return protocol;
			}
		}
		return null;
	}

	public void prepareAssessments(Connection con, String owner,
			String modUser, BigDecimal secLabel) throws Exception {
		// only for age, gender and MMSE for now
		con.setAutoCommit(false);
		try {

			asMan.removeAssessment(con, "MMSE");
			// returns BigDecimal mmseID 
			asMan.addAssessment(con, "MMSE",
					"Mini Mental State Examination", owner, modUser, secLabel);
        	// BigDecimal demographichsID = 
        		asMan.addAssessment(con,
					"Demographics", "Patient Demographics", owner, modUser,
					secLabel);
			// BigDecimal diagID = 
				asMan.addAssessment(con, "Diagnosis",
					"Diagnosis Information", owner, modUser, secLabel);

			asMan.addAssessmentScore2(con, "MMSE", owner, modUser, secLabel,
					"MMSE Score", "integer", new BigDecimal("0"), null, null, // scoreLevel,
																				// parentAsID,
																				// parentScore
					"DI", null, true, "UMLS", "Assessment");

			asMan.addAssessmentScore2(con, "Demographics", owner, modUser,
					secLabel, "Gender", "varchar", new BigDecimal("0"), null,
					null, // scoreLevel, parentAsID, parentScore
					"DI", null, true, "UMLS", "Assessment");

			asMan.addAssessmentScore2(con, "Demographics", owner, modUser,
					secLabel, "Age", "integer", new BigDecimal("0"), null,
					null, // scoreLevel, parentAsID, parentScore
					"DI", null, true, "UMLS", "Assessment");
			asMan.addAssessmentScore2(con, "Diagnosis", owner, modUser,
					secLabel, "Diagnosis", "varchar", new BigDecimal("0"),
					null, null, // scoreLevel, parentAsID, parentScore
					"DI", null, true, "UMLS", "Assessment");

			con.commit();
		} catch (Exception x) {
			con.rollback();
			throw x;
		} finally {
			con.setAutoCommit(true);
		}

	}

	// specific to ADRC data
	public void prepareExperimentInfos(Connection con, String owner,
			String modUser, BigDecimal secLabel) throws Exception {
		try {
			con.setAutoCommit(false);

			ColumnInfo birnidCol = columnMap.get("birnID");
			// ColumnInfo ageCol = (ColumnInfo) columnMap.get("age");
			ColumnInfo scanCol = columnMap.get("scan-date");
			ColumnInfo diagCol = columnMap.get("diagnosis");
			// ColumnInfo genderCol = (ColumnInfo) columnMap.get("gender");
			ColumnInfo mmseCol = columnMap.get("mmse");
			ColumnInfo mmseDateCol = columnMap.get("mmse-date");

			List<?> protocols = getProtocols(con);

			for (Iterator<List<String>> it = dataRows.iterator(); it.hasNext();) {
				List<String> row = it.next();
				String[] colValues = new String[row.size()];
				colValues = (String[]) row.toArray(colValues);

				String researchGroupName = "Control";
				String diag = diagCol.getColValue(colValues).toLowerCase();
				if (diag.startsWith("alzheim")) {
					researchGroupName = "Alzheimer Subject";
				} else if (diag.startsWith("mem")) {
					researchGroupName = "Memory Impaired Subject";
				} else if (diag.startsWith("control")) {
					researchGroupName = "Control";
				} else
					throw new Exception("Not a known research group:" + diag);
				/** @todo remove hardcoded experiment parameters */
				/**
				 * @todo check image protocols (jerginan_spgr_ge_signa4_sag
				 *       jerginan_spgr_ge_signa4_cor jerginan_spgr_ge_signa8_sag
				 *       jerginan_spgr_ge_signa8_cor
				 */

				Protocol protocol = findProtocol("scan", protocols);
				String birnIDStr = birnidCol.getColValue(colValues);
				expMan.addExperimentInfo2(con, birnIDStr, // birn ID
						"Alzheimer Testbed", // experimentName
						researchGroupName, new BigDecimal("1"), // visit id
						(Date) scanCol.transform(colValues), // scan date
						"", // visit description
						"scan", // visit type
						new BigDecimal("1"), // segmentID
						"MRI scan", // segment description
						new BigDecimal("1"), // protocol version
						protocol.getProtocolid(), // protocolID
						owner, modUser, secLabel);
				if (!mmseCol.getColValue(colValues).equalsIgnoreCase("n/a")) {
					protocol = findProtocol("clinical", protocols);
					expMan.addExperimentInfo2(con,
							colValues[0], // birn ID
							"Alzheimer Testbed", researchGroupName,
							new BigDecimal("2"), // visit id
							(Date) mmseDateCol.transform(colValues), // visit
																		// date
							"", // visit description
							"clinical", // visit type
							new BigDecimal("1"), // segmentID
							"Clinical Assessment", // segment description
							new BigDecimal("1"), // protocol version
							protocol.getProtocolid(), // protocolID
							owner, modUser, secLabel);
				}
			}
			con.commit();
		} catch (Exception x) {
			con.rollback();
			throw x;
		} finally {
			con.setAutoCommit(true);
		}
	}

	protected List<?> getAssessments(Connection con, Assessment criteria)
			throws Exception {
		List<?> asList = null;
		AssessmentDAO dao = new AssessmentDAO();
		asList = dao.find(con, criteria);
		return asList;
	}

	public static void main(String[] args) {
		UploadManager um = new UploadManager();
		Connection con = null;
		Properties props = null;
		try {
			props = GenUtils.loadProperties("assessments.properties");
			String dbURL = props.getProperty("dburl");
			String user = props.getProperty("user");
			String pwd = props.getProperty("pwd");
			String dataFile = props.getProperty("data_file");
			Class.forName("oracle.jdbc.driver.OracleDriver");

			con = DriverManager.getConnection(dbURL, user, pwd);
			String owner = "ucsd_fmri";
			String modUser = "ucsd_fmri";

			um.loadData(dataFile); // "/home/bozyurt/dev/java/clinical/conf/clinical_data.csv");

			// um.prepareExperimentInfos(con, owner, modUser);
			// um.prepareAssessments(con,owner, modUser);

			// using public:loc,brn security label for everything uploaded here
			um.insertScoreValues(con, owner, modUser, new BigDecimal("11000"));
		} catch (Exception x) {
			x.printStackTrace();
		} finally {
			if (con != null)
				try {
					con.close();
				} catch (Exception x) { /* ignore */
				}
		}
	}
}