package clinical.test;

import java.io.FileInputStream;
import java.io.IOException;

import junit.framework.Test;
import junit.framework.TestSuite;

import org.dbunit.dataset.DataSetException;
import org.dbunit.dataset.IDataSet;
import org.dbunit.dataset.ITable;
import org.dbunit.dataset.ITableIterator;
import org.dbunit.dataset.NoSuchTableException;
import org.dbunit.dataset.SortedTable;
import org.dbunit.dataset.filter.DefaultColumnFilter;
import org.dbunit.dataset.xml.FlatXmlDataSet;

import clinical.test.framework.DBAssertion;
import clinical.utils.Assertion;

/**
 * @author I. Burak Ozyurt
 * @version $Id: DBStaticDataIntegrityTest.java,v 1.1 2007/11/27 02:18:46
 *          bozyurt Exp $
 */

public class DBStaticDataIntegrityTest extends MyDBTestCase {
    private final static boolean VERBOSE = false;
	public DBStaticDataIntegrityTest(String testName, String propsFile,
			boolean skipQueryProcessorCache) throws IOException {
		super(testName, propsFile, skipQueryProcessorCache);
	}

	@Override
	protected IDataSet getDataSet() throws Exception {
		String staticDataFile = mixin.getProperty("static.data.file");
		Assertion.assertNotNull(staticDataFile);
		return new FlatXmlDataSet(new FileInputStream(staticDataFile));
	}

	public void testStaticDataIntegrity() throws Exception {
		IDataSet dbDataSet = getConnection().createDataSet();
		IDataSet expectedDataSet = getDataSet();
		if (VERBOSE) {
			System.out.println("Tables in schema:");
			ITableIterator it = dbDataSet.iterator();
			while (it.next()) {
				System.out.println(it.getTableMetaData().getTableName());
			}
			System.out.println("--------------");
		}
		testTableIntegrity(dbDataSet, expectedDataSet, "nc_visittype",
				new String[] { "visittype" });
		testTableIntegrity(dbDataSet, expectedDataSet,
				"nc_assessmentinformant", new String[] { "informantrelation" });
		testTableIntegrity(dbDataSet, expectedDataSet, "nc_userclass",
				new String[] { "userclass", "name", "description" });
		testTableIntegrity(dbDataSet, expectedDataSet, "nc_dataclassification",
				new String[] { "name", "description" });
		testTableIntegrity(dbDataSet, expectedDataSet, "nc_tableid",
				new String[] { "tablename" });
		testTableIntegrity(dbDataSet, expectedDataSet, "nc_tupleclass",
				new String[] { "tupleclass" });
		testTableIntegrity(dbDataSet, expectedDataSet, "nc_animalspecies",
				new String[] { "name" });
		testTableIntegrity(dbDataSet, expectedDataSet, "nc_userstatus",
				new String[] { "userstatus", "name", "description" });
		testTableIntegrity(dbDataSet, expectedDataSet, "nc_assessmentstatus",
				new String[] { "status", "name", "description" });
		testTableIntegrity(dbDataSet, expectedDataSet,
				"nc_securityclassification", new String[] {
						"securityclassification", "description" });

		testTableIntegrity(dbDataSet, expectedDataSet, "nc_ontologysource",
				new String[] { "ontologysource" });

		testTableIntegrity(dbDataSet, expectedDataSet, "nc_researchgrouptype",
				new String[] { "name" });

		testTableIntegrity(dbDataSet, expectedDataSet, "nc_schemaversion",
				new String[] { "majorversion", "minorversion" });

		System.out.println("All the necessary static data checks for 12 tables and schema version checks are successful!");

	}

	protected ITable getTable(IDataSet dataSet, String tableName) throws DataSetException {
		ITable table = null;
		try {
			table = dataSet.getTable(tableName);
		} catch(NoSuchTableException e) {
			try {
				table = dataSet.getTable(tableName.toUpperCase());
			} catch(NoSuchTableException e1) {
				table = dataSet.getTable(tableName.toLowerCase());
			}
		}
		return table;
	}


	protected void testTableIntegrity(IDataSet dbDataSet,
			IDataSet expectedDataSet, String tableName, String[] columnNames)
			throws DataSetException {
		ITable actualTable = getTable(dbDataSet, tableName);
		ITable expectedTable = getTable(expectedDataSet, tableName);

		actualTable = DefaultColumnFilter.includedColumnsTable(actualTable,
				columnNames);
		expectedTable = DefaultColumnFilter.includedColumnsTable(expectedTable,
				columnNames);

		DBAssertion.assertFullyContains(new SortedTable(expectedTable),
				new SortedTable(actualTable, expectedTable.getTableMetaData()));
	}

	public static Test suite() throws IOException {
		TestSuite suite = new TestSuite();
		suite.addTest(new DBStaticDataIntegrityTest("testStaticDataIntegrity",
				"test.properties", true));

		return suite;
	}

	public static void main(String[] args) throws IOException {
		junit.textui.TestRunner.run(suite());
	}

}
