package clinical.test.framework;

import java.io.FileOutputStream;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.dbunit.database.DatabaseConnection;
import org.dbunit.database.IDatabaseConnection;
import org.dbunit.database.QueryDataSet;
import org.dbunit.dataset.IDataSet;
import org.dbunit.dataset.xml.FlatDtdDataSet;
import org.dbunit.dataset.xml.FlatXmlDataSet;
import org.jdom.Attribute;
import org.jdom.Element;

import clinical.tools.db.SchemaExtractor;
import clinical.utils.FileUtils;
import clinical.web.ConnectionSupportMixin;

/**
 * @author I. Burak Ozyurt
 * @version $Id: DBTestDataExtractionUtils.java,v 1.1 2007/11/28 02:27:48
 *          bozyurt Exp $
 */
public class DBTestDataExtractionUtils {

   public static void prepDBDTD(Connection con, String dtdFilename,
         String schemaName) throws Exception {
      IDatabaseConnection ic = new DatabaseConnection(con, schemaName);
      FlatDtdDataSet.write(ic.createDataSet(),
            new FileOutputStream(dtdFilename));
   }

   public static void exportDB2FlatSet(Connection con, String flatSetFile,
         List<String> tableNames, String schemaName) throws Exception {
      IDatabaseConnection ic = new DatabaseConnection(con, schemaName);
      QueryDataSet partialDataSet = new QueryDataSet(ic);
      for (String tn : tableNames) {
         partialDataSet.addTable(tn);
      }
      FlatXmlDataSet.write(partialDataSet, new FileOutputStream(flatSetFile));
   }

   public static void exportFullDB2FlatSet(Connection con, String flatSetFile,
         String schemaName) throws Exception {
      IDatabaseConnection ic = new DatabaseConnection(con, schemaName);
      IDataSet partialDataSet = ic.createDataSet();
      FlatXmlDataSet.write(partialDataSet, new FileOutputStream(flatSetFile));
   }

   public static void exportDBSchema2Xml(Connection con, String schemaName,
         String schemaXmlFile) throws Exception {
      SchemaExtractor se = new SchemaExtractor();
      se.loadDBTables(con, schemaName, schemaXmlFile, false);
   }

   public static List<String> getRegexFilteredTables(String schemaXmlFile,
         String tableNameRegexStr) throws Exception {
      List<String> tableNames = new ArrayList<String>();
      Element rootEl = FileUtils.loadXML(schemaXmlFile);
      List<?> tableEls = rootEl.getChildren("table");
      Pattern p = Pattern.compile(tableNameRegexStr);
      for (Object o : tableEls) {
         Element tableEl = (Element) o;
         String name = tableEl.getAttributeValue("name");
         if (p.matcher(name).find()) {
            tableNames.add(name);
         }
      }

      return tableNames;
   }

   public static void findFlatSetDiffs(String baseFlatSetFile,
         String newFlatSetFile, String diffFlatSetFile, boolean verbose)
         throws Exception {
      Element baseRoot = FileUtils.loadXML(baseFlatSetFile);
      if (!baseRoot.getName().equals("dataset")) {
         throw new RuntimeException("Not a DBUnit flat set XML file:"
               + baseFlatSetFile);
      }
      Element newRoot = FileUtils.loadXML(newFlatSetFile);
      Element diffRoot = new Element("dataset");
      Map<String, Element> baseMap = new HashMap<String, Element>();
      List<?> baseChildren = baseRoot.getChildren();
      for (Object item : baseChildren) {
         Element childEl = (Element) item;
         String key = prepKey(childEl);
         baseMap.put(key, childEl);
      }
      baseRoot = null;
      baseChildren = null;
      List<?> newChildren = newRoot.getChildren();
      for (Object item : newChildren) {
         Element childEl = (Element) item;
         String key = prepKey(childEl);
         if (!baseMap.containsKey(key)) {
            if (verbose) {
               System.out.println("different: key:" + key);
            }
            Element copy = (Element) childEl.clone();
            diffRoot.addContent(copy);
         }
      }

      FileUtils.saveXML(diffRoot, diffFlatSetFile);
   }

   protected static String prepKey(Element fsRowEl) {
      StringBuilder sb = new StringBuilder(128);
      sb.append(fsRowEl.getName()).append('_');
      List<?> attrs = fsRowEl.getAttributes();
      for (Iterator<?> it = attrs.iterator(); it.hasNext();) {
         Attribute attr = (Attribute) it.next();
         sb.append(attr.getName()).append(':').append(attr.getValue());
         if (it.hasNext())
            sb.append('_');
      }
      return sb.toString();
   }

   public static void main(String[] args) throws Exception {
      @SuppressWarnings("unused")
      String schemaXmlFile = "/home/bozyurt/dev/java/BIRN/clinical/test/db_test_data/fbirn_test2_schema.xml";
      @SuppressWarnings("unused")
      String dtdFilename = "/home/bozyurt/dev/java/BIRN/clinical/test/db_test_data/fbirn_test2_schema.dtd";
      String flatSetFile = "/home/bozyurt3/dev/java/B_SLICER_WS/BIRN/clinical/test/db_test_data/clinical_data_flat.xml";

      ConnectionSupportMixin csm = null;
      Connection con = null;
      try {
         csm = new ConnectionSupportMixin("test_clinical.properties", true);
         csm.startup();
         con = csm.getConnection();
         // DBTestDataExtractionUtils.exportDBSchema2Xml(con, "FBIRN_TEST2",
         // schemaXmlFile);

         // DBTestDataExtractionUtils
         // .prepDBDTD(con, dtdFilename, "FBIRN_TEST2");
         DBTestDataExtractionUtils.exportFullDB2FlatSet(con, flatSetFile, null);

      } finally {
         if (csm != null) {
            csm.releaseConnection(con);
            csm.shutdown();
         }
      }
   }
}
