package clinical.test.framework;

import java.sql.SQLException;

import org.dbunit.DatabaseUnitException;
import org.dbunit.database.IDatabaseConnection;
import org.dbunit.database.statement.SimplePreparedStatement;
import org.dbunit.dataset.DataSetException;
import org.dbunit.dataset.IDataSet;
import org.dbunit.dataset.ITable;
import org.dbunit.dataset.ITableIterator;
import org.dbunit.dataset.ITableMetaData;
import org.dbunit.dataset.RowOutOfBoundsException;
import org.dbunit.operation.AbstractOperation;
import org.dbunit.operation.DatabaseOperation;
import org.dbunit.operation.InsertOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InsertIfMissingOperation extends AbstractOperation {
   private static final Logger logger = LoggerFactory
         .getLogger(InsertIfMissingOperation.class);
   private final InsertOperation insertOp;

   public InsertIfMissingOperation() {
      insertOp = (InsertOperation) DatabaseOperation.INSERT;
   }

   @Override
   public void execute(IDatabaseConnection connection, IDataSet dataSet)
         throws DatabaseUnitException, SQLException {
      if (logger.isDebugEnabled())
         logger
               .debug("execute(connection=" + connection + ", dataSet) - start");

      // for each table
      ITableIterator iterator = dataSet.iterator();
      while (iterator.next()) {
         ITable table = iterator.getTable();

         // Do not process empty table
         if (DBUnitUtils.isEmpty(table)) {
            continue;
         }
         ITableMetaData metaData = DBUnitUtils.getOperationMetaData(connection,
               table.getTableMetaData());
         RowOperation insertRowOperation = new InsertRowOperation(connection,
               metaData);

         try {
            // refresh all rows
            for (int i = 0;; i++) {
               RowExistOperation reOp = new RowExistOperation(connection,
                     metaData);
               if (!reOp.execute(table, i)) {
                  insertRowOperation.execute(table, i);
               }
            }
         } catch (RowOutOfBoundsException e) {
            // This exception occurs when records are exhausted
            // and we reach the end of the table. Ignore this error.
            // end of table
         } finally {
            // cleanup
            insertRowOperation.close();
         }
      }

   }

   class InsertRowOperation extends RowOperation {
      private final Logger logger = LoggerFactory
            .getLogger(InsertRowOperation.class);

      private IDatabaseConnection _connection;
      private ITableMetaData _metaData;

      public InsertRowOperation(IDatabaseConnection connection,
            ITableMetaData metaData) throws DataSetException, SQLException {
         _connection = connection;
         _metaData = metaData;
      }

      public boolean execute(ITable table, int row) throws DataSetException,
            SQLException {
         logger.debug("execute(table=" + table + ", row=" + row + ") - start");

         // If current row have a different ignore value mapping than
         // previous one, we generate a new statement
         if (_ignoreMapping == null
               || !DBUnitUtils.equalsIgnoreMapping(_ignoreMapping, table, row)) {
            // Execute and close previous statement
            if (_statement != null) {
               _statement.close();
            }

            _ignoreMapping = DBUnitUtils.getIgnoreMapping(table, row);
            _operationData = insertOp.getOperationData(_metaData,
                  _ignoreMapping, _connection);
            _statement = new SimplePreparedStatement(_operationData.getSql(),
                  _connection.getConnection());
         }

         return super.execute(table, row);
      }

   }

}
