package clinical.test.framework;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.dbunit.database.IDatabaseConnection;
import org.dbunit.dataset.Column;
import org.dbunit.dataset.DataSetException;
import org.dbunit.dataset.ITable;
import org.dbunit.dataset.ITableMetaData;
import org.dbunit.dataset.NoColumnsFoundException;
import org.dbunit.dataset.datatype.DataType;
import org.dbunit.operation.OperationData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class RowExistOperation extends RowOperation {
   private final Logger logger = LoggerFactory
         .getLogger(RowExistOperation.class);

   PreparedStatement _countStatement;

   public RowExistOperation(IDatabaseConnection connection,
         ITableMetaData metaData) throws DataSetException, SQLException {
      // setup select count statement
      _operationData = getSelectCountData(metaData, connection);
      _countStatement = connection.getConnection().prepareStatement(
            _operationData.getSql());
   }

   private OperationData getSelectCountData(ITableMetaData metaData,
         IDatabaseConnection connection) throws DataSetException {
      if (logger.isDebugEnabled())
         logger.debug("getSelectCountData(metaData=" + metaData
               + ", connection=" + connection + ") - start");

      Column[] columns = metaData.getColumns();

      if (columns.length == 0) {
         throw new NoColumnsFoundException(metaData.getTableName());
      }

      // select count
      StringBuffer sqlBuffer = new StringBuffer(128);
      sqlBuffer.append("select COUNT(*) from ");
      sqlBuffer.append(DBUnitUtils.getQualifiedName(connection.getSchema(),
            metaData.getTableName(), connection));

      // where
      sqlBuffer.append(" where ");
      for (int i = 0; i < columns.length; i++) {
         Column column = columns[i];
         if (i > 0) {
            sqlBuffer.append(" and ");
         }
         sqlBuffer.append(DBUnitUtils.getQualifiedName(null, column
               .getColumnName(), connection));
         sqlBuffer.append(" = ?");
      }

      return new OperationData(sqlBuffer.toString(), columns);
   }

   public boolean execute(ITable table, int row) throws DataSetException,
         SQLException {
      if (logger.isDebugEnabled())
         logger.debug("execute(table="
               + table.getTableMetaData().getTableName() + ", row=" + row
               + ") - start");

      Column[] columns = _operationData.getColumns();
      for (int i = 0; i < columns.length; i++) {
         Object value = table.getValue(row, columns[i].getColumnName());
         // System.out.println("col:" + columns[i].getColumnName() + " value:"
         // + value);
         DataType dataType = columns[i].getDataType();
         dataType.setSqlValue(value, i + 1, _countStatement);
      }

      ResultSet resultSet = _countStatement.executeQuery();
      try {
         resultSet.next();
         return resultSet.getInt(1) > 0;
      } finally {
         resultSet.close();
      }
   }

   public void close() throws SQLException {
      logger.debug("close() - start");

      _countStatement.close();
   }
}