package clinical.tools.dbadmin;

import java.io.File;
import java.math.BigDecimal;
import java.sql.Connection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import clinical.server.dao.ExperimentDAO;
import clinical.server.dao.JobProvenanceDAO;
import clinical.server.dao.JobProvenanceParamDAO;
import clinical.server.dao.JobsDAO;
import clinical.server.dao.VisitJobDAO;
import clinical.server.vo.Experiment;
import clinical.server.vo.JobProvenance;
import clinical.server.vo.JobProvenanceParam;
import clinical.server.vo.Jobs;
import clinical.server.vo.VisitJob;
import clinical.utils.Assertion;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.ConnectionSupportMixin;
import clinical.web.DAOFactory;
import clinical.web.services.SecurityService;

public class JobDataDelAdmin extends AbstractAdmin {

	public JobDataDelAdmin(String usersXmlFile) throws Exception {
		csm = new ConnectionSupportMixin(usersXmlFile);
		csm.startup();

		SecurityService ss = SecurityService.getInstance(
				csm.getDbPoolService(), csm.getDbID(), csm.getDbType());
		ss.startup();
	}

	private Experiment getExp(Connection con, String expName) throws Exception {
		ExperimentDAO dao = DAOFactory.createExperimentDAO(csm.getDbID());
		Experiment cr = new Experiment();
		cr.setName(expName);
		List<Experiment> expList = dao.find(con, cr);
		Assertion.assertTrue(expList.size() == 1);
		return expList.get(0);
	}

	private List<VisitJob> getVisitJobs(Connection con, Experiment exp,
			String subjectID, int visitID) throws Exception {
		VisitJobDAO dao = DAOFactory.createVisitJobDAO(csm.getDbID());
		VisitJob cr = new VisitJob();
		cr.setComponentId(new BigDecimal(visitID));
		cr.setExpId(exp.getUniqueid());
		cr.setSubjectid(subjectID);
		return dao.find(con, cr);
	}

	private List<JobProvenance> getMatchingJobProvenances(Connection con,
			List<VisitJob> vjList, String derivedDirName) throws Exception {
		JobProvenanceDAO dao = DAOFactory.createJobProvenanceDAO(csm.getDbID());
		JobProvenance cr = new JobProvenance();
		Set<BigDecimal> jobIDSet = new HashSet<BigDecimal>();
		for (VisitJob vj : vjList) {
			jobIDSet.add(vj.getJobUniqueId());
		}
		boolean found = false;
		List<JobProvenance> jpList = null;
		for (BigDecimal jobUniqueID : jobIDSet) {
			cr.setJobUniqueid(jobUniqueID);

			List<JobProvenance> list = dao.find(con, cr);

			for (JobProvenance jp : list) {
				if (jp.getDatauri().endsWith(derivedDirName)) {
					found = true;
					jpList = list;
					break;
				}
			}
			if (found) {
				break;
			}
		}
		return jpList;
	}

	void deleteJobProvenanceData(Connection con, JobProvenance jp)
			throws Exception {
		JobProvenanceDAO dao = DAOFactory.createJobProvenanceDAO(csm.getDbID());
		JobProvenance cr = new JobProvenance();
		JobProvenanceParamDAO paramDAO = DAOFactory
				.createJobProvenanceParamDAO(csm.getDbID());
		JobProvenanceParam paramCR = new JobProvenanceParam();
		paramCR.setJobProvId(jp.getUniqueid());
		paramDAO.delete(con, paramCR);
		cr.setUniqueid(jp.getUniqueid());
		dao.delete(con, cr);
	}

	void deleteJobAndJobVisit(Connection con, BigDecimal jobUniqueID)
			throws Exception {
		VisitJobDAO vjDAO = DAOFactory.createVisitJobDAO(csm.getDbID());
		VisitJob vjCR = new VisitJob();
		JobsDAO dao = DAOFactory.createJobsDAO(csm.getDbID());
		Jobs cr = new Jobs();

		vjCR.setJobUniqueId(jobUniqueID);
		vjDAO.delete(con, vjCR);
		cr.setUniqueid(jobUniqueID);
		dao.delete(con, cr);
	}

	void deleteJobFiles(String dataURI) {
		File dir = new File(dataURI);
		Assertion.assertTrue(dir.isDirectory());
		if (dir.isDirectory()) {
			FileUtils.deleteRecursively(dir);
		}
	}

	public void deleteJobData(String expName, String subjectID, int visitID,
			String derivedDirName) throws Exception {
		Connection con = null;
		try {
			con = csm.getConnection();
			con.setAutoCommit(false);
			Experiment exp = getExp(con, expName);
			List<VisitJob> vjList = getVisitJobs(con, exp, subjectID, visitID);
			List<JobProvenance> jpList = getMatchingJobProvenances(con, vjList,
					derivedDirName);
			if (jpList != null) {
				for (JobProvenance jp : jpList) {
					System.out.println(jp);
				}
			}
			con.commit();
		} catch (Exception x) {
			con.rollback();
			x.printStackTrace();
		} finally {
			csm.releaseConnection(con);
		}
	}

	public static void usage() {
		System.err
				.println("Usage:JobDataDelAdmin [-h] -s <subjectID> -e <exp-name> -v <visit-id> -d <job-dir-suffix>\n");
		System.exit(1);
	}

	public static void main(String[] args) throws Exception {
		String subjectID = "000859760550";
		String expName = "ASL";
		int visitID = 1;
		String derivedDirName = "derived.1";
		if (args.length != 1 && args.length != 8) {
			// usage();
		}

		int i = 0;
		while (i < args.length) {
			String argName = args[i];
			if (argName.equals("-s")) {
				subjectID = args[i + 1];
				i += 2;
			} else if (argName.equals("-h")) {
				usage();
			} else if (argName.equals("-v")) {
				visitID = GenUtils.toInt(args[i + 1], -1);
				i += 2;
			} else if (argName.equals("-e")) {
				expName = args[i + 1];
				i += 2;
			} else if (argName.equals("-d")) {
				derivedDirName = args[i + 1];
				i += 2;
			} else {
				i++;
			}
		}

		JobDataDelAdmin admin = null;
		try {
			admin = new JobDataDelAdmin("users.xml");

			admin.deleteJobData(expName, subjectID, visitID, derivedDirName);
		} finally {
			if (admin != null) {
				admin.shutdown();
			}
		}
	}
}
