package clinical.tools.dbadmin;

import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.json.JSONObject;

import clinical.server.dao.DeriveddataDAO;
import clinical.server.dao.JobProvParamTypeDAO;
import clinical.server.dao.JobProvenanceDAO;
import clinical.server.dao.JobProvenanceParamDAO;
import clinical.server.vo.Deriveddata;
import clinical.server.vo.JobProvenance;
import clinical.server.vo.JobProvenanceParam;
import clinical.server.vo.Jobs;
import clinical.server.vo.VisitJob;
import clinical.utils.Assertion;
import clinical.utils.DateTimeUtils;
import clinical.web.ConnectionSupportMixin;
import clinical.web.DAOFactory;
import clinical.web.ISequenceHelper;
import clinical.web.MinimalServiceFactory;
import clinical.web.ServiceFactory;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.services.IJobProvenanceService;
import clinical.web.services.SecurityService;
import clinical.web.vo.JobProvenanceInfo.JobProvenanceParamInfo;
import clinical.web.vo.JobProvenanceInfo.JobProvenanceParamType;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class ProvenanceRecAdmin extends AbstractAdmin {
	public ProvenanceRecAdmin(String usersXmlFile) throws Exception {
		csm = new ConnectionSupportMixin(usersXmlFile);
		csm.startup();

		SecurityService ss = SecurityService.getInstance(
				csm.getDbPoolService(), csm.getDbID(), csm.getDbType());
		ss.startup();
	}

	public void removeDuplicateProvenanceRecs() throws Exception {
		Connection con = null;
		try {
			con = csm.getConnection();
			con.setAutoCommit(false);
			JobProvenanceDAO dao = DAOFactory.createJobProvenanceDAO(csm
					.getDbID());
			List<JobProvenance> jpList = dao.find(con, new JobProvenance());
			Map<String, List<JobProvenance>> map = new HashMap<String, List<JobProvenance>>();
			for (JobProvenance jp : jpList) {
				if (jp.getDescription() == null
						|| !jp.getDescription().startsWith("no param info")) {
					continue;
				}
				List<JobProvenance> list = map.get(jp.getDatauri());
				if (list == null) {
					list = new ArrayList<JobProvenance>(1);
					map.put(jp.getDatauri(), list);
				}
				list.add(jp);
			}
			for (List<JobProvenance> list : map.values()) {
				if (list.size() > 1) {
					for (JobProvenance jp : list) {
						JobProvenance cr = new JobProvenance();
						cr.setUniqueid(jp.getUniqueid());
						System.out.println("removing " + jp);
						dao.delete(con, cr);
					}
					System.out.println("-------------------------");
				}
			}
			con.commit();
		} catch (Exception x) {
			con.rollback();
			x.printStackTrace();
		} finally {
			csm.releaseConnection(con);
		}
	}

	public void addMissingProvenances() throws Exception {
		Connection con = null;
		try {
			con = csm.getConnection();
			con.setAutoCommit(false);

			List<JobContext> jobsWithMissingProvenance = findJobsWithMissingProvenance(con);
			for (JobContext jc : jobsWithMissingProvenance) {
				System.out.println(jc);
				String derivedDirPath = getDerivedDirPath(con, jc);
				System.out.println("derivedDirPath:" + derivedDirPath);
				if (derivedDirPath != null) {
					JobProvenance jp = addProvenanceRec(con, jc, derivedDirPath);
					System.out.println("added " + jp);
				} else {
					System.out.println("skipped...");
				}

				System.out
						.println("--------------------------------------------");
			}
			con.commit();

		} catch (Exception x) {
			con.rollback();
			x.printStackTrace();
		} finally {
			csm.releaseConnection(con);
		}
	}

	public void addJobFinishDateToProvenances() throws Exception {
		Connection con = null;
		try {
			con = csm.getConnection();
			con.setAutoCommit(false);
			addJobFinishDateToProvenances(con);
			con.commit();
		} catch (Exception x) {
			con.rollback();
			x.printStackTrace();
		} finally {
			csm.releaseConnection(con);
		}
	}

	private void addProvenanceParamsFromJobContextIfMissing(Connection con,
			JobContext jc, JobProvenance jprov) throws Exception {
		if (jc.job.getJobContext() == null
				|| jc.job.getJobContext().length() == 0) {
			return;
		}
		JobProvenanceParamDAO dao = DAOFactory.createJobProvenanceParamDAO(csm
				.getDbID());
		JobProvenanceParam cr = new JobProvenanceParam();
		cr.setJobProvId(jprov.getUniqueid());
		List<JobProvenanceParam> list = dao.find(con, cr);
		if (list.size() > 1) {
			for (JobProvenanceParam param : list) {
				if (param.getName().equals("fieldmap")) {
					return;
				}
			}
		}

		JSONObject js = new JSONObject(jc.job.getJobContext());
		List<JobProvenanceParamInfo> jpiList = new ArrayList<JobProvenanceParamInfo>();
		boolean doB0Correction = false;
		boolean doSenseVal = false;
		if (js.has("doB0Correction")) {
			doB0Correction = js.getBoolean("doB0Correction");
			jpiList.add(new JobProvenanceParamInfo("fieldmap",
					doB0Correction ? "yes" : "no"));
		}
		if (js.has("doReg")) {
			boolean doReg = js.getBoolean("doReg");
			jpiList.add(new JobProvenanceParamInfo("registration",
					doReg ? "yes" : "no"));
		}

		if (js.has("doSenseval")) {
			doSenseVal = js.getBoolean("doSenseval");
			jpiList.add(new JobProvenanceParamInfo("sense", doSenseVal ? "yes"
					: "no"));
		}
		if (js.has("doSkullStripping")) {
			boolean doSkull = js.getBoolean("doSkullStripping");
			jpiList.add(new JobProvenanceParamInfo("skullStripping",
					doSkull ? "yes" : "no"));
		}
		boolean hasAutoCSF = false;
		boolean hasPVC = false;
		if (js.has("autoCSF")) {
			hasAutoCSF = js.getBoolean("autoCSF");
		}
		jpiList.add(new JobProvenanceParamInfo("automated", hasAutoCSF ? "yes"
				: "no"));

		if (js.has("doPVC")) {
			hasPVC = js.getBoolean("doPVC");
		}
		jpiList.add(new JobProvenanceParamInfo("pvc", hasPVC ? "yes" : "no"));
		if (js.has("gmThreshold")) {
			String gmThreshold = js.getString("gmThreshold");
			jpiList.add(new JobProvenanceParamInfo("gmThresh", gmThreshold));
		}

		if (!jpiList.isEmpty()) {
			System.out
					.println("adding partial provenance from the job context...");
			IJobProvenanceService jps = ServiceFactory
					.getJobProvenanceService(csm.getDbID());
			jps.addUpdateProvenanceParams(con, jc.job.getUniqueid().intValue(),
					jpiList);

		}
	}

	private JobProvenance addProvenanceRec(Connection con, JobContext jc,
			String derivedDirPath) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(csm.getDbID());
		JobProvenanceDAO dao = DAOFactory.createJobProvenanceDAO(csm.getDbID());
		JobProvenance cr = new JobProvenance();
		// cr.setJobUniqueid(jc.job.getUniqueid());
		cr.setDatauri(derivedDirPath);
		List<JobProvenance> list = dao.find(con, cr);
		if (!list.isEmpty()) {
			Assertion.assertTrue(list.size() == 1);
			System.out
					.println("provenance already exists returning the existing record...");
			JobProvenance jp = list.get(0);
			if (jc.job.getUniqueid() == jp.getJobUniqueid()) {
				addProvenanceParamsFromJobContextIfMissing(con, jc, jp);
			}
			return jp;
		}

		JobProvenance jp = new JobProvenance();

		jp.setDatauri(derivedDirPath);
		jp.setName("");
		jp.setDescription("no param info");
		jp.setJobUniqueid(jc.job.getUniqueid());
		jp.setModtime(new Date());
		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				"nc_job_provenance", "uniqueid");
		jp.setUniqueid(uniqueid);
		dao.insert(con, jp);
		return jp;
	}

	private clinical.server.vo.JobProvParamType addProvenanceParamType(
			Connection con, String name, String type) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(csm.getDbID());
		JobProvParamTypeDAO dao = DAOFactory.createJobProvParamTypeDAO(csm
				.getDbID());

		clinical.server.vo.JobProvParamType cr = new clinical.server.vo.JobProvParamType();
		cr.setName(name);
		List<clinical.server.vo.JobProvParamType> list = dao.find(con, cr);
		if (list.isEmpty()) {
			BigDecimal uniqueid = sequenceHelper.getNextUID(con,
					"nc_job_prov_param_type", "uniqueid");
			clinical.server.vo.JobProvParamType jppt = new clinical.server.vo.JobProvParamType();
			jppt.setUniqueid(uniqueid);
			jppt.setName(name);
			jppt.setDataType(type);

			dao.insert(con, jppt);
			return jppt;
		} else {
			Assertion.assertTrue(list.get(0).getDataType().equals(type));
			System.out.println("provenance param:" + name
					+ " already has a type:" + list.get(0).getDataType());
			return list.get(0);
		}
	}

	private void addFinishDateProvenanceParam(Connection con, Jobs job,
			JobProvenance prov) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(csm.getDbID());
		JobProvenanceParamDAO dao = DAOFactory.createJobProvenanceParamDAO(csm
				.getDbID());
		String name = "Finish Date";
		JobProvenanceParam cr = new JobProvenanceParam();
		cr.setName(name);
		cr.setJobProvId(prov.getUniqueid());
		List<JobProvenanceParam> list = dao.find(con, cr);
		if (!list.isEmpty()) {
			System.out.println("Already exists:" + list.get(0));
			return;
		}
		JobProvenanceParam p = new JobProvenanceParam();
		p.setName(name);
		p.setJobProvId(prov.getUniqueid());

		String formattedDate = DateTimeUtils.formatDate(job.getJobenddate());
		p.setValue(formattedDate);
		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				"nc_job_provenance_param", "uniqueid");
		p.setUniqueid(uniqueid);

		dao.insert(con, p);
	}

	public void addProvenanceInfoFromJobContext() throws Exception {
		Connection con = null;
		try {
			con = csm.getConnection();
			con.setAutoCommit(false);
			addProvenanceInfoFromJobContext(con);
			con.commit();
		} catch (Exception x) {
			con.rollback();
			x.printStackTrace();
		} finally {
			csm.releaseConnection(con);
		}
	}

	public void addJobFinishDateToProvenances(Connection con) throws Exception {
		List<JobProvenance> provenances = findProvenances(con);
		Map<BigDecimal, JobProvenance> provMap = new HashMap<BigDecimal, JobProvenance>();
		for (JobProvenance jp : provenances) {
			provMap.put(jp.getJobUniqueid(), jp);
		}

		String name = "Finish Date";
		String type = JobProvenanceParamType.DATE.toString();
		addProvenanceParamType(con, name, type);

		TSQLProcessor tsp = new TSQLProcessor(csm.getSqlDialect());
		StringBuilder sb = new StringBuilder(128);
		sb.append("select a.*, b.* from Jobs as a, VisitJob as b ");
		sb.append("where a.uniqueid = b.jobUniqueId and ");
		sb.append("a.jobsize is not null");

		List<?> results = tsp.executeQuery(con, sb.toString());
		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			Object[] row = (Object[]) it.next();
			Jobs job = (Jobs) row[0];
			// VisitJob vj = (VisitJob) row[1];
			if (provMap.containsKey(job.getUniqueid())) {
				if (job.getJobenddate() != null) {
					JobProvenance prov = provMap.get(job.getUniqueid());
					addFinishDateProvenanceParam(con, job, prov);
				}
			}
		}
	}

	public void addProvenanceInfoFromJobContext(Connection con)
			throws Exception {
		List<JobProvenance> provenances = findProvenances(con);
		Map<BigDecimal, JobProvenance> provMap = new HashMap<BigDecimal, JobProvenance>();
		for (JobProvenance jp : provenances) {
			provMap.put(jp.getJobUniqueid(), jp);
		}

		TSQLProcessor tsp = new TSQLProcessor(csm.getSqlDialect());
		StringBuilder sb = new StringBuilder(128);
		sb.append("select a.*, b.* from Jobs as a, VisitJob as b ");
		sb.append("where a.uniqueid = b.jobUniqueId and ");
		sb.append("a.jobsize is not null");

		List<?> results = tsp.executeQuery(con, sb.toString());
		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			Object[] row = (Object[]) it.next();
			Jobs job = (Jobs) row[0];
			VisitJob vj = (VisitJob) row[1];
			JobContext jc = new JobContext(job, vj);
			if (provMap.containsKey(job.getUniqueid())) {
				JobProvenance prov = provMap.get(job.getUniqueid());
				if (prov.getDescription().startsWith("no param info")) {
					addProvenanceParamsFromJobContextIfMissing(con, jc, prov);
				}
			}
		}
	}

	public List<JobContext> findJobsWithMissingProvenance(Connection con)
			throws Exception {
		List<JobProvenance> provenances = findProvenances(con);
		Map<BigDecimal, JobProvenance> provMap = new HashMap<BigDecimal, JobProvenance>();
		for (JobProvenance jp : provenances) {
			provMap.put(jp.getJobUniqueid(), jp);
		}
		List<JobContext> jcList = new ArrayList<JobContext>();
		TSQLProcessor tsp = new TSQLProcessor(csm.getSqlDialect());
		StringBuilder sb = new StringBuilder(128);
		sb.append("select a.*, b.* from Jobs as a, VisitJob as b ");
		sb.append("where a.uniqueid = b.jobUniqueId and ");
		sb.append("a.jobsize is not null");

		List<?> results = tsp.executeQuery(con, sb.toString());
		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			Object[] row = (Object[]) it.next();
			Jobs job = (Jobs) row[0];
			VisitJob vj = (VisitJob) row[1];
			if (!provMap.containsKey(job.getUniqueid())) {
				jcList.add(new JobContext(job, vj));
			}
		}
		Map<String, List<JobContext>> map = new HashMap<String, List<JobContext>>();
		for (JobContext jc : jcList) {
			String key = prepKey(jc.visitJob);
			List<JobContext> list = map.get(key);
			if (list == null) {
				list = new ArrayList<JobContext>(1);
				map.put(key, list);
			}
			list.add(jc);
		}
		jcList.clear();
		for (List<JobContext> list : map.values()) {
			if (list.size() == 1) {
				jcList.add(list.get(0));
			} else {

				// find the latest job and discard others
				Collections.sort(list, new java.util.Comparator<JobContext>() {
					@Override
					public int compare(JobContext o1, JobContext o2) {
						Date d1 = o1.job.getJobstartdate();
						Date d2 = o2.job.getJobstartdate();

						return d2.compareTo(d1);
					}
				});
				jcList.add(list.get(0));

				// System.out.println(list);
			}
		}

		return jcList;
	}

	String getDerivedDirPath(Connection con, JobContext jc) throws Exception {
		DeriveddataDAO dao = DAOFactory.createDeriveddataDAO(csm.getDbID());
		Deriveddata cr = new Deriveddata();
		cr.setNcExperimentUniqueid(jc.visitJob.getExpId());
		cr.setSubjectid(jc.visitJob.getSubjectid());
		cr.setComponentid(jc.visitJob.getComponentId());
		List<Deriveddata> existingDDList = dao.find(con, cr);
		if (existingDDList.isEmpty()) {
			return null;
		}
		Deriveddata dd = existingDDList.get(0);
		String datauri = dd.getDatauri();
		int idx = datauri.indexOf("derived/");
		Assertion.assertTrue(idx != -1);
		String path = datauri.substring(0, idx + 7);
		return path;
	}

	public static String prepKey(VisitJob vj) {
		StringBuilder sb = new StringBuilder();
		sb.append(vj.getSubjectid()).append(':');
		sb.append(vj.getExpId()).append(':');
		sb.append(vj.getComponentId());
		return sb.toString();
	}

	public List<JobProvenance> findProvenances(Connection con) throws Exception {
		JobProvenanceDAO dao = DAOFactory.createJobProvenanceDAO(csm.getDbID());
		List<JobProvenance> list = dao.find(con, new JobProvenance());
		return list;
	}

	public static class JobContext {
		Jobs job;
		VisitJob visitJob;

		public JobContext(Jobs job, VisitJob visitJob) {
			this.job = job;
			this.visitJob = visitJob;
		}

		@Override
		public String toString() {
			return "JobContext [job=" + job + "\n, visitJob=" + visitJob + "]";
		}

	}// ;

	public static void usage() {
		System.err
				.println("Usage:ProvenanceRecAdmin [-h] [-date] [-context]\n");
		System.err
				.println("\t-date add job finish date param to the existing provenance records");
		System.err
				.println("\t-context add any existing partial provenance info from job context  to pre-provenance feature empty provenance records");
		System.err
				.println("\tDefault operation is addition of missing provenance records for before provenance support jobs");
		System.exit(1);
	}

	public static void main(String[] args) throws Exception {
		if (args.length != 1) {
			usage();
		}
		boolean addFinishDates = false;
		boolean addJobContextProv = false;
		if (args.length == 1) {
			String argName = args[0];
			if (argName.equals("-date")) {
				addFinishDates = true;
			} else if (argName.equals("-h")) {
				usage();
			} else if (argName.equals("-context")) {
				addJobContextProv = true;
			}
		}

		ProvenanceRecAdmin admin = null;
		try {
			admin = new ProvenanceRecAdmin("users.xml");
			if (addJobContextProv) {
				admin.addProvenanceInfoFromJobContext();
			}
			if (!addFinishDates) {
				admin.addMissingProvenances();
			}
			// admin.removeDuplicateProvenanceRecs();

			if (addFinishDates) {
				admin.addJobFinishDateToProvenances();
			}
		} finally {
			if (admin != null) {
				admin.shutdown();
			}
		}
	}
}
