package clinical.tools.dbadmin.migration;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import org.jdom.Attribute;
import org.jdom.Element;

import clinical.utils.Assertion;
import clinical.utils.DateTimeUtils;
import edu.emory.mathcs.backport.java.util.Collections;

/**
 * Class responsible for managing XML serializers/deserializers for any value
 * object.
 * 
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class VOMigrationManager {
	private static VOMigrationManager instance = null;
	@SuppressWarnings("unchecked")
	private Map<String, VOXMLPersister<? extends Object>> persisterRepository = Collections
			.synchronizedMap(new HashMap<String, VOXMLPersister<? extends Object>>());

	private VOMigrationManager() {
	}

	public static synchronized VOMigrationManager getInstance() {
		if (instance == null) {
			instance = new VOMigrationManager();
		}
		return instance;
	}

	public void register(VOXMLPersister<? extends Object> persister) {
		persisterRepository.put(persister.getClassName(), persister);
	}

	public VOXMLPersister<? extends Object> get(String className) {
		return persisterRepository.get(className);
	}

	public static class VOXMLPersister<T> {
		Map<String, PropertyDescriptor> pdMap = new LinkedHashMap<String, PropertyDescriptor>();
		Class<T> voClazz;
		final static Object[] emptyArr = new Object[] {};
		final static SimpleDateFormat sdf = new SimpleDateFormat(
				"EEE MMM dd HH:mm:ss zzz yyyy");

		public VOXMLPersister(Class<T> voClazz) throws Exception {
			this.voClazz = voClazz;
			// System.out.println(voClazz.getSimpleName());
			BeanInfo bi = Introspector.getBeanInfo(voClazz);
			for (PropertyDescriptor pd : bi.getPropertyDescriptors()) {
				if (pd.getName().equals("class")) {
					continue;
				}
				// System.out.println(pd.getName());
				pdMap.put(pd.getName(), pd);
			}
		}

		public String getClassName() {
			return voClazz.getSimpleName();
		}

		public boolean hasProperty(String propertyName) {
			return pdMap.containsKey(propertyName);
		}

		public String getStringValue(T vo, String propertyName)
				throws Exception {
			Method m = getReadMethod(propertyName);
			return (String) m.invoke(vo, emptyArr);
		}

		public BigDecimal getBigDecimalValue(T vo, String propertyName)
				throws Exception {
			Method m = getReadMethod(propertyName);
			return (BigDecimal) m.invoke(vo, emptyArr);
		}

		public Date getDateValue(T vo, String propertyName) throws Exception {
			Method m = getReadMethod(propertyName);
			return (Date) m.invoke(vo, emptyArr);
		}

		protected Method getReadMethod(String propertyName) {
			PropertyDescriptor pd = pdMap.get(propertyName);
			Assertion.assertNotNull(pd);
			Method m = pd.getReadMethod();
			return m;
		}

		public void setValue(T vo, String propertyName, Object value)
				throws Exception {
			PropertyDescriptor pd = pdMap.get(propertyName);
			Assertion.assertNotNull(pd);
			Method m = pd.getWriteMethod();
			m.invoke(vo, new Object[] { value });
		}

		public Element toXml(T vo) throws Exception {
			Element el = new Element("vo");
			el.setAttribute("voName", voClazz.getSimpleName());
			for (PropertyDescriptor pd : pdMap.values()) {
				Method m = pd.getReadMethod();
				Assertion.assertNotNull(m);
				Object value = m.invoke(vo, emptyArr);
				if (value != null) {
					el.setAttribute(pd.getName(), value.toString());
				}
			}
			return el;
		}

		public T fromXml(Element voEl) throws Exception {
			String voName = voEl.getAttributeValue("voName");
			Assertion.assertTrue(voName.equals(voClazz.getSimpleName()));
			Constructor<T> constructor = voClazz
					.getConstructor(new Class<?>[] {});
			T vo = constructor.newInstance(emptyArr);
			for (PropertyDescriptor pd : pdMap.values()) {
				String name = pd.getName();
				Attribute attr = voEl.getAttribute(name);
				if (attr != null) {
					Method m = pd.getWriteMethod();
					Assertion.assertNotNull(m);
					Class<?> type = pd.getPropertyType();
					if (type == String.class) {
						m.invoke(vo, new Object[] { attr.getValue() });
					} else if (type == Boolean.class) {
						m.invoke(vo, new Object[] { attr.getBooleanValue() });
					} else if (type == BigDecimal.class) {
						m.invoke(
								vo,
								new Object[] { new BigDecimal(attr.getValue()) });
					} else if (type == Date.class) {
						m.invoke(vo,
								new Object[] { sdf.parse(attr.getValue()) });
					} else if (type == Float.class) {
						m.invoke(vo, new Object[] { attr.getFloatValue() });
					} else if (type == java.sql.Timestamp.class) {
						Date d = DateTimeUtils.convertTimestampPostgres(attr
								.getValue());
						m.invoke(vo,
								new Object[] { new Timestamp(d.getTime()) });
					}
				}
			}
			return vo;
		}

	}

}
